<?php
namespace Filanco\ApiClient;

use Exception;
use SoapClient;

/**
 * Class ApiClientAbstract
 *
 * @abstract
 */
abstract class ApiClientAbstract
{
    protected $log;
    /**
     * @var null|SoapClient
     */
    protected $soap = null;

    /** @var int сколько секунд ждать подключения */
    protected $connect_timeout = 10;

    /** @var int сколько секунд ждать ответа */
    protected $timeout = 20;

    public function setTimeout($timeout)
    {
        $this->timeout = $timeout;
    }

    public function setConnectTimeout($connect_timeout)
    {
        $this->connect_timeout = $connect_timeout;
    }

    /**
     * Выполняет запрос в API
     *
     * @param $url
     * @param array $params
     * @param $method
     * @param $auth
     * @param $content_type
     * @param $chain_request
     * @param $parent_curl
     * @param array $use_cert
     * @return mixed
     * @throws Exception
     */
    protected function makeRequest(
        $url,
        $params = array(),
        $method = 'GET',
        $auth = array(),
        $content_type = 'json',
        $chain_request = array(),
        $parent_curl = null,
        $use_cert = array()
    ) {
        $ssl_version = CURL_SSLVERSION_TLSv1_0;
        if (isset($params['ssl_version'])) {
            $ssl_version = $params['ssl_version'];
            unset($params['ssl_version']);
        }
        $this->logStart();
        if (is_null($parent_curl)) {
            $curl = curl_init();
        } else {
            $curl = $parent_curl;
        }
        if ($curl === false) {
            throw new Exception('Невозможно инициализировать сеанс curl');
        }
        $query = $this->buildQuery($params);
        if (count($auth) > 0) {
            curl_setopt($curl, CURLOPT_USERPWD, $auth['username'] . ":" . $auth['password']);
        }
        if ($method == 'GET') {
            curl_setopt($curl, CURLOPT_URL, $url . '?' . $query);
        } else {
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_POSTFIELDS, $query);
        }
        $this->logRequest($url, $query, $params);

        switch ($content_type) {
            case 'form':
                curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-type: application/x-www-form-urlencoded'));
                break;
            case 'json':
                curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-type: application/json'));
                break;
            case 'pkcs7':
                curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-type: application/pkcs7-mime'));
                break;
            case 'xml':
                curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-type: text/xml'));
                break;
        }
        curl_setopt($curl, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 5.1; rv:31.0) Gecko/20100101 Firefox/31.0');
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, $this->connect_timeout);
        curl_setopt($curl, CURLOPT_TIMEOUT, $this->timeout);
        curl_setopt($curl, CURLOPT_COOKIEJAR, '/dev/null');
        curl_setopt($curl, CURLOPT_COOKIEFILE, '/dev/null');

        if (strpos($url, 'https://') !== false) {
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($curl, CURLOPT_SSLVERSION, $ssl_version);
            if (count($use_cert) > 0) {
                curl_setopt($curl, CURLOPT_SSLCERT, $use_cert['certificate']);
                curl_setopt($curl, CURLOPT_SSLKEY, $use_cert['key']);
                curl_setopt($curl, CURLOPT_HEADER, 0);
            }
        }

        curl_setopt($curl, CURLOPT_VERBOSE, true);

        $verbose = fopen('php://temp', 'rw+');
        curl_setopt($curl, CURLOPT_STDERR, $verbose);

        $data = curl_exec($curl);
        $info = curl_getinfo($curl);
        $error = curl_error($curl);
        rewind($verbose);
        $info['verbose_log'] = stream_get_contents($verbose);
        if (!empty($error)) {
            $check = [
                'status' => false,
                'message' => sprintf('curl error: %s.', $error),
                'display_error_text' => false,
            ];
        } else {
            $check = $this->checkResponseData($data, $info);
        }

        $this->logResponse($check, $data);
        $this->logEnd();
        $result = $this->handleCheckResult($check);
        if (count($chain_request) > 0) {
            $result = $this->makeRequest(
                $chain_request['url'],
                $chain_request['params'],
                $chain_request['method'],
                $chain_request['auth'],
                $chain_request['content_type'],
                $chain_request['chain_request'],
                $curl
            );
        } else {
            curl_close($curl);
        }
        return $result;
    }

    protected function makeSoapRequest(
        $action,
        $args = array(),
        $options = array(),
        $wsdl = null,
        $chain = false
    ) {
        $this->logStart();
        if (is_null($this->soap)) {
            $this->soap = new SoapClient($wsdl, $options);
        }
        if (!($this->soap instanceof SoapClient)) {
            throw new Exception('Не могу запустить Soap клиент');
        }
        if (!is_callable([$this->soap, $action])) {
            throw new Exception('невозможно исполнить указанное действие');
        }
        $this->logRequest('Soap action: ' . $action, 'Args: ' . var_export($args, true), $options);
        $data = call_user_func_array([$this->soap, $action], $args);
        $check = $this->checkResponseData($data, null);
        $this->logResponse($check, $data);
        $this->logEnd();
        $result = $this->handleCheckResult($check);
        if (!$chain) {
            $this->soap = null;
        }
        return $result;
    }

    /**
     * @param $check
     * @return mixed
     * @throws Exception
     */
    protected function handleCheckResult($check)
    {
        if ($check['status'] === false) {
            if ($check['display_error_text']) {
                throw new Exception($check['message']);
            }
            throw new Exception('Произошла ошибка');
        }
        return $check['data'];
    }

    /**
     * Строит параметры запроса к АПИ
     *
     * @param array $params
     * @return string
     */
    protected function buildQuery($params)
    {
        $url = '';
        foreach ($params as $key => $param) {
            if (is_array($param)) {
                foreach ($param as $part) {
                    $url .= '&';
                    if (is_string($key)) {
                        $url .= $key . '[]=';
                    }
                    $url .= $part;
                }
            } else {
                $url .= '&';
                if (is_string($key)) {
                    $url .= $key . '=';
                }
                $url .= $param;
            }
        }
        return trim($url, '&');
    }

    /**
     * Проверяет полученные данные на наличие ошибок и возвращает значение
     *
     * @param $data
     * @param $info
     * @return array
     */
    protected function checkResponseData($data, $info)
    {
        if ($info['http_code'] == 200) {
            $check = ['status' => true, 'data' => $data];
        } else {
            $check['status'] = false;
            $check['message'] = sprintf('Curl response is not 200');
            $check['display_error_text'] = false;
        }

        return $check;
    }

    /**
     * Инициализация средств логирования
     */
    protected function logStart()
    {

    }

    /**
     * Логирует запрос
     *
     * @param $url
     * @param $query
     * @param $params
     */
    protected function logRequest($url, $query, $params)
    {

    }

    /**
     * Логирует пришедший ответ
     *
     * Ответ в формате, возвращаемом функцией _checkResponseData
     * @see _checkResponseData
     * @param $response
     * @param $raw_data
     */
    protected function logResponse($response, $raw_data)
    {

    }

    /**
     * Завершает сеанс логирования
     */
    protected function logEnd()
    {

    }
}
