<?php
namespace Filanco\Certificate\Api;

use Exception;
use Filanco\Certificate\Exception\CertificatePurchasedButNotConfiguredException;
use Filanco\Certificate\Model\Certificate;
use Filanco\Certificate\ResellerApiAbstract;

/**
 * Class Enom
 * @package Filanco\Certificate\Api
 */
class Enom extends ResellerApiAbstract
{
    /**
     * @inheritdoc
     */
    protected function executeCommand($command, $params = [])
    {
        $params['Command'] = $command;
        $params['UID'] = $this->reseller->getLogin();
        $params['PW'] = $this->reseller->getPassword();
        $params['ResponseType'] = $this->reseller->getContentType();
        $params['EndUserIP'] = '';

        return $this->makeRequest($this->reseller->getUri(), $params);
    }

    /**
     * @inheritdoc
     */
    protected function parseResponse($raw_response)
    {
        libxml_use_internal_errors(true);
        $document = simplexml_load_string($raw_response);
        if ($document === false) {
            throw new Exception(
                'Ошибка разбора ответа!' . PHP_EOL .
                implode(PHP_EOL, libxml_get_errors()) . PHP_EOL .
                'Тело ответа который разбираем: ' . $raw_response
            );
        }
        if (property_exists($document, 'ErrCount') && (string)$document->ErrCount > 0) {
            $errors = [];
            foreach ($document->errors as $error) {
                $errors[] = (string)$error->children();
            }
            throw new Exception(
                'Запрос в "Enom SSL API" вернул ошибки: ' . PHP_EOL .
                implode(PHP_EOL, $errors) . PHP_EOL .
                'Тело ответа: ' . $raw_response
            );
        }

        return $document;
    }

    /**
     * @param Certificate $certificate
     * @return array возможные варианты email для подтверждения
     */
    public function getApproverEmails(Certificate $certificate)
    {
        $email_response = $this->executeCommand('CertGetApproverEmail', [
            'Domain' => $certificate->getDomainName()
        ]);
        $result = [];
        foreach ($email_response->CertGetApproverEMail->Approver as $approver) {
            $email = (string)$approver->ApproverEmail;
            if (!empty($email)) {
                $result[] = $email;
            }
        }

        return $result;
    }

    /**
     * @inheritdoc
     */
    public function createOrder(Certificate $certificate, $years)
    {
        $response = $this->executeCommand('PurchaseServices', [
            'Service' => $certificate->getTypeResellersItemId(),
            'NumYears' => $years,
        ]);

        try {
            $resellers_item_id = (string)$response->certid;
            $this->configureCertificate($certificate, $resellers_item_id);
            $this->purchase($certificate, $resellers_item_id);
            return $resellers_item_id;
        } catch (Exception $e) {
            throw new CertificatePurchasedButNotConfiguredException($certificate, $e);
        }
    }

    public function purchase(Certificate $certificate, $resellers_item_id)
    {
        $purchase_params = [
            'CertID' => $resellers_item_id,
        ];

        $anketa = $certificate->getAnketa();

        $purchase_params['ApproverEmail'] = $anketa->getApproverEmail();

        if (strpos(strtolower($certificate->getCaTitle()), 'comodo') !== false) {
            $purchase_params['CSROrganization'] = $anketa->getOrganizationName();
            $purchase_params['CSROrganizationUnit'] = $anketa->getOrganizationalUnit();
            $purchase_params['CSRAddress1'] = $anketa->getAddress();
            $purchase_params['CSRAddress2'] = $anketa->getAddress();
            $purchase_params['CSRAddress3'] = $anketa->getAddress();
            $purchase_params['CSRLocality'] = $anketa->getCityName();
            $purchase_params['CSRStateProvince'] = $anketa->getRegionName();
            $purchase_params['CSRPostalCode'] = $anketa->getZipCode();
            $purchase_params['CSRCountry'] = $anketa->getCountryCode();
        }

        $this->executeCommand('CertPurchaseCert', $purchase_params);
    }

    public function clearCertificateConfig(Certificate $certificate)
    {
        $this->executeCommand('CertModifyOrder', [
            'CertID' => $certificate->getResellersItemId()
        ]);
    }

    public function configureCertificate(Certificate $certificate, $resellers_item_id)
    {
        $certificate_params = [
            'CSR' => $certificate->getRequest(),
            'CertID' => $resellers_item_id,
        ];

        $anketa = $certificate->getAnketa();

        if ($certificate->getTypeUcc()) {
            $additional_domain_names = $certificate->getAdditionalDomainNames();
            $certificate_params['DomainListNumber'] = $purchase_params['DomainListNumber'] =
                count($additional_domain_names);
            $i = 0;
            foreach ($additional_domain_names as $additional_domain_name) {
                $domain_list_key = 'UCCDomainList' . $i;
                $email_list_key = 'UCCEmailList' . $i;
                $certificate_params[$domain_list_key] = $purchase_params[$domain_list_key] =
                    $additional_domain_name;
                $additional_approver_emails = $anketa->getAdditionalDomainApproverEmails();
                if (isset($additional_approver_emails[$additional_domain_name])) {
                    $certificate_params[$email_list_key] = $purchase_params[$email_list_key] =
                        $additional_approver_emails[$additional_domain_name];
                } else {
                    throw new Exception('У домена ' . $additional_domain_name . '
                    отсутствует email для подтверждения!');
                }
                $i++;
            }
        }

        $certificate_params['CSR'] = $certificate->getRequest();

        if ($certificate->getClientType() == Certificate::CLIENT_TYPE_ORGANIZATION) {
            $person_params['OrgName'] = $anketa->getOrganizationName();
            $person_params['JobTitle'] = 'Technical';
        }

        $person_params['FName'] = $anketa->getFirstName();
        $person_params['LName'] = $anketa->getLastName();
        $person_params['Address1'] = $anketa->getAddress();
        $person_params['City'] = $anketa->getCityName();
        $person_params['Province'] = $anketa->getRegionName();
        $person_params['StateProvinceChoice'] = 'P';
        $person_params['PostalCode'] = $anketa->getZipCode();
        $person_params['Country'] = $anketa->getCountryCode();
        $person_params['Phone'] = $this->formatNumber($anketa->getPhone());

        $fax_number = $anketa->getFax();
        if (!empty($fax_number)) {
            $person_params['Fax'] = $this->formatNumber($fax_number);
        }

        $person_params['EmailAddress'] = $anketa->getEmail();

        if (strpos(strtolower($certificate->getCaTitle()), 'comodo') === false) {
            $certificate_params['WebServerType'] = 18;//Other
        } else {
            $certificate_params['WebServerType'] = 1000;//Other
        }

        foreach (['Admin', 'Billing', 'Tech'] as $person_type) {
            foreach ($person_params as $key => $value) {
                $certificate_params[$person_type . $key] = $value;
            }
        }

        $this->executeCommand('CertConfigureCert', $certificate_params);
    }

    /**
     * @inheritdoc
     */
    public function getValidationData(Certificate $certificate)
    {
        /**
         * Еном поддерживает только один тип валидации - валидацию по email
         */
        return 'admin@' . $certificate->getDomainName();
    }

    public function getCertificateInformation(Certificate $certificate)
    {
        $result = $this->executeCommand('CertGetCertDetail', [
            'CertID' => $certificate->getResellersItemId()
        ]);
        return $result->CertGetCertDetail;
    }

    /**
     * @inheritdoc
     */
    public function getCertificate(Certificate $certificate)
    {
        $certificate_info = $this->getCertificateInformation($certificate);
        return (string)$certificate_info->SSLCertificate;
    }

    /**
     * @inheritdoc
     */
    protected function getExternalStatusMap()
    {
        return [
            '1' => Certificate::STATUS_REQUEST,
            '2' => Certificate::STATUS_REQUEST,
            '4' => Certificate::STATUS_ENABLED,
            '6' => Certificate::STATUS_DISABLED,
            '9' => Certificate::STATUS_REQUEST,
            '10' => Certificate::STATUS_ENABLED,
            '12' => Certificate::STATUS_ENABLED,
            '13' => Certificate::STATUS_ENABLED,
        ];
    }

    /**
     * @inheritdoc
     */
    public function getExternalStatus(Certificate $certificate)
    {
        $certificate_info = $this->getCertificateInformation($certificate);
        return (string)$certificate_info->CertStatusID;
    }

    /**
     * @inheritdoc
     */
    public function getCatalog()
    {
        return $this->executeCommand('GetCerts');
    }

    protected function formatNumber($number)
    {
        $phone = preg_replace('/[^0-9]*/', '', $number);

        return '+' . substr($phone, 0, 1) . '.' . str_replace(' ', '', substr($phone, 1));
    }
}
