<?php
namespace Filanco\Certificate;

use Filanco\ApiClient\ApiClientAbstract;
use Filanco\Certificate\Exception\CertificateStatusUnknownException;
use Filanco\Certificate\Model\Certificate;

/**
 * Class Reseller
 * @package Filanco\Certificate
 */
abstract class ResellerApiAbstract extends ApiClientAbstract
{
    /**
     * @var ResellerInterface $reseller
     */
    protected $reseller;

    public function __construct(ResellerInterface $reseller, ResellerApiLogInterface $logger = null)
    {
        $this->reseller = $reseller;
        $this->log = $logger;
    }

    public function getCertificateStatus(Certificate $certificate)
    {
        $status = $this->getExternalStatus($certificate);
        $status_map = $this->getExternalStatusMap();
        if (!isset($status_map[$status])) {
            throw new CertificateStatusUnknownException($certificate, $status, $status_map);
        }
        return $status_map[$status];
    }

    /**
     * @param $data
     * @param $info
     * @return array
     */
    protected function checkResponseData($data, $info)
    {
        $parsed_response = $this->parseResponse($data);

        $check = [
            'status' => true,
            'data' => $parsed_response
        ];

        return $check;
    }

    /**
     * @param Certificate $certificate
     * @return string статус сертификата у реселлера
     */
    abstract protected function getExternalStatus(Certificate $certificate);

    /**
     * @return array внешний статус => системный статус
     */
    abstract protected function getExternalStatusMap();

    /**
     * Преобразовывает ответ в удобный для работы формат
     * @param string $raw_response
     * @return mixed
     */
    abstract protected function parseResponse($raw_response);

    /**
     * Регистрация заказа во внешней системе
     * @param Certificate $certificate
     * @param int $years количество лет
     * @return int id клиентского сертификата в партнерской системе
     */
    abstract public function createOrder(Certificate $certificate, $years);

    /**
     * Получает информацию о сертификате сертификат
     * @param Certificate $certificate
     * @return string
     */
    abstract public function getCertificate(Certificate $certificate);

    /**
     * @return mixed Каталог сертификатов
     */
    abstract public function getCatalog();

    /**
     * @param Certificate $certificate
     * @return array возможные варианты email для подтверждения
     */
    abstract public function getApproverEmails(Certificate $certificate);

    /**
     * @param Certificate $certificate
     * @return string|array информация необходимая для прохождения валидации
     */
    abstract public function getValidationData(Certificate $certificate);

    protected function logStart()
    {
        if ($this->log) {
            $this->log->setWho($this->reseller->getName());
        }
    }

    protected function logRequest($url, $query, $params)
    {
        if ($this->log) {
            $this->log->setRequest($url . '?' . $query);
        }
    }

    protected function logResponse($response, $raw_data)
    {
        if ($this->log) {
            $this->log->setResponse(var_export($response, true));
        }
    }

    protected function logEnd()
    {
        if ($this->log) {
            $this->log->setStamp(time());
            $this->log->save();
        }
    }
}
