<?php
namespace Filanco\Certificate\Model;

use Exception;
use Filanco\BaseModel;

/**
 * Class Certificate
 * @package Halk\Module\Order\Model
 * @method string getId() Уникальный идентификатор
 * @method string getClientType() Юридическая форма клиента
 * @method string getCaTitle() Название сертификационного центра
 * @method string getTypeUcc() Поддержка сертификатом UCC
 * @method string getTypeResellersItemId() Идентификатор типа сертификата у поставщика
 * @method string getResellersItemId() Идентификатор заказанного сертификата у поставщика
 * @method string getDomainName() Имя домена
 * @method string getRequest() Сгенерированный запрос на выпуск сертификата
 * @method string getPrivateKey() Приватный ключ
 * @method array getAdditionalDomainNames() Дополнительные домены
 * @method string getVerificationType() Тип проверки владельца домена
 * @method string getDomainVerificationType() Подтип проверки управлением домена
 * @method Anketa getAnketa() Тип верификации
 */
class Certificate extends BaseModel
{
    const
        CLIENT_TYPE_PERSON = 'person',
        CLIENT_TYPE_ORGANIZATION = 'organization';

    const
        DOMAIN_VERIFICATION_HTTP_CSR_HASH = 'HTTP',
        DOMAIN_VERIFICATION_CNAME_CSR_HASH = 'CNAME',
        DOMAIN_VERIFICATION_EMAIL = 'EMAIL';

    const
        VERIFICATION_DV = 'DV',
        VERIFICATION_EV = 'EV',
        VERIFICATION_OV = 'OV';

    const
        STATUS_ENABLED = 'ENABLED',
        STATUS_DISABLED = 'DISABLED',
        STATUS_DELETED = 'DELETED',
        STATUS_KEYPAIR = 'KEYPAIR',
        STATUS_REQUEST = 'REQUEST',
        STATUS_NEWCERT = 'NEWCERT';

    protected function fields()
    {
        return [
            'id' => 'Уникальный идентификатор',
            'client_type' => 'Юридическая форма клиента',
            'ca_title' => 'Название сертификационного центра',
            'type_ucc' => 'Поддержка сертификатом UCC',
            'type_resellers_item_id' => 'Идентификатор типа сертификата у поставщика',
            'resellers_item_id' => 'Идентификатор заказанного сертификата у поставщика',
            'domain_name' => 'Имя домена',
            'request' => 'Сгенерированный запрос на выпуск сертификата',
            'private_key' => 'Приватный ключ',
            'additional_domain_names' => 'Дополнительные домены',
            'verification_type' => 'Тип проверки владельца домена',
            'domain_verification_type' => 'Подтип проверки управлением домена',
            'anketa' => 'Анкета для заказа'
        ];
    }

    public function getPrivateKeyHash()
    {
        return static::getModulus(openssl_get_privatekey($this->getPrivateKey()));
    }

    public static function getCertificateHash($data)
    {
        return static::getModulus(openssl_pkey_get_public($data));
    }

    /**
     * Подсчет md5 хешей модулей
     * https://kb.wisc.edu/middleware/page.php?id=4064
     * @param resource $resource
     * @return string
     * @throws Exception
     */
    protected static function getModulus($resource)
    {
        if (!$resource) {
            throw new Exception('Ошибка в сертификате');
        }
        $key_details = openssl_pkey_get_details($resource);

        return md5($key_details['rsa']['n']);
    }

}