<?php
namespace Filanco\Certificate\Api;

use Exception;
use Filanco\Certificate\Exception\CertificateException;
use Filanco\Certificate\Exception\CertificateStatusUnknownException;
use Filanco\Certificate\Exception\CertificateValidationTypeUnknownException;
use Filanco\Certificate\Model\Anketa;
use Filanco\Certificate\Model\Certificate;
use Filanco\Certificate\ResellerApiAbstract;
use Filanco\Certificate\ResellerApiLogInterface;
use Filanco\Certificate\ResellerInterface;
use ZipArchive;

class Leader extends ResellerApiAbstract
{
    protected $raw_response;

    protected static $validation_type_map = [
        Certificate::DOMAIN_VERIFICATION_EMAIL => 'EMAIL',
        Certificate::DOMAIN_VERIFICATION_CNAME_CSR_HASH => 'CNAME_CSR_HASH',
        Certificate::DOMAIN_VERIFICATION_HTTP_CSR_HASH => 'HTTP_CSR_HASH',
    ];

    protected $auth_token;

    public function __construct(ResellerInterface $reseller, ResellerApiLogInterface $logger = null)
    {
        parent::__construct($reseller, $logger);

        $response = $this->executeCommand(
            'users/signin',
            ['login'=>$this->reseller->getLogin(), 'password'=>$this->reseller->getPassword()],
            'POST'
        );

        if (!property_exists($response, 'auth_token')) {
            throw new Exception(
                'Не найден токен!' . PHP_EOL .
                'Ответ сервиса: ' . json_encode($response)
            );
        }

        $this->auth_token = $response->auth_token;
    }

    /**
     * @inheritdoc
     */
    public function getValidationData(Certificate $certificate)
    {
        switch ($certificate->getVerificationType()) {
            case Certificate::DOMAIN_VERIFICATION_HTTP_CSR_HASH:
            case Certificate::DOMAIN_VERIFICATION_CNAME_CSR_HASH:
                $response = $this->executeCommand("ssl_certificates/{$certificate->getResellersItemId()}/dcv");

                /***** допилить обработку ошибки @TODO ****/
/*                if(!$response->success && !empty($response->error)) {
                    return $response;
                }*/

                $validation_information = json_decode(json_encode($response), true);
                $validation_information = array_shift($validation_information['dcv']);
                $result = $validation_information['hashes'];
                break;
            case Certificate::DOMAIN_VERIFICATION_EMAIL:
                $result = 'admin@' . $certificate->getDomainName();
                break;
            default:
                throw new CertificateValidationTypeUnknownException($certificate, $certificate->getVerificationType());
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getCatalog()
    {
        return $this->executeCommand('products');
    }

    /**
     * @inheritdoc
     */
    protected function parseResponse($raw_response)
    {
        $response = json_decode($raw_response);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception(
                'Ошибка разбора ответа!' . PHP_EOL .
                'Код ошибки: ' . json_last_error() . PHP_EOL .
                'Тело ответа который разбираем: ' . $raw_response
            );
        }
        return $response;
    }

    protected function getClientParams(Anketa $anketa)
    {
        $params = [
            'order'=> [
                'first_name' => $anketa->getFirstName(),
                'last_name' => $anketa->getLastName(),
                'phone' => $anketa->getPhone(),
                'email' => $anketa->getEmail(),
                'title' => $anketa->getOrganizationName(),
                'fax' => $anketa->getFax(),
                'company' => $anketa->getOrganizationName(),
                'address_1' => $anketa->getAddress(),
                'city_name' => $anketa->getCityName(),
                'region_name' => $anketa->getRegionName(),
                'zip_code' => $anketa->getZipCode(),
                'country_id' => $anketa->getCountryCode(),
            ]
        ];
        return $params;
    }

    /**
     * @param Certificate $certificate
     * @param int $years
     * @return array параметры сертификата
     */
    protected function getCertificateParams(Certificate $certificate, $years)
    {
        $anketa = $certificate->getAnketa();
        $domain_name = $certificate->getDomainName();
        
        $leader_validation_type = static::$validation_type_map[$certificate->getVerificationType()];

        $params = [
            'nomenclature_id' => $certificate->getTypeResellersItemId(),
            'days' => $years * 365,
            'ssl_server_software_id' => 37,
            'dcv_type' => $leader_validation_type,
            'domain_name' => $domain_name,
            'csr' => $certificate->getRequest(),
            'country_name' => $anketa->getCountryCode(),
            'region_name' => $anketa->getRegionName(),
            'city_name' => $anketa->getCityName(),
            'organization_name' => $anketa->getOrganizationName(),
        ];

        if ($certificate->getDomainVerificationType() == Certificate::DOMAIN_VERIFICATION_EMAIL) {
            $params['dcv_email'] = $anketa->getApproverEmail();
        }

        $server_software_id = false;

        if (strpos(strtolower($certificate->getCaTitle()), 'comodo') !== false) {
            /**
             * https://api.leaderssl.com/#certificate-details - 34 - Other
             */
            $server_software_id = 34;
        }

        if (strpos(strtolower($certificate->getCaTitle()), 'symantec') !== false ||
            strpos(strtolower($certificate->getCaTitle()), 'thawte') !== false
        ) {
            /**
             * https://api.leaderssl.com/#certificate-details - 54 - Other
             */
            $server_software_id = 54;
        }

        if ($server_software_id !== false) {
            $params['ssl_server_software_id'] = $server_software_id;
        }

        if ($certificate->getVerificationType() == Certificate::VERIFICATION_OV) {
            $params['organizational_unit'] = $anketa->getOrganizationalUnit();
            $params['zip_code'] = $anketa->getZipCode();
            $params['address_1'] = $anketa->getAddress();
        }

        $additional_domain_names = $certificate->getAdditionalDomainNames();

        if (!empty($additional_domain_names)) {
            $params['additional_domain_names'] = implode(',', $additional_domain_names);
        }

        return $params;
    }

    /**
     * @inheritdoc
     */
    public function createOrder(Certificate $certificate, $years)
    {
        $anketa = $certificate->getAnketa();
        $order_params = $this->getClientParams($anketa);
        $certificate_params = $this->getCertificateParams($certificate, $years);
        $order_params['order']['items'] = $certificate_params;
        $result = $this->executeCommand('order', $order_params, 'POST');
        if (property_exists($result, 'certificate') && !empty($result->certificate)) {
            return $result->certificate;
        } else {
            throw new CertificateException($certificate,
                'Ошибка создания заказа на сертификат на LeaderSSL!' . PHP_EOL .
                'Параметры запроса: ' . json_encode($order_params) . PHP_EOL .
                'Тело ответа: ' . json_encode($result)
            );
        }
    }

    /**
     * @inheritdoc
     */
    public function getCertificate(Certificate $certificate)
    {
        $result = $this->executeCommand("ssl_certificates/{$certificate->getResellersItemId()}/download", [], 'GET', true);

        $tmp_file_resource = tmpfile();
        fputs($tmp_file_resource, $result);
        $meta_data = stream_get_meta_data($tmp_file_resource);
        $tmp_file_name = $meta_data['uri'];

        /**
         * Проходим по архиву в поисках файлов *.crt
         */
        $zip = new ZipArchive();
        $opened = $zip->open($tmp_file_name);

        if ($opened !== true) {
            throw new Exception('Не удалось открыть zip-file! Контент: base_64' . base64_encode($result));
        }

        $crt_files = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $information = $zip->statIndex($i);
            $extension = strtolower(pathinfo($information['name'], PATHINFO_EXTENSION));
            if ($extension == 'crt') {
                $crt_files[] = $information['name'];
            }
        }

        $finded_certificate_body = '';

        $private_key_hash = $certificate->getPrivateKeyHash();
        
        /**
         * Проходим по файликам *.crt берем md5 хеш от модуля сертификата.
         * Если он совпадает с md5 хешем модуля приватного ключа, то считаем что это наш сертификат.
         */
        foreach ($crt_files as $crt_file) {
            $crt_stream = $zip->getStream($crt_file);
            $certificate_body = '';
            while (!feof($crt_stream)) {
                $certificate_body .= fread($crt_stream, 2);
            }

            $certificate_hash = Certificate::getCertificateHash($certificate_body);

            if ($certificate_hash == $private_key_hash) {
               $finded_certificate_body = $certificate_body;
            }
        }

        $zip->close();

        fclose($tmp_file_resource);

        if (empty($finded_certificate_body)) {
            throw new Exception(
                'В полученном архиве не найдено' .
                'не одного сертификата подходящего по хешу!' . PHP_EOL .
                json_encode($crt_files)
            );
        }

        return $finded_certificate_body;
    }

    /**
     * @inheritdoc
     */
    protected function getExternalStatusMap()
    {
        return [
            'new' => Certificate::STATUS_REQUEST,
            'in_progress' => Certificate::STATUS_REQUEST,
            'issued' => Certificate::STATUS_ENABLED,
            'revoked' => Certificate::STATUS_DISABLED,
            'expired' => Certificate::STATUS_DISABLED,
            'rejected' => Certificate::STATUS_DISABLED,
            'annulled' => Certificate::STATUS_DISABLED,
            'waiting_for_approval' => Certificate::STATUS_REQUEST
        ];
    }

    /**
     * @inheritdoc
     */
    protected function getExternalStatus(Certificate $certificate)
    {
        $result = $this->executeCommand("ssl_certificates/{$certificate->getResellersItemId()}/status");
        if (!property_exists($result, 'certificate')) {
            throw new Exception('Не удалось получить статус сертификата!' . PHP_EOL . var_export($result, true));
        }
        return $result->certificate;
    }

    /**
     * @inheritdoc
     */
    public function getCertificateStatus(Certificate $certificate)
    {
        $status = $this->getExternalStatus($certificate);
        $status_map = $this->getExternalStatusMap();
        if (!isset($status_map[$status])) {
            throw new CertificateStatusUnknownException($certificate, $status, $status_map);
        }

        return $status_map[$status];
    }

    /**
     * @inheritdoc
     */
    public function getApproverEmails(Certificate $certificate)
    {
        $result = $this->executeCommand(
            "dcv_emails",
            ['domain_name'=>$certificate->getDomainName()]
        );

        return $result->whois;
    }

    /**
     * @return array заказанные сертификаты
     */
    public function getOrderedCertificates()
    {
        $result = $this->executeCommand("ssl_certificates");

        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $result = $this->executeCommand('billing/balance');
        if (!property_exists($result, 'balance') || !property_exists($result->balance, 'sum')) {
            throw new Exception('Не удалось получить баланс!' . PHP_EOL . var_export($result, true));
        }
        return $result->balance->sum;
    }

    /**
     * @inheritdoc
     */
    protected function checkResponseData($data, $info)
    {
        if ($this->raw_response) {
            $parsed_response = $data;
        } else {
            $parsed_response = $this->parseResponse($data);
        }

        $check = [
            'status' => true,
            'data' => $parsed_response
        ];

        return $check;
    }

    /**
     * @inheritdoc
     */
    protected function executeCommand($command, $params = [], $method = 'GET', $raw_response = false)
    {
        if ($raw_response) {
            $this->raw_response = true;
        }

        $result = $this->makeRequest(rtrim($this->reseller->getUri(), '/') . '/' . $command, $params, $method);

        if ($raw_response) {
            $this->raw_response = false;
        }

        return $result;
    }

    protected function makeRequest(
        $url,
        $params = array(),
        $method = 'GET',
        $auth = array(),
        $content_type = 'json',
        $chain_request = array(),
        $parent_curl = null,
        $use_cert = array()
    ) {

        $this->logStart();
        if (is_null($parent_curl)) {
            $curl = curl_init();
        } else {
            $curl = $parent_curl;
        }
        if ($curl === false) {
            throw new Exception('Невозможно инициализировать сеанс curl');
        }
        $query = $this->buildQuery($params);
        if (count($auth) > 0) {
            curl_setopt($curl, CURLOPT_USERPWD, $auth['username'] . ":" . $auth['password']);
        }
        if ($method == 'GET') {
            curl_setopt($curl, CURLOPT_URL, $url . '?' . $query);
        } else {
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($params));
        }
        $headers = [];
        if (!empty($this->auth_token)) {
            $headers[] = 'Content-Type: application/json';
            $headers[] = 'Accept: application/json';
            $headers[] = sprintf('X-AUTH-TOKEN: %s', $this->auth_token);
        }

        $this->logRequest($url, $query, $params);

        switch ($content_type) {
            case 'form':
                $headers[] = 'Content-type: application/x-www-form-urlencoded';
                break;
            case 'json':
                $headers[] = 'Content-type: application/json';
                break;
            case 'pkcs7':
                $headers[] = 'Content-type: application/pkcs7-mime';
                break;
            case 'xml':
                $headers[] = 'Content-type: text/xml';
                break;
        }

        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($curl, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 5.1; rv:31.0) Gecko/20100101 Firefox/31.0');
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($curl, CURLOPT_TIMEOUT, 20);
        curl_setopt($curl, CURLOPT_COOKIEJAR, '/dev/null');
        curl_setopt($curl, CURLOPT_COOKIEFILE, '/dev/null');

        if (strpos($url, 'https://') !== false) {
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($curl, CURLOPT_SSLVERSION, 4);
            if (count($use_cert) > 0) {
                curl_setopt($curl, CURLOPT_SSLCERT, $use_cert['certificate']);
                curl_setopt($curl, CURLOPT_SSLKEY, $use_cert['key']);
                curl_setopt($curl, CURLOPT_HEADER, 0);
            }
        }

        curl_setopt($curl, CURLOPT_VERBOSE, true);

        $verbose = fopen('php://temp', 'rw+');
        curl_setopt($curl, CURLOPT_STDERR, $verbose);

        $data = curl_exec($curl);
        $info = curl_getinfo($curl);
        $error = curl_error($curl);
        rewind($verbose);
        $info['verbose_log'] = stream_get_contents($verbose);
        if (!empty($error)) {
            $check = [
                'status' => false,
                'message' => sprintf('curl error: %s.', $error),
                'display_error_text' => false,
            ];
        } else {
            $check = $this->checkResponseData($data, $info);
        }

        $this->logResponse($check, $data);
        $this->logEnd();
        $result = $this->handleCheckResult($check);
        if (count($chain_request) > 0) {
            $result = $this->makeRequest(
                $chain_request['url'],
                $chain_request['params'],
                $chain_request['method'],
                $chain_request['auth'],
                $chain_request['content_type'],
                $chain_request['chain_request'],
                $curl
            );
        } else {
            curl_close($curl);
        }

        return $result;
    }
}
