<?php
namespace Filanco\Domain\Api;

use Exception;
use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\ClientRegisterInterface;
use Filanco\Domain\Exception\OrderStatusUnknownException;
use Filanco\Domain\FindDomainContractNumberInterface;
use Filanco\Domain\GetExtendedAttributesInterface;
use Filanco\Domain\GetOrderInterface;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\ClientIp;
use Filanco\Domain\Model\ClientOrg;
use Filanco\Domain\Model\ClientPerson;
use Filanco\Domain\Model\Order;
use Filanco\Domain\RegistrarApiAbstract;
use Filanco\Domain\Exception\Auto\BalanceException;
use Filanco\Domain\Exception\Auto\ConnectionException;
use Filanco\Domain\Exception\Internal\AuthException;
use Filanco\Domain\Exception\InternalException;
use Filanco\Domain\Exception\ManualResolvingException;
use Filanco\Domain\Exception\User\DomainAlreadyRegisteredException;
use Filanco\Domain\Exception\User\ReservedDomainNameException;
use Filanco\Domain\Exception\User\StopListException;
use Filanco\Domain\UpdateDomainInterface;
use SimpleXMLElement;

/**
 * Class DomenusApi
 * @package Filanco\Domain\Api
 */
class DomenusApi extends RegistrarApiAbstract implements
    ClientRegisterInterface,
    FindDomainContractNumberInterface,
    GetOrderInterface,
    GetExtendedAttributesInterface,
    BalanceFetchInterface,
    UpdateDomainInterface
{
    protected $order_status_map = [
        0 => Order::STATUS_SUCCESS,
        1 => Order::STATUS_ERROR,
        2 => Order::STATUS_RUNNING,
        3 => Order::STATUS_PENDING,
    ];

    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        $registrant = $client->getRegistrant();
        $action = 'CURRENT/domain/register';
        $params = array (
            'registrant' => $registrant,
            'domain' => $domain,
            'nserver' => $name_servers
        );
        //todo make extended attributes selection for user
        list(, $tld) = explode('.', $domain, 2);
        if ($tld == 'eu') {
            $params['eu_whoispolicy'] = 'I AGREE';
        }
        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $action = 'CURRENT/domain/prolong';
        $params = array(
            'domain' => $domain,
            'years' => $years
        );
        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $action = 'CURRENT/domain/update';

        $params = array(
            'domain' => $domain,
            'nserver' => $name_servers
        );

        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function updateDomain($domain, Client $client)
    {
        $action = 'CURRENT/domain/update';

        $params = array(
            'domain' => $domain,
            'registrant' => $client->getRegistrant(),
        );

        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $action = 'CURRENT/domain/check';

        $params = [
            'domain' => $domain,
            'flag' => 28 // 4 | 8 | 16
        ];
        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);
        if ($result['data']->data->check_result == 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * @inheritdoc
     */
    public function getExtendedAttributes($domain)
    {
        $action = $action = 'CURRENT/domain/getextattributes';

        $params = [
            'domain' => $domain
        ];
        return $this->executeCommand($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        $action = 'CURRENT/domain/list';

        $params = [
            'filter-name' => 'domain',
            'filter-value' => $domain,
            'filter-cond' => 'EQ'
        ];
        $reg_till = null;

        $result = $this->executeCommand($action, $params);
        $total = (int)($result['data']->data->total_count);

        return ($total > 0);
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        $action = 'CURRENT/domain/list';

        $params = [
            'filter-name' => 'domain',
            'filter-value' => $domain,
            'filter-cond' => 'EQ'
        ];
        $reg_till = null;

        $result = $this->executeCommand($action, $params);
        $total = (int)$result['data']->data->total_count;
        if ($total > 1) {
            throw new Exception('Найдено более одного домена с именем ' . $domain);
        }
        /**
         * @var SimpleXMLElement $child
         */
        foreach ($result['data']->data->children() as $child) {
            if (preg_match('/domain_\d+/', $child->getName())) {
                $reg_till = $child->reg_till;
            }
        }

        return $reg_till;
    }

    /**
     * @inheritdoc
     */
    protected function executeCommand($action, $params)
    {
        $params['p_login'] = $this->registrar->getLogin();
        $params['p_password'] = $this->registrar->getPassword();
        $params['responsetype'] = $this->registrar->getContentType();
        return $this->makeRequest($this->registrar->getUri() . '/' . $action, $params);
    }

    /**
     * @inheritdoc
     */
    protected function checkRegistrarResponse($data)
    {
        $data = simplexml_load_string($data);
        if ((int)$data->status === 0) {
            $check = [
                'status' => true,
                'data' => $data
            ];
        } else {
            $check = [
                'status' => false,
                'code' => (int)$data->status,
                'message' => (string)$data->message,
            ];
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        if (!$check['status']) {
            if (array_key_exists('code', $check)) {
                switch ($check['code']) {
                    case 1:
                        throw new InternalException('Недопустимая зона');
                        break;
                    case 2:
                        throw new InternalException('Неверное доменное имя');
                        break;
                    case 3:
                    case 4:
                    case 5:
                        throw new ReservedDomainNameException('Доменное имя является зарезервированным');
                        break;
                    case 115:
                    case 6:
                        throw new StopListException('Доменное имя отклонено');
                        break;
                    case 7:
                        throw new DomainAlreadyRegisteredException('Доменное имя занято');
                        break;
                    case 8:
                        throw new AuthException('В доступе к домену отказано');
                        break;
                    case 9:
                        throw new InternalException('Домен не найден');
                        break;
                    case 10:
                        throw new InternalException('Доменное имя слишком короткое');
                        break;
                    case 11:
                        throw new InternalException('Доменное имя слишком длинное');
                        break;
                    case 12:
                        throw new InternalException('Операцию выполнить невозможно');
                        break;
                    case 51:
                        throw new InternalException('Доменная зона не поддерживается');
                        break;
                    case 53:
                        throw new InternalException('Операция отклонена');
                        break;
                    case 60:
                        throw new ConnectionException('Не удалось обратиться к службе Whois');
                        break;
                    case 65:
                        throw new InternalException('Операция запрещена вышестоящим регистратором');
                        break;
                    case 68:
                        throw new ManualResolvingException('Необходимо заполнить все поля контактной информации');
                        break;
                    case 70:
                        throw new InternalException('Задание уже существует в очереди');
                        break;
                    case 81:
                    case 82:
                    case 80:
                        throw new BalanceException('Недостаточно денег на балансе регистратора Domenus');
                        break;
                    case 9550:
                    case 400:
                        throw new InternalException('Неизвестная ошибка на стороне регистратора');
                        break;
                    case 9560:
                    case 402:
                        throw new InternalException('Данная операция еще не реализована');
                        break;
                    case 405:
                        throw new InternalException('Недопустимая операция дл данного домена');
                        break;
                    case 9510:
                        throw new AuthException('Ошибка авторизации');
                        break;
                    case 9515:
                        throw new InternalException('Ошибочный запрос');
                        break;
                    case 9516:
                        throw new InternalException('Недопустимая команда');
                        break;
                    case 9517:
                        throw new AuthException('Отказано в доступе к команде');
                        break;
                    case 9521:
                        throw new InternalException('Отсутствует необходимый параметр: ' . $check['message'], $check);
                        break;
                    case 9522:
                        throw new InternalException('Указанный пользователь не существует');
                        break;
                    case 9523:
                        throw new InternalException('Ошибочный параметр:' . $check['message'], $check);
                        break;
                    case 9530:
                        throw new InternalException('Недопустимый набор параметров');
                        break;
                    case 9531:
                        throw new AuthException('Отказано в доступе с этого ip');
                        break;
                    default:
                        throw new InternalException('Неизветсный код ошибки', $check);
                }
            } else {
                parent::handleCheckResult($check);
            }
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $array = [];
        $limit = 10; //max
        $offset = 0;

        $action = 'CURRENT/domain/list';
        do {
            $params['offset'] = $offset;
            $result = $this->executeCommand($action, $params);
            $total = (int)$result['data']->data->total_count;
            /**
             * @var SimpleXMLElement $child
             */
            foreach ($result['data']->data->children() as $child) {
                if (preg_match('/domain_\d+/', $child->getName())) {
                    $array[] = [
                        'domain' => (string)$child->domain,
                        'reg_till' => strtotime($child->reg_till),
                    ];
                }
            }
            $old_offset = $offset;
            $offset += $limit;
        } while (($old_offset + $limit) < $total);

        // теперь соберем домены, которые уже expired, т.к. выше метод их не возвращает
        $params = [];
        $limit = 10; //max
        $offset = 0;
        $action = 'CURRENT/domain/list/expired';
        $params['days'] = [0, -30];
        do {
            $params['offset'] = $offset;
            $result = $this->executeCommand($action, $params);
            $total = (int)$result['data']->data->total_count;
            /**
             * @var SimpleXMLElement $child
             */
            foreach ($result['data']->data->children() as $child) {
                if (preg_match('/domain_\d+/', $child->getName())) {
                    $array[] = [
                        'domain' => (string)$child->domain,
                        'reg_till' => strtotime($child->reg_till),
                    ];
                }
            }
            $old_offset = $offset;
            $offset += $limit;
        } while (($old_offset + $limit) < $total);


        return $array;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $array = [];
        $limit = 10; //max
        $offset = 0;

        $action = 'CURRENT/domain/list/expired';
        $params['reg_till'] = [
            date('Y-m-d', strtotime('now - 1 month')),
            date('Y-m-d', strtotime('now + 60 days')),
        ];

        do {
            $params['offset'] = $offset;
            $result = $this->executeCommand($action, $params);
            $total = (int)$result['data']->data->total_count;
            /**
             * @var SimpleXMLElement $child
             */
            foreach ($result['data']->data->children() as $child) {
                if (preg_match('/domain_\d+/', $child->getName())) {
                    $array[] = [
                        'domain' => (string)$child->domain,
                        'reg_till' => strtotime($child->reg_till),
                    ];
                }
            }
            $old_offset = $offset;
            $offset += $limit;
        } while (($old_offset + $limit) < $total);
        return $array;
    }

    /**
     * @inheritdoc
     */
    public function getOrder($order)
    {
        $action = 'CURRENT/order/status';

        $params = [
            'is' => $order,
        ];
        $result = $this->executeCommand($action, $params);

        $order_status = $result['data']->data->status;
        if (!isset($this->order_status_map[$order_status])) {
            throw new OrderStatusUnknownException($order_status, $this->order_status_map);
        }

        return new Order([
            'id' => $order,
            'status' => $this->order_status_map[$order_status]
        ]);
    }

    /**
     * @inheritdoc
     */
    public function registerClient(Client $client)
    {
        $action = 'CURRENT/contact/register';

        $params['address_ru'] = $client->getLegalAddress();
        $params['paddr'] = $client->getPostalAddress();
        $params['address1'] = $client->getAddressEng();
        $params['postalcode'] = $client->getPostalCodeEng();
        $params['country'] = $client->getCountryCode();
        $params['state_province'] = $client->getProvinceEng();
        $params['email'] = $client->getEmail();
        $params['city'] = $client->getCityEng();

        if ($client instanceof ClientPerson) {
            $params['contact_type'] = 'person';
            $params['name'] = $client->getNameEng();
            $params['name_ru'] = $client->getName();
            $params['firstname'] = $client->getFirstNameEng();
            $params['lastname'] = $client->getLastNameEng();
            $params['fax'] = $params['phone'] = $this->formatPhone($client->getPhone());
            $params['passport'] = $client->getPassportData();
            $params['birth_date'] = $client->getBirthDate();
        }

        if ($client instanceof ClientOrg) {
            $params['contact_type'] = 'org';
            $params['organizationname'] = $params['name'] = $client->getOrganizationNameEng();
            $params['name_ru'] = $client->getOrganizationName();
            $params['firstname'] = $client->getContactFirstNameEng();
            $params['lastname'] = $client->getContactLastNameEng();
            $params['who'] = $client->getLegalPerson();
            $params['jobtitle'] = $client->getJobTitleEng();
            $params['phone'] = $this->formatPhone($client->getPhone());
            $params['fax'] = $this->formatPhone($client->getFax());
            $params['ogrn'] = $client->getOgrn();
            $params['org_person'] = $client->getLegalPerson();
            $params['org_action_under'] = $client->getActionUnder();
            $params['kpp'] = $client->getKpp();
            $params['inn'] = $client->getInn();
            $params['bik'] = $client->getBik();
            $params['rs'] = $client->getRs();
            $params['ks'] = $client->getKs();
            $params['bank'] = $client->getBank();
        }

        if ($client instanceof ClientIp) {
            $params['contact_type'] = 'ip';
            $params['organizationname'] = $params['name'] = $client->getNameEng();
            $params['name_ru'] = $client->getName();
            $params['firstname'] = $client->getFirstNameEng();
            $params['lastname'] = $client->getLastNameEng();
            $params['who'] = $params['name_ru'];
            $params['jobtitle'] = $client->getJobTitleEng();
            $params['phone'] = $this->formatPhone($client->getPhone());
            $params['fax'] = $this->formatPhone($client->getFax());
            $params['passport'] = $client->getPassportData();
            $params['birth_date'] = $client->getBirthDate();
            $params['ogrnip'] = $client->getOgrnip();
            $params['ogrnip_passdate'] = $client->getOgrnipIssueDate();
            $params['inn'] = $client->getInn();
            $params['bik'] = $client->getBik();
            $params['rs'] = $client->getRs();
            $params['ks'] = $client->getKs();
            $params['bank'] = $client->getBank();
        }

        $result = $this->executeCommand($action, $params);
        return $result['data']->data->registrant;
    }

    /**
     * @param array $params
     * @return mixed
     */
    protected function buildQuery($params)
    {
        return str_replace(array('+', ' ', '#'), array('%2B', '%20', urlencode('#')), parent::buildQuery($params));
    }

    /**
     * @inheritdoc
     */
    public function findDomainContractNumber($domain)
    {
        $action = 'CURRENT/domain/info';
        $params = [
            'domain' => $domain
        ];
        $result = $this->executeCommand($action, $params);
        if ($result['data']->success === 0) {
            return false;
        }
        return $result['data']->data->registrant;
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $action = 'CURRENT/account/balance';
        $result = $this->executeCommand($action, []);
        return (string)$result['data']->data->personal_balance;
    }
}
