<?php
namespace Filanco\Domain\Api;

use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Model\Client;
use Filanco\Domain\RegistrarApiAbstract;

/**
 * Class R01ruApi
 * @package Filanco\Domain\Api
 */
class R01ruApi extends RegistrarApiAbstract implements BalanceFetchInterface
{
    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        if (is_array($name_servers)) {
            $name_servers = implode("\n", $name_servers);
        }
        $name_servers = trim($name_servers, "\r\n");
        $action = 'addDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'nservers' => $name_servers,
            'descr' => '',
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $action = 'prolongDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'years' => $years
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $action = 'updateDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'nservers' => $name_servers
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $action = 'checkDomainAvailable';
        $params = ['domain_name' => static::punycodeEncode($domain)];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);
        if ($result['data']->status->name === 'DOMAIN_AVAILABLE') {
            return true;
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => static::punycodeEncode($domain)],
            'strict' => 1,
            'limit' => -1
        ];
        $result = $this->execute($action, $params);
        if ($result['data']->status->name === 'RETURNING_DOMAIN_LIST') {
            return true;
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => static::punycodeEncode($domain)],
            'strict' => 1,
            'limit' => -1
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $domain = $domains[0];
        return $domain->{"reg-till"};
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $action = 'getDomains';
        $params = [
            'limit' => -1
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $result = array();
        foreach ($domains as $domain) {
            $result[] = [
                'domain' => strtolower($domain->name),
                'reg_till' => strtotime($domain->{"reg-till"})
            ];
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $action = 'getDomains';
        $params = [
            'params' => [
                'date_from' => date('d-m-Y', strtotime('-30 days')),
                'date_to' => date('d-m-Y', strtotime('+60 days')),
            ],
            'limit' => '-1'
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $result = array();
        foreach ($domains as $domain) {
            $result[] = [
                'domain' => strtolower($domain->name),
                'reg_till' => strtotime($domain->{"reg-till"})
            ];
        }
        return $result;
    }

    /**
     * @param $action
     * @param $params
     * @return mixed
     */
    protected function execute($action, $params)
    {
        $login_result = $this->makeSoapRequest(
            'logIn',
            [$this->registrar->getLogin(), $this->registrar->getPassword()],
            [
                'location' => $this->registrar->getUri(),
                'uri' => 'urn:RegbaseSoapInterface',
                'exceptions' => true,
                'user_agent' => 'RegbaseSoapInterfaceClient',
                'trace' => 1
            ],
            null,
            true
        );
        $this->soap->__setCookie('SOAPClient', $login_result['data']->status->message);
        $result = $this->makeSoapRequest($action, $params);
        return $result;
    }

    /**
     * @inheritdoc
     */
    protected function checkResponseData($data, $info)
    {
        return $data;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        return ['data' => $check];
    }

    /**
     * Получает текущий баланс сервиса
     *
     * @return mixed
     */
    public function getBalance()
    {
        $action = 'getBalanceInfo';
        $result = $this->execute($action, []);
        return $result['data']->data->balance;
    }
}
