<?php
namespace Filanco\Domain\Api;

use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Exception\Auto\BalanceException;
use Filanco\Domain\Exception\AutoResolvingException;
use Filanco\Domain\Exception\InternalException;
use Filanco\Domain\Exception\ManualResolvingException;
use Filanco\Domain\Exception\User\DomainAlreadyRegisteredException;
use Filanco\Domain\Exception\User\EarlyRenewalException;
use Filanco\Domain\Exception\User\OperationNotAllowedException;
use Filanco\Domain\Exception\UserException;
use Filanco\Domain\GetExtendedAttributesInterface;
use Filanco\Domain\IdProtectInterface;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\ClientIp;
use Filanco\Domain\Model\ClientOrg;
use Filanco\Domain\Model\ClientPerson;
use Filanco\Domain\Model\Cost;
use Filanco\Domain\Model\Domain;
use Filanco\Domain\PremiumDomainInterface;
use Filanco\Domain\RegistrarApiAbstract;
use Filanco\Domain\UpdateDomainInterface;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;

/**
 * Class EnomApi
 * @package Filanco\Domain\Api
 */
class EnomApi extends RegistrarApiAbstract implements
    IdProtectInterface,
    GetExtendedAttributesInterface,
    BalanceFetchInterface,
    UpdateDomainInterface,
    PremiumDomainInterface
{
    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        $params = $this->getDomainRegisterParams($client, $domain, $name_servers, $years);
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'Extend',
            'SLD' => $sld,
            'TLD' => $tld,
            'NumYears' => $years,
        ];
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'ModifyNS',
            'SLD' => $sld,
            'TLD' => $tld,
        ];
        $this->setNameServers($params, $name_servers);
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function idProtect($domain, $years = 1)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'PurchaseServices',
            'SLD' => $sld,
            'TLD' => $tld,
            'Service' => 'WPPS',
            'NumYears' => $years,
        ];
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function updateDomain($domain, Client $client)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'Contacts',
            'SLD' => $sld,
            'TLD' => $tld,
            'ConatactType' => 'REGISTRANT',
        ];
        //todo extended attributes
        $params = array_merge($params, $this->getClientParams($client));
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'Check',
            'SLD' => $sld,
            'TLD' => $tld,
        ];
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);
        if ($result['data']->RRPCode == 210) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * @inheritdoc
     */
    public function getExtendedAttributes($domain)
    {
        list(, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'GetExtAttributes',
            'TLD' => $tld,
        ];
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function domainStatus($domain)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'StatusDomain',
            'SLD' => $sld,
            'TLD' => $tld
        ];
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        $result = $this->checkAvailability($domain);
        if ($result) {
            return false;
        }
        $result = $this->domainStatus($domain);
        if ($result->Registrar !== 'Known' || $result->InAccount != 1) {
            return false;
        }
        return true;
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        list($sld, $tld) = explode('.', $domain, 2);
        $params = [
            'command' => 'GetDomainExp',
            'SLD' => $sld,
            'TLD' => $tld
        ];
        $result = $this->executeCommand($params);
        return $result['data']->ExpirationDate;
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $params = [
            'command' => 'AdvancedDomainSearch',
            'RecordsToReturn' => 100,
            'StartPosition' => 1,
            'OrderBy' => 'SLD'
        ];
        $result = [];
        do {
            $arr = $this->executeCommand($params);
            $total = $arr['data']->DomainSearch->TotalResults;
            foreach ($arr['data']->DomainSearch->Domains->Domain as $domain) {
                $domain['domain'] = $domain->SLD . '.' . $domain->TLD;
                $domain['reg_till'] = date('Y-m-d', strtotime($domain->ExpDate));
                $result[] = $domain;
            }
            $params['StartPosition'] += $params['RecordsToReturn'];
        } while ($params['StartPosition'] < $total);

        // не возвращает домены, которые уже EXPIRED - получим и их
        // ведь нам нужен в этом методе весь списко
        $params = [
            'command' => 'GetExpiredDomains',
            'OrderBy' => 'SLD'
        ];
        $arr = $this->executeCommand($params);
        $date = "expiration-date";
        foreach ($arr['data']->DomainDetail as $domain) {
            $domain['domain'] = $domain->DomainName;
            $domain['reg_till'] = date('Y-m-d', strtotime($domain->$date));

            $result[] = $domain;
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $params = [
            'command' => 'AdvancedDomainSearch',
            'DaysTillExpires' => 60,
            'DaysExpired' => 30,
            'RecordsToReturn' => 100,
            'StartPosition' => 1,
            'OrderBy' => 'SLD'
        ];
        $result = array();
        do {
            $arr = $this->executeCommand($params);
            $total = $arr['data']->DomainSearch->TotalResults;
            foreach ($arr['data']->DomainSearch->Domains->Domain as $domain) {
                $part['domain'] = $domain->SLD . '.' . $domain->TLD;
                $part['reg_till'] = strtotime($domain->ExpDate);
                $part['data'] = $domain;
                $result[] = $part;
            }
            $params['StartPosition'] += $params['RecordsToReturn'];
        } while ($params['StartPosition'] < $total);
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $params = [
            'command' => 'GetBalance'
        ];
        $result = $this->executeCommand($params);
        return (string)$result['data']->Balance;
    }

    /**
     * @param $phone_number
     * @return string телефон в формате +7.1234567890
     */
    public function formatPhone($phone_number)
    {
        $phone_util = PhoneNumberUtil::getInstance();
        $phone_number = $phone_util->parse($phone_number, 'RU');
        return str_replace(' ', '.', $phone_util->formatByPattern($phone_number, PhoneNumberFormat::INTERNATIONAL, []));
    }

    /**
     * @inheritdoc
     */
    public function registerPremiumDomain(Client $client, $domain, array $name_servers, $known_price, $years = 1)
    {
        $domain_register_params = $this->getDomainRegisterParams($client, $domain, $name_servers, $years);
        $domain_register_params['CustomerSuppliedPrice'] = $known_price;
        $domain_register_params['PremiumDomain'] = 'Yes';
        return $this->executeCommand($domain_register_params);
    }

    /**
     * @inheritdoc
     */
    public function getPremiumDomainsInfo(array $domains)
    {
        $response = $this->getInfo($domains);
        $result = [];
        for ($i = 1; $i < (count($domains) + 1); $i++) {
            $domain = $response['Domain' . $i];
            $params = [
                'availability' => ($response['RRPCode' . $i] == 210),
                'premium' => false,
                'domain_name' => $domain
            ];

            if (isset($response['IsPremiumName' . $i]) && !preg_match('/\.tv$/ui', $domain)) {
                $params['premium'] = true;
                $params['cost'] = new Cost([
                    'register' => $response['PremiumPrice' . $i],
                    'renewal' => $response['RenewalPrice' . $i],
                    'restore' => $response['RestorePrice' . $i],
                    'transfer' => $response['TransferPrice' . $i],
                ]);
            }

            $result[$domain] = new Domain($params);
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    protected function executeCommand($params)
    {
        $params['UID'] = $this->registrar->getLogin();
        $params['PW'] = $this->registrar->getPassword();
        $params['ResponseType'] = $this->registrar->getContentType();
        return $this->makeRequest($this->registrar->getUri(), $params);
    }

    /**
     * @param Client $client
     * @return array
     */
    protected function getClientParams(Client $client)
    {
        $params = [
            'RegistrantAddress1' => mb_substr($client->getLegalAddress(), 0, 60),
            'RegistrantCity' => $client->getCityEng(),
            'RegistrantCountry' => $client->getCountryCode(),
            'RegistrantStateProvince' => $client->getProvinceEng(),
            'RegistrantPostalCode' => $client->getPostalCodeEng(),
            'RegistrantEmailAddress' => $client->getEmail(),
            'RegistrantPhone' => $this->formatPhone($client->getPhone())
        ];

        if ($client instanceof ClientPerson || $client instanceof ClientIp) {
            $params['RegistrantFirstName'] = $client->getFirstNameEng();
            $params['RegistrantLastName'] = $client->getLastNameEng();
        }

        if ($client instanceof ClientOrg) {
            $params['RegistrantFirstName'] = $client->getFirstNameEng();
            $params['RegistrantLastName'] = $client->getLastNameEng();
            $params['RegistrantOrganizationName'] = $client->getOrganizationNameEng();
            $params['RegistrantJobTitle'] = $client->getJobTitleEng();
        }

        return $params;
    }

    /**
     * @param $params
     * @param array $name_servers
     */
    protected function setNameServers(&$params, array $name_servers)
    {
        if (count($name_servers) > 0) {
            //todo check if there is more than 12 nservers
            $i = 1;
            foreach ($name_servers as $name_server) {
                $name_server = trim($name_server, ".\n\r");
                $params['NS' . $i] = $name_server;
                $i++;
            }
        }
    }

    /**
     * @inheritdoc
     */
    protected function checkRegistrarResponse($data)
    {
        $data = simplexml_load_string($data);
        if ($data->ErrCount == 0) {
            $check = [
                'status' => true,
                'data' => $data
            ];
        } else {
            $errors = [
                RegistrarApiAbstract::ERROR_AUTO => [],
                RegistrarApiAbstract::ERROR_USER => [],
                RegistrarApiAbstract::ERROR_MANUAL => [],
                RegistrarApiAbstract::ERROR_INTERNAL => []
            ];
            foreach ($data->responses as $response) {
                $this->translateResponseCode($response, $errors);
            }
            $check = [
                'status' => false,
                'data' => $data,
                'display_error_text' => true,
            ];

            //Если существуют ошибки с более высоким приоритетом ,то игнорируем ошибки с меньшим приоритетом
            //до их исправления
            //todo elseif
            if (count($errors[RegistrarApiAbstract::ERROR_AUTO]) > 0) {
                $check['code'] = RegistrarApiAbstract::ERROR_AUTO;
                $check['message'] = implode("\n", $errors[RegistrarApiAbstract::ERROR_AUTO]);
            }
            if (count($errors[RegistrarApiAbstract::ERROR_USER]) > 0) {
                $check['code'] = RegistrarApiAbstract::ERROR_USER;
                $check['message'] = implode("\n", $errors[RegistrarApiAbstract::ERROR_USER]);
            }
            if (count($errors[RegistrarApiAbstract::ERROR_MANUAL]) > 0) {
                $check['code'] = RegistrarApiAbstract::ERROR_MANUAL;
                $check['message'] = implode("\n", $errors[RegistrarApiAbstract::ERROR_MANUAL]);
            }
            if (count($errors[RegistrarApiAbstract::ERROR_INTERNAL]) > 0) {
                $check['code'] = RegistrarApiAbstract::ERROR_INTERNAL;
                $check['message'] = implode("\n", $errors[RegistrarApiAbstract::ERROR_INTERNAL]);
            }
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        //Проверяем на наличие RRP кода специфических ошибок
        if (isset($check['data']->RRPCode)) {
            switch ($check['data']->RRPCode) {
                case '546':
                    $check['message'] = 'Недостаточно средств на балансе парнерского договора с Enom';
                    throw new BalanceException($check['message']);
                    break;
                case '548':
                    $check['message'] = 'Продлять домен рано';
                    throw new EarlyRenewalException($check['message']);
                    break;
                case '552':
                    $check['message'] = 'Невозможно совершить требуемую операцию';
                    throw new OperationNotAllowedException($check['message']);
                    break;
                case '540':
                    $check['message'] = 'Доменное имя уже существует';
                    throw new DomainAlreadyRegisteredException($check['message']);
                    break;
                case '555':
                    $check['message'] = 'Идентичный запрос на продление доменного имени уже был выполнен';
                    throw new InternalException($check['message'], $check['data']);
                    break;
                default:
                    break;
            }
        }

        if ($check['status'] === false) {
            if (array_key_exists('code', $check)) {
                switch ($check['code']) {
                    case RegistrarApiAbstract::ERROR_AUTO:
                        throw new AutoResolvingException($check['message']);
                        break;
                    case RegistrarApiAbstract::ERROR_USER:
                        throw new UserException($check['message']);
                        break;
                    case RegistrarApiAbstract::ERROR_MANUAL:
                        throw new ManualResolvingException($check['message']);
                        break;
                    case RegistrarApiAbstract::ERROR_INTERNAL:
                        throw new InternalException($check['message'], $check['data']);
                        break;
                    default:
                        parent::handleCheckResult($check);
                        break;
                }
            } else {
                parent::handleCheckResult($check);
            }
        }

        return $check;
    }

    /**
     * Анализирует код ошибки, определяет ее тип и генерирует текст ошибки на русском.
     * @param $response
     * @param $errors
     * @throws BalanceException
     * @throws InternalException
     */
    protected function translateResponseCode($response, &$errors)
    {
        $response = $response->response;
        $code = (string)$response->ResponseNumber;
        $error_text = '';
        $type = RegistrarApiAbstract::ERROR_AUTO;
        // Response number - xxxxxx (6 digits)
        // System bit      - dxxxxx
        // Error bit       - xddxxx
        // Parameter bit   - xxxddd
        $system_bit = mb_substr($code, 0, 1);
        $error_bit = mb_substr($code, 1, 2);
        $parameter_bit = mb_substr($code, 3);
        //todo 304174 - premium domain
        switch ($system_bit) {
            case '0':
                throw new InternalException('Произошла неизвестная ошибка.', var_export($response, true));
                break;
            case '2':
                $error_text .= 'Ошибка реестра:' . "\n";
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '3':
                $error_text .= 'Ошибка валидации:' . "\n";
                $type = RegistrarApiAbstract::ERROR_USER;
                break;
            case '4':
                $error_text .= 'Ошибка авторизации:' . "\n";
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '5':
                $error_text .= 'Невозможно совершить платеж:' . "\n";
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '6':
                $error_text .= 'Внутреняя ошибка Enom:' . "\n";
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '7':
                $error_text .= 'Нарушение политики:' . "\n";
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            default:
                throw new InternalException('Не могу обработать системный бит.', var_export($response, true));
                break;
        }

        switch ($error_bit) {
            case '00':
                $error_text .= 'Невозможно применить: ';
                break;
            case '01':
                $error_text .= 'Отсутствует: ';
                break;
            case '02':
                $error_text .= 'Значение уже существует: ';
                break;
            case '03':
                $error_text .= 'Значение не попадает в диапазон: ';
                break;
            case '04':
                $error_text .= 'Неверное значение: ';
                break;
            case '05':
                $error_text .= 'Тип значения не соответствует: ';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '06':
                if ($system_bit != '7') {
                    $error_text .= 'Нарушение политики :' . "\n";
                }
                break;
            case '07':
                $error_text .= 'Ошибка подключения к объекту: ';
                $this->setErrorType(RegistrarApiAbstract::ERROR_AUTO, $type);
                break;
            case '08':
                $error_text .= 'Невозможно получить объект: ';
                break;
            case '09':
                $error_text .= 'Невозможно добавить объект: ';
                break;
            case '10':
                $error_text .= 'Невозможно обновить объект: ';
                break;
            case '11':
                $error_text .= 'Невозможно удалить объект: ';
                break;
            case '12':
                $error_text .= 'Выключен: ';
                break;
            case '13':
                $error_text .= 'Неавторизован: ';
                break;
            case '14':
                $error_text .= 'Отклонено ';
                break;
            case '15':
                $error_text .= 'Объект недоступен: ';
                break;
            case '16':
                $error_text .= 'Объект не найден: ';
                break;
            case '17':
                $error_text .= 'Время действия объекта истекло: ';
                break;
            case '18':
                $error_text .= 'Закрыто на обслуживание: ';
                $this->setErrorType(RegistrarApiAbstract::ERROR_AUTO, $type);
                break;
            case '19':
                $error_text .= 'Не удалось удержать: ';
                break;
            case '20':
                $error_text .= 'Не удалось отправить: ';
                break;
            case '24':
                $error_text .= 'Недостаточно';
                break;
            case '99':
                $error_text .= 'Причина неизвестна';
                throw new InternalException($error_text, var_export($response, true));
                break;
                break;
            default:
                throw new InternalException('Не могу обработать бит ошибки.', var_export($response, true));
                break;
        }

        switch ($parameter_bit) {
            case '001':
            case '019':
            case '021':
            case '022':
            case '023':
            case '024':
                $error_text .= 'Имя';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '002':
            case '025':
            case '026':
            case '027':
            case '028':
            case '029':
                $error_text .= 'Фамилия';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '003':
            case '030':
            case '031':
            case '032':
            case '033':
            case '034':
            case '195':
                $error_text .= 'Адрес';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '004':
            case '040':
            case '041':
            case '042':
            case '043':
            case '044':
                $error_text .= 'Город';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '005':
            case '045':
            case '046':
            case '047':
            case '048':
            case '049':
                $error_text .= 'Область';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '006':
            case '050':
            case '051':
            case '052':
            case '053':
            case '054':
                $error_text .= 'Индекс';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '007':
            case '055':
            case '056':
            case '057':
            case '058':
            case '059':
                $error_text .= 'Страна';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '008':
            case '060':
            case '061':
            case '062':
            case '063':
            case '064':
                $error_text .= 'Должность';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '009':
                $error_text .= 'Название организации';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '010':
            case '065':
            case '066':
            case '067':
            case '068':
            case '069':
                $error_text .= 'Номер телефона';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '011':
            case '070':
            case '071':
            case '072':
            case '073':
            case '074':
                $error_text .= 'Факс';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '012':
            case '075':
            case '076':
            case '077':
            case '078':
            case '079':
                $error_text .= 'Email';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '145':
            case '146':
            case '147':
            case '148':
            case '149':
            case '232':
            case '290':
                $error_text .= 'Контактные данные';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '150':
                $error_text .= 'Команда';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '151':
                $error_text .= 'SLD';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '152':
                $error_text .= 'TLD';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '153':
                $error_text .= 'Доменное имя';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '154':
                $error_text .= 'Дата истечения';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '155':
                $error_text .= 'Логин или пароль';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '158':
                $error_text .= 'Имя Name server-а';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '159':
                $error_text .= 'IP Name server-а';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '163':
                $error_text .= 'Настройки продления';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '166':
                $error_text .= 'Connection timeout';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '167':
                $error_text .= 'Session timeout';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '174':
                $error_text .= 'Цена';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '181':
                $error_text .= 'Баланс лицевого счета';
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                if ($code == '524181') {
                    throw new BalanceException('Недостаточно денег на балансе регистратора');
                }
                break;
            case '182':
                $error_text .= 'Тип продукта';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '196':
                $error_text .= 'Месяц';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '197':
                $error_text .= 'Год';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '207':
                $error_text .= 'Год';
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '208':
                $error_text .= 'Адрес не соотвествует остальной информации';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '209':
                $error_text .= 'Индекс не соотвествует остальной информации';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '217':
                $error_text .= 'Дата';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '220':
                $error_text .= 'Услуга';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '237':
                $error_text .= 'Достигнут предел количества символов';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '239':
                $error_text .= 'Статус';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '247':
                $error_text .= 'Reseller API';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '248':
                $error_text .= 'Reseller';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '258':
                $error_text .= 'Авторизация';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '285':
                $error_text .= 'Name server';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '289':
                $error_text .= 'ДОполнительная контактная информация';
                $this->setErrorType(RegistrarApiAbstract::ERROR_USER, $type);
                break;
            case '380':
                $error_text .= 'Домен не найден на парнерском договоре';
                $this->setErrorType(RegistrarApiAbstract::ERROR_INTERNAL, $type);
                break;
            case '385':
                $error_text .= 'Adult';
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '386':
                $error_text .= 'Hard core';
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '387':
                $error_text .= 'China';
                $this->setErrorType(RegistrarApiAbstract::ERROR_MANUAL, $type);
                break;
            case '999':
                $error_text .= 'Неизвестный объект';
                throw new InternalException($error_text, var_export($response, true));
                break;
            default:
                throw new InternalException('Не могу обработать бит параметра.', var_export($response, true));
                break;
        }

        $errors[$type][] = $error_text;
    }

    /**
     * Выставляет тип ошибки в зависимости от ранее выставленного.
     * Если ранее ошибка имела более высокий приоритет, то выставляемое значение будет проигнорировано.
     *
     * @param $new_type
     * @param int $old_type
     */
    protected function setErrorType($new_type, &$old_type = 0)
    {
        if ($new_type > $old_type) {
            $old_type = $new_type;
        }
    }

    /**
     * @inheritdoc
     */
    protected function buildQuery($params)
    {
        return str_replace(' ', '%20', parent::buildQuery($params));
    }

    /**
     * @param Client $client
     * @param $domain
     * @param array $name_servers
     * @param $years
     * @return array
     */
    protected function getDomainRegisterParams(Client $client, $domain, array $name_servers, $years)
    {
        list($sld, $tld) = static::getSldTld($domain);
        $params = [
            'command' => 'Purchase',
            'SLD' => $sld,
            'TLD' => $tld,
            'NumYears' => $years,
        ];
        $this->setNameServers($params, $name_servers);
        if (preg_match('/^xn--(.*)/', $sld)) {
            $params['IDNCode'] = 'rus';
        }
        $params = array_merge($params, $this->getClientParams($client));
        return $params;
    }

    /**
     * @param array $domains
     * @return array
     */
    protected function getInfo(array $domains)
    {
        $params = [
            'command' => 'Check',
            'DomainList' => implode(',', $domains),
        ];

        return (array)$this->executeCommand($params)['data'];
    }
}
