<?php
namespace Filanco\Domain\Api;

use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Exception\Auto\ConnectionException;
use Filanco\Domain\Exception\AutoResolvingException;
use Filanco\Domain\Exception\InternalException;
use Filanco\Domain\Exception\ManualResolvingException;
use Filanco\Domain\Exception\OrderStatusUnknownException;
use Filanco\Domain\Exception\ParseResponseException;
use Filanco\Domain\FindDomainContractNumberInterface;
use Filanco\Domain\GetOrderInterface;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\ClientIp;
use Filanco\Domain\Model\ClientOrg;
use Filanco\Domain\Model\ClientPerson;
use Filanco\Domain\Model\Order;
use Filanco\Domain\RegistrarApiAbstract;
use Filanco\Whois;

/**
 * Class NicruApi
 * @package Filanco\Domain\Api
 */
class NicruApi extends RegistrarApiAbstract implements
    GetOrderInterface,
    FindDomainContractNumberInterface,
    BalanceFetchInterface
{
    protected $order_status_map = [
        'ok' => Order::STATUS_SUCCESS,
        'waiting' => Order::STATUS_PENDING,
        'failed' => Order::STATUS_ERROR,
        'deleted' => Order::STATUS_SUCCESS,
        'running' => Order::STATUS_RUNNING,
    ];

    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        $data = "request:order\n";
        $data .= "operation:create\n";
        $data .= "subject-contract:" . $client->getRegistrant() . "\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[order-item]\n";
        $data .= "service:domain\n";
        $data .= "template:client_ru\n";
        $data .= "descr:Registered by hoster.ru\n";
        $data .= "action:new\n";
        $data .= "domain:" . $domain . "\n";
        $this->setNameServers($data, $name_servers);
        return $this->executeCommand($data);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $registrant = $this->findDomainContractNumber($domain);
        $data = "request:order\n";
        $data .= "operation:create\n";
        $data .= "subject-contract:" . $registrant . "\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[order-item]\n";
        $data .= "action:prolong\n";
        $data .= "service:domain\n";
        $data .= "template:client_ru\n";
        $data .= "domain:" . mb_strtoupper($domain) . "\n";
        $data .= "prolong:" . $years . "\n";
        return $this->executeCommand($data);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $registrant = $this->findDomainContractNumber($domain);
        $data = "request:order\n";
        $data .= "operation:create\n";
        $data .= "subject-contract:" . $registrant . "\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[order-item]\n";
        $data .= "action:update\n";
        $data .= "service:domain\n";
        $data .= "template:client_ru\n";
        $data .= "descr:Registered by hoster.ru\n";
        $data .= "domain:" . mb_strtoupper($domain) . "\n";
        $this->setNameServers($data, $name_servers);
        return $this->executeCommand($data);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $whois = new Whois($domain);
        return $whois->isAvailable();
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        return $this->checkDomain($domain);
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        $data = "request:domain\n";
        $data .= "operation:search\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[domain]\n";
        $data .= "domain:" . mb_strtoupper($domain) . "\n";

        $result =$this->executeCommand($data);

        $result = explode("\n\n", $result['data']);
        foreach ($result as $part) {
            if (strpos($part, '[domain-list]') !== false) {
                $part = explode("\n", $part);
                foreach ($part as $line) {
                    list($key, $val) = explode(':', $line);
                    if ($key === 'domains-found') {
                        if ($val == 1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;

    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain_name)
    {
        $data = "request:domain\n";
        $data .= "operation:search\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[domain]\n";
        $data .= "domain:" . mb_strtoupper($domain_name) . "\n";

        $result =$this->executeCommand($data);

        $result = explode("\n\n", $result['data']);
        foreach ($result as $part) {
            if (strpos($part, '[domain]') !== false) {
                $part = explode("\n", $part);
                $domain = array();
                foreach ($part as $line) {
                    $parts = explode(':', $line);
                    if (count($parts) == 2) {
                        list($key, $val) = $parts;
                        if ($key === 'domain') {
                            $domain['name'] = $val;
                        }
                        if ($key === 'end-date') {
                            list($day, $month, $year) = explode('.', $val);
                            $domain['reg_till'] = date('Y-m-d', mktime(0, 0, 0, $month, $day, $year));
                            if (isset($domain['name']) && $domain['name'] === mb_strtoupper($domain_name)) {
                                return $domain['reg_till'];
                            }
                        }
                    }
                }
            }
        }
        return null;
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $data = "request:domain\n";
        $data .= "operation:search\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[domain]\n";

        $result = $this->executeCommand($data);
        $list = array();
        $result = explode("\n\n", $result['data']);
        foreach ($result as $part) {
            if (strpos($part, '[domain]') !== false) {
                $part = explode("\n", $part);
                $domain = array();
                foreach ($part as $line) {
                    $parts = explode(':', $line);
                    if (count($parts) == 2) {
                        list($key, $val) = $parts;
                        if ($key === 'domain') {
                            $domain['domain'] = $val;
                        }
                        if ($key === 'end-date') {
                            list($day, $month, $year) = explode('.', $val);
                            $domain['reg_till'] = date('Y-m-d', mktime(0, 0, 0, $month, $day, $year));
                        }
                    }
                }
                $list[] = $domain;
            }
        }
        return $list;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $data = "request:domain\n";
        $data .= "operation:search\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[domain]\n";
        $data .= "end-date-start:" . date('d.m.Y', strtotime('-30 days')) . "\n";
        $data .= "end-date-end:" .  date('d.m.Y', strtotime('+60 days')) . "\n";

        $result = $this->executeCommand($data);
        $list = array();
        $result = explode("\n\n", $result['data']);
        foreach ($result as $part) {
            if (strpos($part, '[domain]') !== false) {
                $part = explode("\n", $part);
                $domain = array();
                foreach ($part as $line) {
                    $parts = explode(':', $line);
                    if (count($parts) == 2) {
                        list($key, $val) = $parts;
                        if ($key === 'domain') {
                            $domain['domain'] = $val;
                        }
                        if ($key === 'end-date') {
                            list($day, $month, $year) = explode('.', $val);
                            $domain['reg_till'] = mktime(0, 0, 0, $month, $day, $year);
                        }
                    }
                }
                $list[] = $domain;
            }
        }
        return $list;
    }

    /**
     * @inheritdoc
     */
    public function getOrder($order)
    {
        $data = "request:order\n";
        $data .= "operation:get\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[order]\n";
        $data .= "order_id" . $order . "\n";

        $result = $this->executeCommand($data);
        $result = explode("\n\n", $result['data']);
        $status = false;
        foreach ($result as $part) {
            if (strpos($part, '[order]') !== false) {
                $part = explode("\n", $part);
                foreach ($part as $line) {
                    if (strpos($line, 'state') !== false) {
                        list(, $status) = explode(':', $line);
                    }
                }
            }
        }

        $order_status = false;
        if ($status === 'completed') {
            foreach ($result as $part) {
                if (strpos($part, '[order-item]') !== false) {
                    $part = explode("\n", $part);
                    foreach ($part as $line) {
                        if (strpos($line, 'state') !== false) {
                            list(, $order_status) = explode(':', $line);
                        }
                    }
                }
            }
        }

        if ($status === false || !isset($this->order_status_map[$order_status])) {
            throw new OrderStatusUnknownException($status, $this->order_status_map);
        }

        return new Order([
            'id' => $order,
            'status' => $this->order_status_map[$order_status]
        ]);
    }

    /**
     * @inheritdoc
     */
    public function registerClient(Client $client)
    {
        if ($client instanceof ClientIp || $client instanceof ClientOrg) {
            $client_fax = $this->formatPhone($client->getFax());
        } else {
            $client_fax = '';
        }

        $data = "request:contract\n";
        $data .= "operation:create\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[contract]\n";
        $data .= "password:" . substr(md5($client->getId()), 0, 5) . "\n";
        $data .= "currency_id:RUR\n";
        $data .= "country:" . $client->getCountryCode() . "\n";
        $data .= "p-addr:" . $client->getPostalAddress() . "\n";
        $data .= "phone:+" . $this->formatPhone($client->getPhone()) . "\n";
        $data .= "fax-no:" . $client_fax . "\n";
        $data .= "e-mail:" . $client->getEmail() . "\n";
        $data .= "mnt-nfy:support@hoster.ru\n";

        if ($client instanceof ClientPerson) {
            $data .= "contract-type:PRS\n";
            $data .= "person:" . $client->getNameEng() . "\n";
            $data .= "person-r:" . $client->getName() . "\n";
            $data .= "passport:" . $client->getPassportData() . "\n";
            $data .= "passport:" . 'зарегистрирован по адресу' . $client->getLegalAddress() . "\n";
            $data .= "birth-date:" . date('d.m.Y', strtotime($client->getBirthDate())) . "\n";
        }

        if ($client instanceof ClientIp) {
            $data .= "contract-type:PRS\n";
            $data .= "person:" . $client->getNameEng() . "\n";
            $data .= "person-r:ИП " . $client->getName() . "\n";
            $data .= "passport:" . $client->getPassportData() . "\n";
            $data .= "passport:" . 'зарегистрирован по адресу ' . $client->getLegalAddress() . "\n";
            $data .= "address-r:" . $client->getPostalAddress() . "\n";
            $data .= "code:" . $client->getInn() . "\n";
            $data .= "birth-date:" . date('d.m.Y', strtotime($client->getBirthDate())) . "\n";
        }

        if ($client instanceof ClientOrg) {
            $data .= "contract-type:ORG\n";
            $data .= "org:" . $client->getOrganizationNameEng() . "\n";
            $data .= "org-r:" . $client->getOrganizationName() . "\n";
            $data .= "code:" . $client->getInn() . "\n";
            $data .= "kpp:" . $client->getKpp() . "\n";
            $data .= "address-r:" . $client->getLegalAddress() . "\n";
        }

        $result = $this->executeCommand($data);
        $result = explode("\n\n", $result['data']);

        $registrant = null;
        foreach ($result as $part) {
            if (strpos($part, 'login') !== false) {
                $part = explode("\n", $part);
                foreach ($part as $line) {
                    list($key, $val) = explode(':', $line);
                    if ($key === 'login') {
                        $registrant = $val;
                        break;
                    }
                }
            }
        }
        
        if (!$registrant) {
            throw new ParseResponseException('Не найден ключ login!', $result);
        }

        return $registrant;
    }

    /**
     * @return string
     */
    protected function generateRequest()
    {
        return date('YmdHis') . '.' . getmygid() . "@filanco.ru";
    }

    /**
     * @param $data
     * @param array $name_servers
     */
    protected function setNameServers(&$data, array $name_servers)
    {
        if (count($name_servers) > 0) {
            foreach ($name_servers as $name_server) {
                $name_server = trim($name_server, ".\n\r");
                $data .= "nserver:" . $name_server . "\n";
            }
        }
    }

    /**
     * @param $data
     * @return mixed
     */
    protected function executeCommand($data)
    {
        $data = "lang:ru\n"
            . "login:" . $this->registrar->getLogin() . "\n"
            . "password:" . $this->registrar->getPassword() . "\n"
            . $data;
        $data = trim($data, "\n");
        $data = mb_convert_encoding($data, 'KOI8-R', 'UTF-8');
        $params['SimpleRequest'] = rawurlencode($data);
        $result = $this->makeRequest(
            'https://' . $this->registrar->getUri(),
            $params,
            'POST',
            array(),
            'form'
        );
        return $result;
    }

    /**
     * @inheritdoc
     */
    protected function checkRegistrarResponse($data)
    {
        $check = array();
        $data = mb_convert_encoding($data, 'UTF-8', 'KOI8-R');
        $content = explode("\r\n\r\n", $data, 2);
        if (count($content) !== 0) {
            $header = $content[0];
            $body = '';
            if (count($content) === 2) {
                $body = $content[1];
            }
            $code = mb_substr(trim($header), 7, 3);
            if ($code != 200) {
                $check['status'] = false;
                $check['code'] = $code;
                $check['message'] = $body;
            } else {
                $check['status'] = true;
                $check['data'] = $body;
            }
        } else {
            $check['status'] = false;
            $check['code'] = 0;
            $check['message'] = 'Регистратор не ответил на запрос';
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        if (!$check['status']) {
            if (array_key_exists('code', $check)) {
                switch ($check['code']) {
                    case 0:
                        throw new AutoResolvingException('Регистратор не ответил');
                        break;
                    case 400:
                        throw new InternalException('Неправильный заголовок запроса. ' . $check['message'], $check);
                        break;
                    case 401:
                        throw new InternalException('Ошибка авторизации: неверный или отсутствующий пароль', $check);
                        break;
                    case 402:
                        throw new ManualResolvingException('Ошибки в теле запроса. ' . $check['message']);
                        break;
                    case 403:
                        throw new InternalException('Запрос такого типа не может быть выполнен', $check);
                        break;
                    case 404:
                        throw new ManualResolvingException('Запрашиваемый объект не найден. ' . $check['message']);
                        break;
                    case 405:
                        throw new ConnectionException('Превышение допустимого количества запросов');
                        break;
                    case 500:
                        throw new AutoResolvingException('Внутренняя ошибка модуля, обслуживающего запросы');
                        break;
                    case 501:
                        throw new ConnectionException('База данных регистратора временно недоступна');
                        break;
                    case 502:
                        throw new ConnectionException('Сервер обработки запросов временно не доступен');
                        break;
                    default:
                        throw new InternalException('Неизветсный код ошибки', $check);
                }
            } else {
                parent::handleCheckResult($check);
            }
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    public function findDomainContractNumber($domain)
    {
        $data = "request:contract\n";
        $data .= "operation:search\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "\n";
        $data .= "[contract]\n";
        $data .= "domain:" . $domain . "\n";
        $result = $this->executeCommand($data);
        $result = explode("\n\n", $result['data']);
        if (count($result) !== 2) {
            return false;
        }
        $contract_string = substr($result[1], strpos($result[1], 'contract-num'));
        list($contract_string,) = explode("\n", $contract_string, 2);
        list(,$contract_number) = explode(":", $contract_string, 2);
        return $contract_number;
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $data = "request:account\n";
        $data .= "operation:get\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $result = $this->executeCommand($data);
        $result = $result['data'];
        $balance = substr($result, strpos($result, 'balance'));
        $balance = explode("\n", $balance, 2);
        $balance = explode(":", $balance[0], 2);
        $balance = trim($balance[1], ' RUR');
        return $balance;
    }

    public function getDataFromClientProfile($registrant)
    {
        $data = "lang:ru\n";
        $data .= "request:contract\n";
        $data .= "operation:get\n";
        $data .= "request-id:" . $this->generateRequest() . "\n";
        $data .= "subject-contract:" . $registrant . "\n";

        return $this->executeCommand($data);
    }
}
