<?php
namespace Filanco\Domain;

use Filanco\ApiClient\ApiClientAbstract;
use Filanco\Domain\Exception\RegistrarApiHandleCheckResultException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use TrueBV\Punycode;

/**
 * Class RegistrarApiAbstract
 * @package Filanco\Domain
 */
abstract class RegistrarApiAbstract extends ApiClientAbstract implements RegistrarApiInterface
{
    const
        ERROR_AUTO = 0,
        ERROR_USER = 1,
        ERROR_MANUAL = 2,
        ERROR_INTERNAL = 3;

    /**
     * @var RegistrarInterface
     */
    protected $registrar;

    /**
     * @var RegistrarApiLogInterface
     */
    protected $log;
    protected $test;

    /**
     * @param RegistrarInterface $registrar
     * @param RegistrarApiLogInterface $logger
     * @param $test
     */
    public function __construct(RegistrarInterface $registrar, RegistrarApiLogInterface $logger, $test)
    {
        $this->registrar = $registrar;
        $this->test = $test;
        $this->log = $logger;
    }

    protected function logStart()
    {
        $this->log->setWho($this->registrar->getName());
    }

    /**
     * @param $url
     * @param $query
     * @param $params
     */
    protected function logRequest($url, $query, $params)
    {
        $this->log->setRequest($url . '?' . $query);
    }

    /**
     * @param $response
     * @param $raw_data
     */
    protected function logResponse($response, $raw_data)
    {
        $this->log->setResponse(var_export($response, true));
    }

    protected function logEnd()
    {
        $this->log->setStamp(time());
        $this->log->save();
    }


    /**
     * @param $data
     * @param $info
     * @return array
     */
    protected function checkResponseData($data, $info)
    {
        if ($info['http_code'] == 200) {
            $check = $this->checkRegistrarResponse($data);
        } else {
            $check['status'] = false;
            $check['message'] = 'Curl response is not 200. Response: ' . var_export($info, true);
            $check['display_error_text'] = false;
        }
        return $check;
    }

    /**
     * @param $data
     * @return array
     */
    protected function checkRegistrarResponse($data)
    {
        $data = json_decode($data, true);
        $check = ['status' => true, 'data' => $data];
        if ($data['error'] === true) {
            $check['status'] = false;
            $check['message'] = $data['status']['message'];
            $check['display_error_text'] = false;
        }
        return $check;
    }

    /**
     * @param $check
     * @return mixed
     * @throws RegistrarApiHandleCheckResultException
     */
    protected function handleCheckResult($check)
    {
        if ($check['status'] === false) {
            throw new RegistrarApiHandleCheckResultException($check['message']);
        }
        return $check;
    }

    /**
     * @param $domain
     * @return array
     */
    public static function getSldTld($domain)
    {
        return explode('.', $domain, 2);
    }

    public static function punycodeEncode($domain)
    {
        $punycode = new Punycode();
        return $punycode->encode($domain);
    }

    public static function punycodeDecode($domain)
    {
        $punycode = new Punycode();
        return $punycode->decode($domain);
    }

    /**
     * @param $phone_number
     * @return string телефон в формате +7 123 4567890
     */
    public function formatPhone($phone_number)
    {
        $phone_util = PhoneNumberUtil::getInstance();
        $phone_number = $phone_util->parse($phone_number, 'RU');
        $phone_number = str_replace('-', '', $phone_util->format($phone_number, PhoneNumberFormat::INTERNATIONAL));
        if (preg_match('/(\+\d+\s\d+\s)(.*)/ui', $phone_number, $matches)) {
            $phone_number = $matches[1] . str_replace(' ', '', $matches[2]);
        }
        return $phone_number;
    }
}
