<?php
namespace Filanco\Domain\Model;

use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberUtil;

/**
 * Class Client
 * @package Filanco\Domain\Model
 *
 * @method string getId() Идентификатор клиента
 * @method string getLegalAddress() Адрес регистрации
 * @method string getPostalAddress() Почтовый адрес
 * @method string getAddressEng() Адрес на англ.
 * @method string getStreetEng() Адрес без города на англ.
 * @method string getCity() Город
 * @method string getCityEng() Город на англ.
 * @method string getPostalCode() Почтовый индекс
 * @method string getPostalCodeEng() Почтовый индекс на англ.
 * @method string getProvince() Область
 * @method string getProvinceEng() Штат/Область на англ.
 * @method string getCountryCode() Двухбуквенный код страны
 * @method string getEmail() Адрес электронной почты
 * @method string getRegistrant() Регистрант
 */
abstract class Client extends BaseModel
{
    const
        TYPE_PERSON = 'person',
        TYPE_ORG = 'org',
        TYPE_IP = 'ip';

    protected function fields()
    {
        return [
            'id' => 'Идентификатор клиента',
            'legal_address' => 'Адрес регистрации',
            'postal_address' => 'Почтовый адрес',
            'address_eng' => 'Адрес на англ.',
            'street_eng' => 'Адрес без города на англ.',
            'city' => 'Город',
            'city_eng' => 'Город на англ.',
            'postal_code' => 'Почтовый индекс',
            'postal_code_eng' => 'Почтовый индекс на англ.',
            'province' => 'Область',
            'province_eng' => 'Штат/Область на англ.',
            'country_code' => 'Двухбуквенный код страны',
            'email' => 'Адрес электронной почты',
            'phone' => 'Номер телефона',
            'registrant' => 'Регистрант',
        ];
    }

    protected function validators()
    {
        return [
            'legal_address' => function ($value) {
                return !empty($value);
            },
            'postal_address' => function ($value) {
                return !empty($value);
            },
            'city' => function ($value) {
                return !empty($value);
            },
            'city_eng' => function ($value) {
                return !empty($value);
            },
            'postal_code' => function ($value) {
                return !empty($value);
            },
            'postal_code_eng' => function ($value) {
                return !empty($value);
            },
            'province' => function ($value) {
                return !empty($value);
            },
            'province_eng' => function ($value) {
                return !empty($value);
            },
            'country_code' => function ($value) {
                return !empty($value) && strlen($value) == 2;
            },
            'email' => function ($value) {
                return filter_var($value, FILTER_VALIDATE_EMAIL);
            },
            'phone' => function ($value) {
                $phone_util = PhoneNumberUtil::getInstance();
                try {
                    $phone_util->parse($value, 'RU');
                    $valid_number = true;
                } catch (NumberParseException $e) {
                    $valid_number = false;
                }
                return !empty($value) && $valid_number;
            },
            'registrant' => function ($value) {
                return !empty($value);
            },
        ];
    }

    /**
     * @param $data
     * @return Client
     */
    public static function factory($data)
    {
        $class_name = __NAMESPACE__ . '\\Client' . ucfirst($data['type']);
        unset($data['type']);
        return new $class_name($data);
    }

    public function getPhone()
    {
        $phone = sprintf('+%s', preg_replace('/[^0-9]+/ui', '', $this->fields['phone']));
        $this->validate('phone', $phone);
        return $phone;
    }
}
