<?php
namespace Filanco\Domain\Api;

use Exception;
use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Exception\ManualResolvingException;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\ClientIp;
use Filanco\Domain\Model\ClientOrg;
use Filanco\Domain\Model\ClientPerson;
use Filanco\Domain\Model\Domain;
use Filanco\Domain\RegistrarApiAbstract;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;

/**
 * Class WebnamesApi
 * @package Filanco\Domain\Api
 */
class WebnamesApi extends RegistrarApiAbstract implements BalanceFetchInterface
{
    /** @var int сколько секунд ждать ответа */
    protected $timeout = 3600;

    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        $domain = static::punycodeEncode($domain);
        list(, $tld) = $this->getSldTld($domain);
        if (!in_array(static::punycodeDecode($tld), ['moscow', 'москва', 'by', 'tatar', 'дети'])) {
            throw new Exception('Домен в этой зоне зарегистрировать нельзя.');
        }

        $params = [
            'thisPage' => 'pispRegistration',
            'domain_name' => $domain,
            'period' => $years,
        ];

        $this->setNameServers($params, $name_servers);
        $this->setContacts($params, $client);

        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $params = [
            'thisPage' => 'pispRenewDomain',
            'domain_name' => static::punycodeEncode($domain),
            'period' => $years,
        ];

        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $params = [
            'thisPage' => 'pispRedelegation',
            'domain_name' => static::punycodeEncode($domain),
        ];
        $this->setNameServers($params, $name_servers);
        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $params = [
            'thisPage' => 'pispCheckDomain',
            'domain_name' => static::punycodeEncode($domain),
        ];

        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);
        return !empty($result['data']['avail']);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailabilityList($domains)
    {
        $domain_list = [];
        $params = [
            'thisPage' => 'pispCheckDomain'
        ];
        $i = 0;
        foreach ($domains as $domain) {
            $domain = static::punycodeEncode($domain);
            if ($i) {
                $params['domain_name' . $i] = $domain;
            } else {
                $params['domain_name'] = $domain;
            }
            $i++;
        }

        $result = $this->executeCommand($params);

        foreach ($result['data']['avail'] as $avail) {
            $d_name = static::punycodeEncode($avail['dname']);
            $params = [
                'availability' => 1,
                'domain_name' => $d_name
            ];
            $domain_list[$d_name] = new Domain($params);
        }
        foreach ($result['data']['unavail'] as $avail) {

            $d_name = static::punycodeEncode($avail['dname']);
            $params = [
                'availability' => 0,
                'domain_name' => $d_name
            ];
            $domain_list[$d_name] = new Domain($params);
        }

        return $domain_list;
    }

    /**
     * @param $domain
     * @return mixed
     */
    public function getDomainInfo($domain)
    {
        $params = [
            'thisPage' => 'pispDomainInfo',
            'domain_name' => static::punycodeEncode($domain),
        ];

        return $this->executeCommand($params);
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        try {
            $this->getDomainInfo($domain);
        } catch (Exception $e) {
            return false;
        }
        return true;
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        $result = $this->getDomainInfo($domain);
        return $result['data']['expiration_date'];
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $params = [
            'thisPage' => 'pispAllDomainsInfo'
        ];
        $result = $this->executeCommand($params);
        $list = array();
        foreach ($result['data'] as $domain) {
            $list[] = [
                'domain' => $domain['dname'],
                'reg_till' => date('Y-m-d', strtotime($domain['ExpirationDate'])),
                'free_date' => date('Y-m-d', strtotime($domain['ExpirationDate'] . ' + 30 days')),
            ];
        }
        return $list;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $params = [
            'thisPage' => 'pispAllDomainsInfo'
        ];
        $result = $this->executeCommand($params);
        $expiration_date = date('Y-m-d', strtotime('+60 days'));

        $list = array();
        foreach ($result['data'] as $domain) {
            if ($domain['ExpirationDate'] < $expiration_date) {
                $list[] = [
                    'domain' => $domain['dname'],
                    'reg_till' => strtotime($domain['ExpirationDate']),
                ];
            }
        }
        return $list;
    }

    /**
     * @param $data
     * @param array $name_servers
     */
    protected function setNameServers(&$data, array $name_servers)
    {
        if (count($name_servers) > 0) {
            //todo check if there is more than 4 nservers
            $i = 0;
            foreach ($name_servers as $name_server) {
                $name_server = trim($name_server, ".\n\r");
                $data['ns' . $i] = $name_server;
                $i++;
            }
        }
    }

    /**
     * @param $data
     * @param Client $client
     */
    protected function setContacts(&$data, Client $client)
    {
        $phone = $this->formatPhone($client->getPhone());
        $data['country'] = $client->getCountryCode();
        $data['addr_lang'] = $client->getCountryCode();
        $data['o_phone'] = $phone;
        $data['o_email'] = $client->getEmail();
        $data['addr_street'] = $client->getAddressEng();
        $data['addr_city'] = $client->getCity();
        $data['addr_sp'] = $client->getProvince();
        $data['addr_pc'] = $client->getPostalCode();
        $data['addr_cc'] = $client->getCountryCode();

        if ($client instanceof ClientOrg) {
            $data['company_name'] = $client->getOrganizationNameEng();
            $data['company_tin'] = $client->getInn();
            $data['legal_lang'] = $client->getCountryCode();
            $data['legal_street'] = $client->getLegalAddressEng();
            $data['legal_city'] = $client->getCityEng();
            $data['legal_sp'] = $client->getProvinceEng();
            $data['legal_pc'] = $client->getPostalCodeEng();
        }

        if ($client instanceof ClientPerson || $client instanceof ClientIp) {
            $data['person_r'] = $client->getNameEng();
            $data['birth_date'] = date("d.m.Y", strtotime($client->getBirthDate()));
        }
    }

    /**
     * @param $data
     * @return mixed
     * @throws Exception
     */
    protected function executeCommand($data)
    {
        $data['interface_revision'] = 2;
        $data['interface_lang'] = 'ru';
        $data['utf8'] = 1;
        $data['username'] = $this->registrar->getLogin();
        $data['password'] = $this->registrar->getPassword();

        return $this->makeRequest($this->registrar->getUri(), $data, 'POST');
    }

    /**
     * @inheritdoc
     */
    protected function checkRegistrarResponse($data)
    {
        $data = json_decode($data, true);
        if (isset($data['success'])) {
            $check = [
                'status' => true,
                'data' => $data['success']
            ];
        } else {
            $check = [
                'status' => false,
                'message' => $data['error'],
                'display_error_text' => true,
            ];
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        if (!$check['status']) {
            throw new ManualResolvingException($check['message']);
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $params = [
            'thisPage' => 'pispBalance',
            'ssl_version' => false
        ];
        $result = $this->executeCommand($params);
        $balance = trim($result['data'], 'balance -');
        $balance = trim($balance, 'RUR');
        $balance = trim($balance);
        return $balance;
    }

    /**
     * @param $phone_number
     * @return string телефон в формате +7.1234567890
     */
    public function formatPhone($phone_number)
    {
        $phone_util = PhoneNumberUtil::getInstance();
        $phone_number = $phone_util->parse($phone_number, 'RU');
        return str_replace(' ', '.', $phone_util->formatByPattern($phone_number, PhoneNumberFormat::INTERNATIONAL, []));
    }

    /**
     * @param string $error - тип ошибки
     * @return string
     */
    protected static function getError($error)
    {
        return self::ERROR_WHILE_DOMAIN_CHECK;
    }
}
