<?php
namespace Filanco\Domain\Api;

use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Exception\ApiException;
use Filanco\Domain\Exception\Auto\BalanceException;
use Filanco\Domain\Exception\Auto\ConnectionException;
use Filanco\Domain\Exception\AutoResolvingException;
use Filanco\Domain\Exception\Internal\AuthException;
use Filanco\Domain\Exception\InternalException;
use Filanco\Domain\Exception\ManualResolvingException;
use Filanco\Domain\Exception\ParseResponseException;
use Filanco\Domain\Exception\User\DomainAlreadyRegisteredException;
use Filanco\Domain\Exception\User\DomainExpiredException;
use Filanco\Domain\Exception\User\EarlyRenewalException;
use Filanco\Domain\Exception\User\StopListException;
use Filanco\Domain\Exception\User\UnableToRegisterException;
use Filanco\Domain\Exception\UserException;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\ClientIp;
use Filanco\Domain\Model\ClientOrg;
use Filanco\Domain\Model\ClientPerson;
use Filanco\Domain\Model\Cost;
use Filanco\Domain\Model\Domain;
use Filanco\Domain\PremiumDomainInterface;
use Filanco\Domain\RegistrarApiAbstract;
use Filanco\Domain\UpdateDomainInterface;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;

/**
 * Class RegruApi
 * @package Filanco\Domain\Api
 */
class RegruApi extends RegistrarApiAbstract implements UpdateDomainInterface, BalanceFetchInterface, PremiumDomainInterface
{
    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        $domain = static::punycodeEncode($domain);
        list(, $tld) = static::getSldTld($domain);
        $action = 'domain/create';
        $data = [
            'domain_name' => $domain,
            'period' => $years,
            //todo hardcode
            'enduser_ip' => '31.31.205.43',
            'not_delegated' => 0,
        ];

        $this->setContacts($data, $client, $tld);
        $this->setNameServers($data, $name_servers);

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $action = 'service/renew';
        $data = [
            'period' => $years,
            'domain_name' => static::punycodeEncode($domain),
        ];

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $action = 'domain/update_nss';
        $data = [
            'domain_name' => static::punycodeEncode($domain),
            'undelegate' => 0,
        ];

        $this->setNameServers($data, $name_servers);

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    public function updateDomain($domain, Client $client)
    {
        $domain = static::punycodeEncode($domain);
        list(, $tld) = static::getSldTld($domain);
        $action = 'domain/update_contacts';
        $data = [
            'domains' => [
                'domain_name' => $domain,
            ],
        ];

        $this->setContacts($data, $client, $tld);

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $action = 'domain/check';
        $data = [
            'domain_name' => static::punycodeEncode($domain),
        ];

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);

        foreach ($result['data']['domains'] as $domain_result) {
            if ($domain_result['dname'] === $domain && $domain_result['result'] === 'Available') {
                return true;
            }
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function checkAvailabilityList($domains)
    {
        // подготовим свой формат данных
        $domain_list = [];
        foreach ($domains as $domain) {
            $domain_list[] = ['dname' => static::punycodeEncode($domain)];
        }

        $action = 'domain/check';
        $data = [
            'domains' => $domain_list
        ];

        $result = $this->executeCommand($action, $data);

        $domain_list = [];
        foreach ($result['data']['domains'] as $domain_result) {
            if (!empty($domain_result['error_code'])) {
                if ($domain_result['error_code'] == 'DOMAIN_ALREADY_EXISTS') {
                    $params = [
                        'availability' => 0,
                        'domain_name' => $domain
                    ];

                    $premium_costs = $this->getPremiumCost($domain);
                    if (!empty($premium_costs['data']['domains'][0]['renew_price'])) {
                        $premium_costs_info = $premium_costs['data']['domains'][0];
                        $params['is_premium'] = true;
                        $params['cost'] = new Cost([
                            'renewal' => $premium_costs_info['renew_price'] * 1.2,
                        ]);
                    }

                    $domain_list[$domain_result['dname']] = new Domain($params);
                } else {
                    // ошибка
                    $domain_list[$domain_result['dname']] = self::getError($domain_result['result']);
                }
            } else {
                $params = [
                    'availability' => intval($domain_result['result'] === 'Available'),
                    'domain_name' => $domain
                ];

                if(isset($domain_result['is_premium'])) {
                    $params['is_premium'] = true;
                    $params['cost'] = new  Cost(array(
                        'register' => $domain_result['price'] * 1.2,
                        'renewal' => $domain_result['renew_price'] * 1.2,
                        'restore' => $domain_result['price'] * 1.2,
                        'transfer' => $domain_result['price'] * 1.2,
                    ));
                }

                $domain_list[$domain_result['dname']] = new Domain($params);
            }
        }

        return $domain_list;
    }

    /**
     * @param $action
     * @param $data
     * @return string
     */
    protected function executeCommand($action, $data)
    {
        array_walk_recursive($data, function (&$item) {
            $item = rawurlencode($item);
        });

        $params = [
            'username' => $this->registrar->getLogin(),
            'password' => $this->registrar->getPassword(),
            'input_format' => 'json',
            'lang' => 'ru',
            'input_data' => json_encode($data, JSON_UNESCAPED_UNICODE),
        ];

        return $this->makeRequest($this->registrar->getUri() . '/' . $action, $params);
    }

    /**
     * @param $data
     * @param Client $client
     * @param $tld
     * @throws UserException
     */
    protected function setContacts(&$data, Client $client, $tld)
    {
        $first_name_en = '';
        $first_name_ru = '';
        $last_name_en = '';
        $last_name_ru = '';
        $fax = '';
        if ($client instanceof ClientPerson || $client instanceof ClientIp) {
            $first_name_en = mb_substr($client->getFirstNameEng(), 0, 40);
            $first_name_ru = mb_substr($client->getFirstName(), 0, 40);
            $last_name_en = mb_substr($client->getLastNameEng(), 0, 40);
            $last_name_ru = mb_substr($client->getLastName(), 0, 40);
        }

        if ($client instanceof ClientOrg) {
            $first_name_en = mb_substr($client->getContactFirstNameEng(), 0, 40);
            $first_name_ru = mb_substr($client->getContactFirstName(), 0, 40);
            $last_name_en = mb_substr($client->getContactLastNameEng(), 0, 40);
            $last_name_ru = mb_substr($client->getContactLastName(), 0, 40);
            $fax = $this->formatPhone($client->getFax());
        }

        if ($client instanceof ClientIp) {
            $fax = $this->formatPhone($client->getFax());
        }

        switch ($tld) {
            case 'moscow':
            case static::punycodeEncode('москва'):
                $contact_prefixes = ['o', 'a', 't'];
                break;
            case static::punycodeEncode('дети'):
                $contact_prefixes = ['o', 'a', 't'];
                $data['contacts']['description'] = 'My site';
                break;
            default:
                $contact_prefixes = ['o', 'a', 't', 'b'];
        }

        $phone = $this->formatPhone($client->getPhone()); // formta +7.1234567890
        $phone_ru = $this->formatPhone($client->getPhone(), 'ru'); // formta +7 495 1234567
        foreach ($contact_prefixes as $prefix) {
            $data['contacts'][$prefix . '_email'] = $client->getEmail();
            $data['contacts'][$prefix . '_phone'] = $phone;
            $data['contacts'][$prefix . '_fax'] = $fax ?: '';
            $data['contacts'][$prefix . '_addr'] = mb_substr($client->getLegalAddressEng(), 0, 60);
            $data['contacts'][$prefix . '_city'] = $client->getCityEng();
            $data['contacts'][$prefix . '_state'] = $client->getProvinceEng();
            $data['contacts'][$prefix . '_postcode'] = $client->getPostalCodeEng();
            $data['contacts'][$prefix . '_country_code'] = $client->getCountryCode();
            $data['contacts'][$prefix . '_first_name'] = $first_name_en;
            $data['contacts'][$prefix . '_last_name'] = $last_name_en;
            // именно поле company тепрь и определяет - организация или физ лицо ты
            if ($client instanceof ClientOrg) {
                $data['contacts'][$prefix . '_company'] = $client->getOrganizationNameEng();
            } else {
                $data['contacts'][$prefix . '_company'] = 'Private Person';
            }
        }
        //todo Зоны asia, pp.ua, es, jobs, travel, us, pro, lv, ro требуют особые свойства,
        //todo которые должен задать клиент
        switch ($tld) {
            case 'ru':
            case 'su':
            case static::punycodeEncode('рф'):
                $data['contacts']['sms_security_number'] = $client->getPhone();
                $data['contacts']['p_addr'] = mb_substr($client->getLegalAddress(), 0, 200);
                $data['contacts']['phone'] = $phone_ru;
                $data['contacts']['e_mail'] = $client->getEmail();
                $data['contacts']['country'] = $client->getCountryCode();

                // ИНН
                $data['contacts']['code'] = $client->getInn();

                if ($client instanceof ClientPerson) {
                    $data['contacts']['person'] = $client->getNameEng();
                    $data['contacts']['person_r'] = $client->getName();
                    $data['contacts']['private_person_flag'] = 1;
                    $data['contacts']['passport'] = $client->getPassportData();
                    $data['contacts']['birth_date'] = date("d.m.Y", strtotime($client->getBirthDate()));
                }

                if ($client instanceof ClientIp) {
                    $data['contacts']['person'] = $client->getNameEng();
                    $data['contacts']['person_r'] = $client->getName();
                    $data['contacts']['private_person_flag'] = 1;
                    $data['contacts']['passport'] = $client->getPassportData();
                    $data['contacts']['birth_date'] = date("d.m.Y", strtotime($client->getBirthDate()));

                }

                if ($client instanceof ClientOrg) {
                    $data['contacts']['org'] = $client->getOrganizationNameEng();
                    $data['contacts']['org_r'] = $client->getOrganizationName();
                    $data['contacts']['kpp'] = $client->getKpp();
                    $data['contacts']['address_r'] = mb_substr($client->getLegalAddress(), 0, 200);
                }
                break;
            case 'tj':
                if ($client instanceof ClientPerson) {
                    $data['contacts']['o_type'] = 1;
                    $data['contacts']['o_full_name'] = mb_substr($client->getNameEng(), 0, 60);
                    $data['contacts']['a_full_name'] = $data['contacts']['o_full_name'];
                    $data['contacts']['t_full_name'] = $data['contacts']['o_full_name'];
                    $data['contacts']['a_nic_name'] = mb_substr($client->getFirstNameEng(), 0, 32);
                    $data['contacts']['t_nic_name'] = $data['contacts']['a_nic_name'];
                }
                if ($client instanceof ClientIp) {
                    $data['contacts']['o_type'] = 2; //Юридическое лицо
                    $data['contacts']['o_full_name'] = mb_substr($client->getNameEng(), 0, 60);
                    $data['contacts']['a_full_name'] = $data['contacts']['o_full_name'];
                    $data['contacts']['t_full_name'] = $data['contacts']['o_full_name'];
                    $data['contacts']['a_nic_name'] = mb_substr($client->getFirstNameEng(), 0, 32);
                    $data['contacts']['t_nic_name'] = $data['contacts']['a_nic_name'];
                }
                if ($client instanceof ClientOrg) {
                    $data['contacts']['o_type'] = 2; //Юридическое лицо
                    $data['contacts']['o_full_name'] = mb_substr($client->getOrganizationNameEng(), 0, 60);
                    $data['contacts']['a_full_name'] = mb_substr($client->getLegalPersonEng(), 0, 60);
                    $data['contacts']['t_full_name'] = $data['contacts']['a_full_name'];
                    $data['contacts']['a_nic_name'] = mb_substr($client->getFirstNameEng(), 0, 32);
                    $data['contacts']['t_nic_name'] = $data['contacts']['a_nic_name'];
                }

                $data['contacts']['o_whois'] = '';
                $data['contacts']['o_phone'] = '+' . substr($phone, 0, 1) . '.' . str_replace(' ', '',
                        substr($phone, 1));
                $data['contacts']['o_addr'] = mb_substr($client->getLegalAddressEng(), 0, 120);
                $data['contacts']['a_addr'] = $data['contacts']['o_addr'];
                $data['contacts']['t_addr'] = $data['contacts']['o_addr'];
                $data['contacts']['o_city'] = $client->getCityEng();
                $data['contacts']['a_city'] = $data['contacts']['o_city'];
                $data['contacts']['t_city'] = $data['contacts']['o_city'];
                $data['contacts']['o_country_code'] = $client->getCountryCode();
                $data['contacts']['a_country_code'] = $client->getCountryCode();
                $data['contacts']['t_country_code'] = $client->getCountryCode();
                $data['contacts']['o_email'] = $client->getEmail();
                $data['contacts']['a_email'] = $client->getEmail();
                $data['contacts']['t_email'] = $client->getEmail();
                $data['contacts']['a_postcode'] = $client->getPostalCode();
                $data['contacts']['t_postcode'] = $client->getPostalCode();
                break;
            case 'kz':
                $data['contacts']['srvloc_state'] = 'KAR';
                $data['contacts']['srvloc_city'] = 'Karaganda';
                $data['contacts']['srvloc_street'] = 'Chizhevskogo, 17';
                break;
            case static::punycodeEncode('сайт'):
            case static::punycodeEncode('онлайн'):
                foreach ($contact_prefixes as $prefix) {
                    $data['contacts'][$prefix . '_addr_r'] = mb_substr($client->getLegalAddress(), 0, 250);
                    $data['contacts'][$prefix . '_city_r'] = $client->getCity();

                    if ($client instanceof ClientOrg) {
                        $data['contacts'][$prefix . '_company_r'] = $client->getOrganizationName();
                        $data['contacts'][$prefix . '_first_name_r'] = $client->getContactFirstName();
                        $data['contacts'][$prefix . '_last_name_r'] = $client->getContactLastName();
                    } else {
                        $data['contacts'][$prefix . '_company_r'] = 'Частное лицо';
                        $data['contacts'][$prefix . '_first_name_r'] = $client->getFirstName();
                        $data['contacts'][$prefix . '_last_name_r'] = $client->getLastName();
                    }
                }
                break;
            case 'moscow':
            case static::punycodeEncode('москва'):
            case static::punycodeEncode('дети'):
                foreach ($contact_prefixes as $prefix) {
                    if (isset($data['contacts'][$prefix . '_fax'])) {
                        unset($data['contacts'][$prefix . '_fax']);
                    }
                    $data['contacts'][$prefix . '_phone'] = $phone_ru;

                    $data['contacts'][$prefix . '_addr_ru'] = mb_substr($client->getLegalAddress(), 0, 250);
                    $data['contacts'][$prefix . '_city_ru'] = $client->getCity();
                    $data['contacts'][$prefix . '_state_ru'] = $client->getProvince();
                    // eng
                    $data['contacts'][$prefix . '_addr'] = mb_substr($client->getStreetEng(), 0, 80);
                    $data['contacts'][$prefix . '_city'] = $client->getCityEng();
                    $data['contacts'][$prefix . '_state'] = $client->getProvinceEng();

                    // ИНН
                    $data['contacts'][$prefix . '_code'] = $client->getInn();

                    if (empty($data['contacts'][$prefix . '_code'])) {
                        $data['contacts'][$prefix . '_code'] = '123456789';
                    }

                    if ($client instanceof ClientIp || $client instanceof ClientPerson) {
                        $data['contacts'][$prefix . '_first_name_ru'] = $first_name_ru;
                        $data['contacts'][$prefix . '_last_name_ru'] = $last_name_ru;
                        $data['contacts'][$prefix . '_patronimic'] = mb_substr($client->getMiddleNameEng(), 0, 25);
                        $data['contacts'][$prefix . '_patronimic_ru'] = $client->getMiddleName();
                        $data['contacts'][$prefix . '_birth_date'] = date("d.m.Y", strtotime($client->getBirthDate()));
                        $data['contacts'][$prefix . '_passport_date'] = date("d.m.Y",
                            strtotime($client->getPassportIssueDate()));
                        $passport_number = str_replace(" ", '', $client->getPassportNumber());
                        $passport_number = substr($passport_number, 0, 2) . ' ' . substr($passport_number, 2, 2) . ' '
                            . substr($passport_number, 4);
                        $data['contacts'][$prefix . '_passport_number'] = $passport_number;
                        $data['contacts'][$prefix . '_passport_place'] = $client->getPassportPlace();
                        unset($data['contacts'][$prefix . '_company']);
                    } else {
                        $data['contacts'][$prefix . '_first_name_ru'] = $first_name_ru;
                        $data['contacts'][$prefix . '_last_name_ru'] = $last_name_ru;
                        $data['contacts'][$prefix . '_patronimic'] = mb_substr($client->getContactMiddleNameEng(), 0,
                            25);
                        $data['contacts'][$prefix . '_patronimic_ru'] = $client->getContactMiddleName();
                        $data['contacts'][$prefix . '_company_ru'] = $client->getOrganizationName();
                        $data['contacts'][$prefix . '_l_addr_ru'] = $client->getLegalAddress();
                        $data['contacts'][$prefix . '_l_addr'] = $client->getLegalAddressEng();
                        $data['contacts'][$prefix . '_l_city_ru'] = $client->getCity();
                        $data['contacts'][$prefix . '_l_city'] = $client->getCityEng();
                        $data['contacts'][$prefix . '_l_state_ru'] = $client->getProvince();
                        $data['contacts'][$prefix . '_l_state'] = $client->getProvinceEng();
                        $data['contacts'][$prefix . '_l_postcode'] = $data['contacts'][$prefix . '_postcode'];
                    }
                }
                break;
            case 'com.ua':
                if (! $client instanceof ClientOrg) {
                    foreach ($contact_prefixes as $prefix) {
                        $data['contacts'][$prefix . '_company'] = 'OOO Filanko';
                    }
                }
                break;

            default:
                break;
        }
    }

    /**
     * @param $data
     * @param array $name_servers
     */
    protected function setNameServers(&$data, array $name_servers)
    {
        if (count($name_servers) > 0) {
            //todo check if there is more than 4 nservers
            $i = 0;
            foreach ($name_servers as $name_server) {
                $name_server = trim($name_server, ".\n\r");
                $data['nss']['ns' . $i] = $name_server;
                $i++;
            }
        }
    }

    /**
     * @inheritdoc
     */
    protected function checkRegistrarResponse($data)
    {
        $data = json_decode($data, true);
        switch ($data['result']) {
            case 'success':
                $check = [
                    'status' => true,
                    'data' => $data['answer']
                ];
                break;
            case 'error':
                $check = [
                    'status' => false,
                    'display_error_text' => true,
                    'message' => $data['error_text'],
                    'code' => $data['error_code'],
                ];
                break;
            default:
                $check = [
                    'status' => false,
                    'display_error_text' => true,
                    'message' => 'Невозможно обработать ответ сервиса, поле result содержит недокументированное значение',
                ];
                break;
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        /**
         * @todo ошибка ApiException может не говорит о том что домен не у нас
         */
        try {
            $this->getDomainId($domain);
            return true;
        } catch (ApiException $e) {
            return false;
        }
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        $action = 'service/get_info';
        $data = [
            'domain_name' => static::punycodeEncode($domain),
        ];
        $data = $this->executeCommand($action, $data);
        if (!isset($data['data']['services'][0]['expiration_date'])) {
            throw new ParseResponseException('Не найден индекс expiration_date', $data);
        }
        return $data['data']['services'][0]['expiration_date'];
    }

    /**
     * @param $domain
     * @return string
     */
    protected function getDomainId($domain)
    {
        $action = 'domain/nop';
        $data = [
            'dname' => static::punycodeEncode($domain)
        ];

        return $this->executeCommand($action, $data);
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        if (!$check['status']) {
            if (array_key_exists('code', $check)) {
                switch ($check['code']) {
                    case 'NO_USERNAME':
                    case 'NO_AUTH':
                    case 'PASSWORD_AUTH_FAILED':
                    case 'RESELLER_AUTH_FAILED':
                    case 'ACCESS_DENIED':
                    case 'PURCHASES_DISABLED':
                        throw new AuthException($check['message']);
                        break;
                    case 'DOMAIN_NOT_FOUND':
                    case 'SERVICE_NOT_FOUND':
                    case 'SERVICE_NOT_SPECIFIED':
                    case 'SERVICE_ID_NOT_FOUND':
                    case 'NO_DOMAIN':
                    case 'INVALID_DOMAIN_NAME_FORMAT':
                    case 'INVALID_SERVICE_ID':
                    case 'INVALID_DOMAIN_NAME_PUNYCODE':
                    case 'BAD_USER_SERVID':
                    case 'USER_SERVID_IS_NOT_UNIQUE':
                    case 'TOO_MANY_OBJECTS_IN_ONE_REQUEST':
                    case 'CLIENT_NOT_FOUND':
                    case 'DOMAIN_BAD_NAME_ONLYDIGITS':
                    case 'HAVE_MIXED_CODETABLES':
                    case 'DOMAIN_BAD_TLD':
                    case 'TLD_DISABLED':
                    case 'DOMAIN_NAME_MUSTBEENG':
                    case 'DOMAIN_NAME_MUSTBERUS':
                    case 'DOMAIN_INVALID_LENGTH':
                    case 'NO_DOMAINS_CHECKED':
                    case 'NO_CONTRACT':
                    case 'INVALID_PUNYCODE_INPUT':
                    case 'NOT_FOUND_UNIQUE_REQUIRED_DATA':
                    case 'ORDER_ALREADY_PAYED':
                    case 'UNAVAILABLE_DOMAIN_ZONE':
                    case 'NO_SUCH_COMMAND':
                    case 'HTTPS_ONLY':
                    case 'INTERNAL_ERROR':
                    case 'SERVICE_OPERATIONS_DISABLED':
                    case 'UNSUPPORTED_CURRENCY':
                    case 'PRICES_NOT_FOUND':
                    case 'SERVICE_ALREADY_EXISTS':
                    case 'NSS_NOT_FOUND':
                    case 'DOMAIN_BAD_NAME':
                        throw new InternalException($check['message']);
                        break;
                    case 'BILLING_LOCK':
                    case 'DOMAIN_ORDER_LOCKED':
                    case 'SERVICE_UNAVAILIBLE':
                        throw new AutoResolvingException($check['message']);
                        break;
                    case 'DOMAIN_ALREADY_EXISTS':
                        throw new DomainAlreadyRegisteredException('Доменное имя уже занято');
                        break;
                    case 'DOMAIN_STOP_LIST':
                        throw new StopListException('Домен находится в стоп-листе');
                        break;
                    case 'DOMAIN_STOP_PATTERN':
                        throw new UnableToRegisterException('Невозможно зарегистрировать домен');
                        break;
                    case 'DOMAIN_ALREADY_ORDERED':
                    case 'CANT_OBTAIN_EXPDATE':
                    case 'DOMAIN_CLIENT_TRANSFER_PROHIBITED':
                    case 'DOMAIN_TRANSFER_PROHIBITED_UNKNOWN':
                    case 'DOMAIN_REGISTERED_VIA_DIRECTI':
                    case 'DOUBLE_ORDER':
                    case 'PARAMETER_MISSING':
                    case 'PARAMETER_INCORRECT':
                    case 'FREE_DATE_IN_FUTURE':
                    case 'INVALID_CONTACTS':
                        throw new ManualResolvingException($check['message']);
                        break;
                    case 'CONNECTION_FAILED':
                        throw new ConnectionException($check['message']);
                        break;
                    case 'DOMAIN_TOO_YOUNG':
                        throw new EarlyRenewalException($check['message']);
                        break;
                    case 'DOMAIN_EXPIRED':
                        throw new DomainExpiredException($check['message']);
                        break;
                    case 'NOT_ENOUGH_MONEY':
                        throw new BalanceException('Не хватает средств на балнсе Reg.ru');
                        break;
                    case 'DOMAIN_IS_PREMIUM':
                        throw new UserException('Данный домен является премиальным');
                        break;
                    default:
                        throw new InternalException('Неизвестный тип ощибки', $check);
                        break;
                }
            } else {
                parent::handleCheckResult($check);
            }
        }
        return $check;
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $action = 'service/get_list';
        $data = [
            'servtype' => 'domain'
        ];
        //todo refactor
        return array_values(array_filter(array_map(function ($x) {
                    $x['free_date'] = date('Y-m-d', strtotime($x['expiration_date'] . ' + 30 days'));
                    $x['reg_till'] = date('Y-m-d', strtotime($x['expiration_date']));
                    $x['domain'] = $x['dname'];
                    return (array)$x;
                }, $this->executeCommand($action, $data)['data']['services'])));
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $action = 'service/get_list';
        $data = [
            'servtype' => 'domain'
        ];
        //todo refactor
        $expiration_date = date('Y-m-d', strtotime('+60 days'));
        return array_values(array_filter(array_map(function ($x) use ($expiration_date) {
                    if ($x['expiration_date'] < $expiration_date) {
                        $date_arr = explode('-', $x['expiration_date']);
                        $x['reg_till'] = mktime(0, 0, 0, $date_arr[1], $date_arr[2], $date_arr[0]);
                        $x['domain'] = $x['dname'];
                        return (array)$x;
                    }
                    return null;
                }, $this->executeCommand($action, $data)['data']['services'])));
    }

    /**
     * @inheritdoc
     */
    public function getBalance()
    {
        $action = 'user/get_balance';
        $result = $this->executeCommand($action, ['currency' => 'RUR']);
        return $result['data']['prepay'];
    }

    /**
     * @param $phone_number
     * @return string телефон в формате +7.1234567890
     */
    public function formatPhone($phone_number, $code = '')
    {
        $phone_util = PhoneNumberUtil::getInstance();
        $phone_number = $phone_util->parse($phone_number, 'RU');
        $phone_number = str_replace('-', '', $phone_util->format($phone_number, PhoneNumberFormat::INTERNATIONAL));
        if ($code == 'ru') {
            if (preg_match('/(\+\d+\s\d+\s)(.*)/ui', $phone_number, $matches)) {
                $phone_number = $matches[1] . str_replace(' ', '', $matches[2]);
            }
        } else {
            if (preg_match('/(\+\d+\s)(.*)/ui', $phone_number, $matches)) {
                $phone_number = $matches[1] . '.' . str_replace(' ', '', $matches[2]);
            }
        }

        return $phone_number;
    }

    protected static function getError($error) {
        switch ($error) {
            case 'DOMAIN_BAD_TLD':
                return self::ERROR_WRONG_ZONE;

            case 'TLD_DISABLED':
                return self::ERROR_UNSUPPORTED_ZONE;

            case 'INVALID_DOMAIN_NAME_PUNYCODE':
            case 'DOMAIN_BAD_NAME':
            case 'INVALID_DOMAIN_NAME_FORMAT':
            case 'DOMAIN_INVALID_LENGTH':
            case 'HAVE_MIXED_CODETABLES':
            case 'DOMAIN_BAD_NAME_ONLYDIGITS':
            case 'DOMAIN_NAME_MUSTBEENG':
            case 'DOMAIN_INVALID_LENGTH':
                return self::ERROR_WRONG_DOMAIN_NAME;

            default:
                return self::ERROR_WHILE_DOMAIN_CHECK;

        }
    }

    /**
     * Запрос на регистрацию премиального домена
     * @param Client $client
     * @param string $domain
     * @param array $name_servers
     * @param float $known_price предполагаемая стоимость домена
     * @param int $years
     * @return mixed
     */
    public function registerPremiumDomain(Client $client, $domain, array $name_servers, $known_price, $years = 1)
    {
        $this->getPremiumCost($domain);
        $domain = static::punycodeEncode($domain);
        list(, $tld) = static::getSldTld($domain);
        $action = 'domain/create';
        $data = [
            'domain_name' => $domain,
            'period' => $years,
            //todo hardcode
            'enduser_ip' => '31.31.205.43',
            'not_delegated' => 0,
            'reg_premium' => 1,
        ];

        $this->setContacts($data, $client, $tld);
        $this->setNameServers($data, $name_servers);

        return $this->executeCommand($action, $data);
    }

    /**
     * Запрос на продление премиального домена
     * @param string $domain
     * @param float $known_price предполагаемая стоимость домена
     * @param int $years
     * @return mixed
     */
    public function prolongPremiumDomain($domain, $known_price, $years = 1)
    {
        $this->getPremiumCost($domain);
        $action = 'service/renew';
        $data = [
            'period' => $years,
            'domain_name' => static::punycodeEncode($domain),
        ];

        return $this->executeCommand($action, $data);
    }

    /**
     * check and get premium cost for registration premium domain
     * @param $domain
     * @return string
     */
    public function getPremiumCost($domain)
    {
        $domain = static::punycodeEncode($domain);
        $action = 'domain/get_premium_prices';

        $inputData = [
            'currency' => 'RUR',
            'domains' => [
                $domain
            ]
        ];
//
//        $data = [
//            'input_data' => urlencode(json_encode($inputData))
//        ];

        return $this->executeCommand($action, $inputData);
    }
}


