<?php
namespace Filanco\Domain\Api;

use Exception;
use Filanco\Domain\BalanceFetchInterface;
use Filanco\Domain\Exception\InternalException;
use Filanco\Domain\Exception\UndefinedPropertyException;
use Filanco\Domain\FindDomainContractNumberInterface;
use Filanco\Domain\GetDomainNs;
use Filanco\Domain\Model\Client;
use Filanco\Domain\Model\Domain;
use Filanco\Domain\Model\Registrant;
use Filanco\Domain\RegistrantDataInterface;
use Filanco\Domain\RegistrarApiAbstract;

/**
 * Class R01ruApi
 * @package Filanco\Domain\Api
 */
class R01ruApi extends RegistrarApiAbstract implements
    BalanceFetchInterface,
    FindDomainContractNumberInterface,
    RegistrantDataInterface,
    GetDomainNs
{
    /**
     * @inheritdoc
     */
    public function registerDomain(Client $client, $domain, array $name_servers, $years = 1)
    {
        if (is_array($name_servers)) {
            $name_servers = implode("\n", $name_servers);
        }
        $name_servers = trim($name_servers, "\r\n");
        $action = 'addDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'nservers' => $name_servers,
            'descr' => '',
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function prolongDomain($domain, $years = 1)
    {
        $action = 'prolongDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'years' => $years
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function redeligateDomain($domain, array $name_servers)
    {
        $action = 'updateDomain';
        $params = [
            'domain' => static::punycodeEncode($domain),
            'nservers' => $name_servers
        ];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkDomain($domain)
    {
        $action = 'checkDomainAvailable';
        $params = ['domain_name' => static::punycodeEncode($domain)];
        return $this->execute($action, $params);
    }

    /**
     * @inheritdoc
     */
    public function checkAvailability($domain)
    {
        $result = $this->checkDomain($domain);
        if ($result['data']->status->name === 'DOMAIN_AVAILABLE') {
            return true;
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function checkAvailabilityList($domains)
    {
        $domains_list = [];
        foreach ($domains as $domain) {
            $domain = static::punycodeEncode($domain);
            try {
                $result = $this->checkDomain($domain);

                $params = [
                    'domain_name' => $domain
                ];

                $result_status = $result['data']->status->name;
                if ($result_status === 'DOMAIN_AVAILABLE') {
                    $params['availability'] = 1;

                    $domains_list[$domain] = new Domain($params);
                } elseif ($result_status === 'DOMAIN_ALREADY_EXISTS') {
                    $params['availability'] = 0;

                    $domains_list[$domain] = new Domain($params);
                } else {
                    // ошибка
                    $domains_list[$domain] = self::getError($result_status);
                }
            } catch (Exception $e) {
                // ошибка
                $domains_list[$domain] = self::getError($e->getMessage());
            }

        }

        return $domains_list;
    }

    /**
     * @inheritdoc
     */
    public function checkIfOurs($domain)
    {
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => static::punycodeEncode($domain)],
            'strict' => 1,
            'limit' => -1
        ];
        $result = $this->execute($action, $params);
        if (!empty($result['data']->data->domainarray)) {
            return true;
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function getExpirationDate($domain)
    {
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => static::punycodeEncode($domain)],
            'strict' => 1,
            'limit' => -1
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $domain = $domains[0];
        return $domain->{"reg-till"};
    }

    /**
     * @inheritdoc
     */
    public function getDomainList()
    {
        $action = 'getDomains';
        $params = [
            'params' => [
                'date_from' => '01-01-1900'
            ],
            'strict' => 1,
            'sort_field' => 'domain',
            'sort_dir' => 'asc',
            'limit' => -1,
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $result = array();
        foreach ($domains as $domain) {
            $result[] = [
                'domain' => strtolower($domain->name),
                'reg_till' => date('Y-m-d', strtotime($domain->{"reg-till"})),
                'free_date' => date('Y-m-d', strtotime($domain->{"reg-till"} . ' + 30 days'))
            ];
        }
        return $result;
    }

    /**
     * @inheritdoc
     */
    public function getExpiredDomainList()
    {
        $action = 'getDomains';
        $params = [
            'params' => [
                'date_from' => date('d-m-Y', strtotime('-30 days')),
                'date_to' => date('d-m-Y', strtotime('+60 days')),
            ],
            'limit' => '-1'
        ];
        $result = $this->execute($action, $params);
        $domains = (array)$result['data']->data->domainarray;
        $result = array();
        foreach ($domains as $domain) {
            $result[] = [
                'domain' => strtolower($domain->name),
                'reg_till' => strtotime($domain->{"reg-till"})
            ];
        }
        return $result;
    }

    /**
     * @param $action
     * @param $params
     * @return mixed
     */
    protected function execute($action, $params)
    {
        $login_result = $this->makeSoapRequest(
            'logIn',
            [$this->registrar->getLogin(), $this->registrar->getPassword()],
            [
                'location' => $this->registrar->getUri(),
                'uri' => 'urn:RegbaseSoapInterface',
                'exceptions' => true,
                'user_agent' => 'RegbaseSoapInterfaceClient',
                'trace' => 1
            ],
            null,
            true
        );
        $this->soap->__setCookie('SOAPClient', $login_result['data']->status->message);
        $result = $this->makeSoapRequest($action, $params);
        return $result;
    }

    /**
     * @inheritdoc
     */
    protected function checkResponseData($data, $info)
    {
        return $data;
    }

    /**
     * @inheritdoc
     */
    protected function handleCheckResult($check)
    {
        return ['data' => $check];
    }

    /**
     * Получает текущий баланс сервиса
     *
     * @return mixed
     */
    public function getBalance()
    {
        $action = 'getBalanceInfo';
        $result = $this->execute($action, []);
        return $result['data']->data->balance;
    }

    /**
     * @param string $error - тип ошибки
     * @return string
     */
    protected static function getError($error) {
        switch ($error) {
            case 'NO_SUCH_ZONE':
                return self::ERROR_WRONG_ZONE;

            case 'UNSUPPORTED_ZONE':
                return self::ERROR_UNSUPPORTED_ZONE;

            case 'WRONG_DOMAIN_NAME':
            case 'DOMAIN_NAME_TOO_LONG':
            case 'DOMAIN_NAME_TOO_SHORT':
                return self::ERROR_WRONG_DOMAIN_NAME;

            default:
                return self::ERROR_WHILE_DOMAIN_CHECK;

        }
    }

    /**
     * @param string $domain
     * @return string
     */
    public function findDomainContractNumber($domain)
    {
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => $domain],
            'strict' => 1,
            'limit' => -1
        ];
        $response = $this->execute($action, $params);
        $domains = (array)$response['data']->data->domainarray;
        $domain = $domains[0];
        return $domain->{"admin-o"};
    }

    /**
     * @param string $domain
     * @return Registrant
     * @throws UndefinedPropertyException
     */
    public function getRegistrantDataByDomain($domain)
    {
        $domain = static::punycodeEncode($domain);
        $contract_number = $this->findDomainContractNumber($domain);
        return $this->getRegistrantDataByContractNumber($contract_number);
    }

    /**
     * @param string $contract_number
     * @return Registrant
     * @throws UndefinedPropertyException
     */
    public function getRegistrantDataByContractNumber($contract_number)
    {
        $action = 'getInfoAboutDadmin';
        $params = ['nic_hdl' => $contract_number];
        $response = $this->execute($action, $params);
        $response_array = $this->parseStdClass2Array($response['data'])['data'];

        foreach ($response_array as $item => $value) {
            $response_array[$item] = str_replace(["\r\n", "\r", "\n"], ' ', $value);
        }

        $result = [
            'nic_hdl' => $contract_number,
            'is_resident' => $response_array['isresident'] == 1 ? true : false,
            'email' => empty($response_array['email']) ? null : $response_array['email'],
            'phone' => empty($response_array['phone']) ? null : $response_array['phone'],
            'fax' => empty($response_array['fax']) ? null : $response_array['fax'],
            'passport' => empty($response_array['passport']) ? null : $response_array['passport'],
            'birth_date' => empty($response_array['birth_date']) ?
                null : date('Y-m-d', strtotime($response_array['birth_date'])),
            'inn' => empty($response_array['inn']) ? null : $response_array['inn'],
            'ogrn' => empty($response_array['ogrn']) ? null : $response_array['ogrn'],
            'kpp' => empty($response_array['kpp']) ? null : $response_array['kpp'],
            'rs' => empty($response_array['ras_schet']) ? null : $response_array['ras_schet'],
            'ks' => empty($response_array['kor_schet']) ? null : $response_array['kor_schet'],
            'bik' => empty($response_array['bik']) ? null : $response_array['bik'],
            'bank' => empty($response_array['bank']) ? null : $response_array['bank'],
            'legal_address' => empty($response_array['legal_addr']) ? null : $response_array['legal_addr'],
            'postal_address' => empty($response_array['postal_addr']) ? null : $response_array['postal_addr'],
        ];

        if (isset($response_array['is_org']) && $response_array['is_org'] == 1) {
            $result['type'] = Client::TYPE_ORG;
            $result['organization_name'] = empty($response_array['orgname_ru']) ? null : $response_array['orgname_ru'];
            $result['organization_name_eng'] = empty($response_array['orgname_en']) ? null : $response_array['orgname_en'];
        } else {
            $result['type'] = Client::TYPE_PERSON;
            $result['name'] = empty($response_array['fiorus']) ? null : $response_array['fiorus'];
            $result['name_eng'] = empty($response_array['fioeng']) ? null : $response_array['fioeng'];
        }

        return new Registrant($result);
    }

    /**
     * @param string $domain
     * @return mixed
     */
    public function getDomainDetailInfo($domain)
    {
        $domain = static::punycodeEncode($domain);
        $action = 'getDomains';
        $params = [
            'params' => ['domain' => $domain],
            'strict' => 1,
            'limit' => -1
        ];
        $response = $this->execute($action, $params);
        $domains = (array)$response['data']->data->domainarray;
        return $this->parseStdClass2Array($domains[0]);
    }

    /**
     * @param string $domain
     * @return array
     * @throws InternalException
     */
    public function getDomainNs($domain)
    {
        $domain = static::punycodeEncode($domain);
        $response = $this->getDomainDetailInfo($domain);

        if (!isset($response['nserver'])) {
            throw new InternalException(sprintf('Не удалось получить список NS домена %s', $domain));
        }

        return !empty($response['nserver']) ? explode(PHP_EOL, $response['nserver']) : [];
    }

    /**
     * @param $std_object
     * @return mixed
     */
    private function parseStdClass2Array($std_object)
    {
        $json = json_encode($std_object);
        return json_decode($json,true);
    }
}


