<?php
namespace Filanco\RipnEpp;

use DOMDocument;
use DOMElement;
use DOMNode;
use DOMNodeList;
use Filanco\RipnEpp\Exception\Registry\ContentTypeMismatchException;
use Filanco\RipnEpp\Exception\Registry\EmptyResponseException;
use Filanco\RipnEpp\Exception\Registry\InternalErrorException;
use Filanco\RipnEpp\Exception\Registry\ResponseErrorResultException;
use Filanco\RipnEpp\Exception\Registry\ResponseResultNotFoundException;
use Filanco\RipnEpp\Exception\RequestException;
use Filanco\RipnEpp\Exception\UndefinedPropertyException;

/**
 * Class Client
 * @package Filanco\RipnEpp
 */
class Client
{
    const
        RESPONSE_AUTHENTICATION_ERROR_CODE = 2200,
        RESPONSE_AUTHORIZATION_ERROR_CODE = 2201,
        RESPONSE_OBJECT_DOES_NOT_EXISTS_CODE = 2303,
        RESPONSE_SESSION_LIMIT_EXCEEDED_CODE = 2502,
        RESPONSE_INVALID_PARAMETER_VALUE = 2005,
        RESPONSE_REQUEST_LIMIT_EXCEEDED_CODE = 2503;

    /** @var string */
    public $host;

    /** @var string */
    public $port;

    /** @var string */
    public $login;

    /** @var string */
    public $password;

    /** @var string */
    public $registry;

    /** @var string */
    public $version;

    /** @var string */
    public $language;

    /** @var string */
    public $session_key;

    /** @var resource */
    public $transport;

    /** @var string */
    public $certificate;

    /** @var resource */
    public $private_key;

    /** @var string */
    public $tls_version;

    /** @var boolean */
    public $extended_schema = true;

    /** @var boolean */
    protected $connected = false;

    protected static $response_code_exception_map = [
        self::RESPONSE_AUTHENTICATION_ERROR_CODE => 'Filanco\RipnEpp\Exception\Registry\AuthenticationErrorException',
        self::RESPONSE_AUTHORIZATION_ERROR_CODE => 'Filanco\RipnEpp\Exception\Registry\AuthorizationErrorException',
        self::RESPONSE_REQUEST_LIMIT_EXCEEDED_CODE => 'Filanco\RipnEpp\Exception\Registry\RequestLimitExceededException',
        self::RESPONSE_SESSION_LIMIT_EXCEEDED_CODE => 'Filanco\RipnEpp\Exception\Registry\SessionLimitExceededException',
        self::RESPONSE_OBJECT_DOES_NOT_EXISTS_CODE => 'Filanco\RipnEpp\Exception\Registry\ObjectDoesNotExistsException',
        self::RESPONSE_INVALID_PARAMETER_VALUE => 'Filanco\RipnEpp\Exception\Registry\InvalidParameterValueException',
    ];

    /**
     * Client constructor.
     * @param array $params
     * @throws UndefinedPropertyException
     */
    public function __construct(array $params)
    {
        $class_name = get_called_class();
        foreach ($params as $key => $value) {
            if (!property_exists($class_name, $key)) {
                throw new UndefinedPropertyException($class_name, $key);
            }
            $this->{$key} = $value;
        }
        $this->transport = curl_init();
    }

    public function __destruct()
    {
        if ($this->connected) {
            $this->disconnect();
        }
    }

    public function connect()
    {
        $document = new DOMDocument('1.0', 'UTF-8');
        $command = $document->createElement('login');
        $command->appendChild($document->createElement('clID', $this->login));
        $command->appendChild($document->createElement('pw', $this->password));

        $options = $document->createElement('options');
        $options->appendChild($document->createElement('version', $this->version));
        $options->appendChild($document->createElement('lang', $this->language));
        $command->appendChild($options);

        $schemas_node = $document->createElement('svcs');
        $schemas = [
            'http://www.ripn.net/epp/ripn-epp-1.0',
            'http://www.ripn.net/epp/ripn-eppcom-1.0',
            'http://www.ripn.net/epp/ripn-contact-1.0',
            'http://www.ripn.net/epp/ripn-domain-1.0',
            'http://www.ripn.net/epp/ripn-host-1.0',
            'http://www.ripn.net/epp/ripn-registrar-1.0',
        ];

        if ($this->extended_schema) {
            $schemas[] = 'http://www.ripn.net/epp/ripn-domain-1.1';
        }

        foreach ($schemas as $schema) {
            $schemas_node->appendChild($document->createElement('objURI', $schema));
        }
        $command->appendChild($schemas_node);
        $document->appendChild($command);
        $this->send($command);
    }

    public function disconnect()
    {
        $document = new DOMDocument('1.0', 'UTF-8');
        $command = $document->createElement('logout');
        $document->appendChild($command);
        $this->send($command);
    }

    /**
     * @param DOMElement $command_content
     * @return DOMDocument
     */
    public function send(DOMElement $command_content)
    {
        if (!$this->connected) {
            $this->connected = true;
            $this->connect();
        }

        $request = $this->buildRequest($command_content);
        return $this->makeXmlRequest($request);
    }

    public function getRequestId()
    {
        $random_bytes = openssl_random_pseudo_bytes(512);
        return substr(hash('sha256', $random_bytes), 20);
    }

    protected function buildRequest(DOMElement $command_content)
    {
        $document = new DOMDocument('1.0', 'UTF-8');
        $document->formatOutput = true;
        $command_content = $document->importNode($command_content, true);
        $epp = $document->createElement('epp');
        $epp->setAttribute('xmlns', 'http://www.ripn.net/epp/ripn-epp-1.0');
        $epp->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $epp->setAttribute('xsi:schemaLocation', 'http://www.ripn.net/epp/ripn-epp-1.0 ripn-epp-1.0.xsd');
        $command = $document->createElement('command');
        $command->appendChild($command_content);

        $request_id = $this->getRequestId();
        $command->appendChild($document->createElement('clTRID', $request_id));
        $epp->appendChild($command);
        $document->appendChild($epp);
        return $document->saveXML();
    }

    /**
     * @param $payload
     * @return DOMDocument
     * @throws ContentTypeMismatchException
     * @throws EmptyResponseException
     * @throws InternalErrorException
     * @throws RequestException
     */
    protected function makeXmlRequest($payload)
    {
        curl_setopt_array($this->transport, [
            CURLOPT_URL => $this->host . ':' . $this->port,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_HTTPHEADER => ['Content-type: text/xml; charset=UTF-8'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_COOKIEFILE => '/dev/null',
            CURLOPT_COOKIEJAR => '/dev/null',
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSLVERSION => $this->tls_version,
            CURLOPT_SSLCERT => $this->certificate,
            CURLOPT_SSLKEY => $this->private_key,
        ]);

        $data = curl_exec($this->transport);
        $error = curl_error($this->transport);
        $response_code = curl_getinfo($this->transport, CURLINFO_HTTP_CODE);
        $content_length = curl_getinfo($this->transport, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        $content_type = curl_getinfo($this->transport, CURLINFO_CONTENT_TYPE);

        if (!empty($error)) {
            throw new RequestException($error);
        }

        if ($response_code == 500) {
            throw new InternalErrorException();
        }

        if ($content_length == 0) {
            throw new EmptyResponseException();
        }

        if ($content_type != 'text/xml; charset=UTF-8') {
            throw new ContentTypeMismatchException($content_type);
        }

        $response = new DOMDocument();
        $response->loadXML($data);

        $this->checkErrors($response);

        return $response;
    }

    /**
     * @param DOMDocument $data
     * @throws ResponseErrorResultException
     * @throws ResponseResultNotFoundException
     */
    protected function checkErrors(DOMDocument $data)
    {
        $results = $data->getElementsByTagName('result');
        if ($results->length == 0) {
            throw new ResponseResultNotFoundException();
        }
        /** @var DOMElement $result */
        $result = $results->item(0);
        $code = $result->getAttribute('code');

        $is_error_code = (strpos($code, '2') === 0);

        if ($is_error_code) {
            if (in_array($code, array_keys(self::$response_code_exception_map))) {
                throw new self::$response_code_exception_map[$code]($result, $code);
            } else {
                $error_message = $data->getElementsByTagName('msg')->item(0)->textContent;
                throw new ResponseErrorResultException($code, $error_message);
            }
        }
    }
}
