<?php
namespace Filanco\HalkApiClient\Module\Order\Service;

use Filanco\HalkApiClient\Module\Order\Ledger;
use Filanco\HalkApiClient\Module\Order\LedgerRepository;
use Filanco\HalkApiClient\Module\Order\Order;
use Filanco\HalkApiClient\Service\ApiClient;
use GuzzleHttp\Exception\GuzzleException;

class EditOrder
{
    /**
     * @var ApiClient
     */
    private $api_client;

    /**
     * @var LedgerRepository
     */
    private $ledger_repository;

    public function __construct(ApiClient $api_client, LedgerRepository $ledger_repository)
    {
        $this->api_client = $api_client;
        $this->ledger_repository = $ledger_repository;
    }

    /**
     * @param Order $order
     * @param Ledger $ledger
     * @param string $period
     * @param string $payment_description
     * @return Ledger
     * @throws GuzzleException
     */
    public function addProduct(Order $order, Ledger $ledger, $period, $payment_description = null)
    {
        $period_ledgers = $this->ledger_repository->getByParams(['order_id' => $order->id, 'period' => $period]);

        array_push($period_ledgers, $ledger);

        $this->api_client->send('panel.order.update', [
            'order_id' => $order->id,
            'packages' => array_map(function(Ledger $ledger){
                return (array) $ledger;
            }, $period_ledgers),
            'payment_description' => $payment_description,
            'operation_time' => date('Y-m-01 00:00:00', strtotime($period)),
        ]);

        $old_ledger_ids = array_map(function(Ledger $ledger){
            return $ledger->id;
        }, $period_ledgers);

        $new_ledgers = $this->ledger_repository->getByParams(['order_id' => $order->id, 'period' => $period]);

        $created_ledgers = array_filter($new_ledgers, function (Ledger $ledger) use ($old_ledger_ids) {
            return !in_array($ledger->id, $old_ledger_ids);
        });

        return array_shift($created_ledgers);
    }

    /**
     * @param Order $order
     * @param Ledger $ledger
     * @param string $payment_description
     * @throws GuzzleException
     */
    public function deleteProduct(Order $order, Ledger $ledger, $payment_description = null)
    {
        $period = date('Y-m-01', strtotime($ledger->date_start));

        $period_ledgers = $this->ledger_repository->getByParams(['order_id' => $order->id, 'period' => $period]);

        $packages = array_values(array_filter($period_ledgers, function($current_ledger) use ($ledger) {
            return $current_ledger->id !== $ledger->id;
        }));

        $this->api_client->send('panel.order.update', [
            'order_id' => $order->id,
            'packages' => array_map(function(Ledger $ledger){
                return (array) $ledger;
            }, $packages),
            'payment_description' => $payment_description,
            'operation_time' => date('Y-m-01 00:00:00', strtotime($ledger->date_start)),
        ]);
    }

    /**
     * @param Order $order
     * @param $disable_time
     * @throws GuzzleException
     */
    public function disable(Order $order, $disable_time)
    {
        $this->api_client->send('panel.order.disable', [
            'order_id' => $order->id,
            'disable_time' => $disable_time,
        ]);
    }

    /**
     * @param Order $order
     * @param $start_time
     * @param $period
     * @param string $account_type
     * @param bool $credit
     * @throws GuzzleException
     */
    public function enable(Order $order, $start_time, $period, $account_type = 'main', $credit = false)
    {
        $this->api_client->send('panel.order.enable', [
            'order_id' => $order->id,
            'start_time' => $start_time,
            'period' => $period,
            'account_type' => $account_type,
            'credit' => $credit,
        ]);
    }
}