<?php
/**
 * Utils.
 *
 * PHP version 5.3
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Bootstrap
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    SVN: $Id: halk_utils.php 10926 2012-04-05 11:10:16Z i.kolombet $
 * @link       http://halk.filanco.ru
 */
use Halk\Core\Config;
use Halk\Core\ConnectionPool;
use Halk\Core\Exception\AssertException;
use Halk\Core\Exception\CoreException;
use Halk\Core\Logger;
use Halk\Module\Core\Model\ObjectNamespace;

/**
 * prints <pre> var_dump($args) </pre>
 */
function var_dump_pre()
{
    echo '<pre>';
    call_user_func_array('var_dump', func_get_args());
    echo '</pre>';
}

function print_pre()
{
    echo '<pre>';
    call_user_func_array('print_r', func_get_args());
    echo '</pre>';
}

/**
 *
 * @return float
 */
function microtime_float()
{
    list($usec, $sec) = explode(" ", microtime());
    return ((float)$usec + (float)$sec);
}

/**
 * @return mixed
 */
function coalesce()
{

    $arr = array_filter(
        func_get_args(),
        function ($x) {
            return !is_null($x);
        }
    );
    if (empty($arr)) {
        return null;
    }

    return current($arr);
}

/**
 * @param      $haystack
 * @param      $needle
 * @param bool $case
 *
 * @return bool
 */
function starts_with($haystack, $needle, $case = true)
{

    return $case ? (stripos($haystack, $needle) === 0) : (strpos($haystack, $needle) === 0);
}

/**
 * @param      $haystack
 * @param      $needle
 * @param bool $case
 * @return bool
 */
function ends_with($haystack, $needle, $case = true)
{

    $haystack = strrev($haystack);
    $needle = strrev($needle);

    return starts_with($haystack, $needle, $case);
}

/**
 * @param array $array
 * @return string
 */
function dump_sql_params(array $array)
{

    return implode(
        ', ',
        array_map(
            function ($x) {
                return is_null($x) ? 'NULL' : $x;
            },
            $array
        )
    );
}

/**
 * Generates PostgreSQL array.
 *
 * @param      $arr
 * @param bool $normal_format
 * @param bool $inner
 *
 * @return string
 */
function toPgArray($arr, $normal_format = false, $inner = false)
{
    if (is_array($arr) && empty($arr)) {
        return '{}';
    }
    if (!$inner && !is_array($arr)) {
        return $arr;
    }
    $ret = '';
    if (is_array($arr)) {
        foreach ($arr as $v) {
            $ret .= (empty($ret) ? '' : ',') . toPgArray($v, $normal_format, true);
        }
        $ret = $normal_format ? 'ARRAY[' . $ret . ']' : '{' . $ret . '}';
    } elseif (is_null($arr)) {
        $ret = 'NULL';
    } else {
        if ($normal_format) {
            $ret = '\'' . pg_escape_string($arr) . '\'';
        } else {
            $ret = '"' . str_replace('"', '\\"', pg_escape_string($arr)) . '"';
        }
    }
    return $ret;
}

/**
 * Generates PHP array from PgArray (+ multidimensional arrays pgsql)
 *
 * @param string $text
 * @param $output
 * @param bool $limit
 * @param int $offset
 * @return array
 */
function fromPgArrays($text, &$output, $limit = false, $offset = 1)
{
    if (false === $limit) {
        $limit = strlen($text) - 1;
        $output = array();
    }
    if ('{}' != $text) {
        do {
            if ('{' != $text{$offset}) {
                preg_match("/(\\{?\"([^\"\\\\]|\\\\.)*\"|[^,{}]+)+([,}]+)/", $text, $match, 0, $offset);
                $offset += strlen($match[0]);
                $output[] = ('"' != $match[1]{0} ? $match[1] : stripcslashes(substr($match[1], 1, -1)));
                if ('},' == $match[3]) {
                    return $offset;
                }
            } else {
                $offset = fromPgArray($text, $output, $limit, $offset + 1);
            }
        } while ($limit > $offset);
    }
    return $output;
}

/**
 * Generates PHP array from PgArray
 *
 * @param string $text
 * @return array
 */
function fromPgArray($text)
{
    if (preg_match('/^{(.*)}$/', $text, $matches) && !empty($matches[1])) {
        return str_getcsv($matches[1]);
    } else {
        return [];
    }
}

/**
 * @param $arr
 * @param bool $normal_format
 * @param bool $inner
 * @return string
 */
function toPgType($arr, $normal_format = false, $inner = false)
{
    if (is_array($arr) && empty($arr)) {
        return '{}';
    }
    if (!$inner && !is_array($arr)) {
        return $arr;
    }
    $ret = '';
    if (is_array($arr)) {
        foreach ($arr as $v) {
            $ret .= (empty($ret) ? '' : ',') . toPgType($v, $normal_format, true);
        }
        $ret = $normal_format ? '(' . $ret . ')' : '(' . $ret . ')';
    } elseif (is_null($arr)) {
        $ret = 'NULL';
    } else {
        if ($normal_format) {
            $ret = '\'' . pg_escape_string($arr) . '\'';
        } else {
            $ret = '"' . str_replace('"', '\\"', pg_escape_string($arr)) . '"';
        }

    }
    return $ret;
}

/**
 * @param null $input_text
 * @param string $delimiter
 * @param string $text_qualifier
 * @param string $array_text_start
 * @param string $array_text_end
 * @param string $text_escape
 * @return array
 */
function pgArray(
    $input_text = null,
    $delimiter = ',',
    $text_qualifier = '"',
    $array_text_start = '{',
    $array_text_end = '}',
    $text_escape = '\\'
)
{

    if (is_null($input_text)) {
        return array();
    }
    if ($input_text == '{}') {
        return array();
    }
    $level = -1;
    $levels = array();
    $level_i = 0;
    $flag_esc = false;
    $flag_text = false;
    $i = 0;
    $strlen = strlen($input_text);

    while ($i < $strlen) {
        if ($i == 0 && $input_text{$i} !== $array_text_start) {
            $flag_text = true;
        }
        if ($flag_esc) {
            empty($levels[$level]) && $levels[$level] = array();
            !array_key_exists($level_i, $levels[$level]) && $levels[$level][$level_i] = '';
            $levels[$level][$level_i] .= $input_text{$i++};
            $flag_esc = false;
            continue;
        }
        if ($flag_text && !in_array($input_text{$i}, array($text_escape, $text_qualifier))) {
            empty($levels[$level]) && $levels[$level] = array();
            !array_key_exists($level_i, $levels[$level]) && $levels[$level][$level_i] = '';
            $levels[$level][$level_i] .= $input_text{$i++};
            continue;
        }
        if (in_array($input_text{$i}, array($array_text_end, $delimiter))) {
            if ($levels[$level][$level_i] == 'NULL' && $input_text{$i - 1} != $text_qualifier) {
                $levels[$level][$level_i] = null;
            }
        }

        switch ($input_text{$i}) {
            case $text_escape:
                $flag_esc = true;
                break;

            case $array_text_start:
                $level++;
                $level_i = 0;
                $levels[$level] = array();
                break;

            case $array_text_end:
                $level--;
                $level_i = 0;
                empty($levels[$level]) && $levels[$level] = array();
                $levels[$level][] = empty($levels[$level + 1]) ? null : $levels[$level + 1];
                break;

            case $text_qualifier:
                $flag_text = !$flag_text;
                break;

            case $delimiter:
                $level_i++;
                break;

            default:
                empty($levels[$level]) && $levels[$level] = array();
                !array_key_exists($level_i, $levels[$level]) && $levels[$level][$level_i] = '';
                $levels[$level][$level_i] .= $input_text{$i};
                break;
        }
        $i++;
    }
    return $levels[-1][0];
}

/**
 * @param      $name
 * @param null $default
 * @param bool $allow_null
 *
 * @return mixed
 */
function ob($name, $default = null, $allow_null = false)
{

    $tmp = \Halk\Core\RequestData::ob($name, $default, $allow_null);
    if (!is_array($tmp) && !is_null($tmp)) {
        $tmp = trim($tmp);
    }

    return $tmp;
}

function halk_tuple_to_array($string)
{

    $string = substr($string, 1);
    $string = substr($string, 0, -1);
    return explode(',', $string);
}

/**
 * @param $condition
 * @param $err_msg
 * @throws \Halk\Core\Exception\AssertException
 */
function halk_assert($condition, $err_msg)
{

    if (!$condition) {
        throw new AssertException($err_msg);
    }
}

/**
 * @param $str
 * @param $from
 * @param $to
 * @return mixed
 */
function mb_strtr($str, $from, $to)
{
    return str_replace(mb_str_split($from), mb_str_split($to), $str);
}

/**
 * @param $str
 * @return array
 */
function mb_str_split($str)
{
    return preg_split('~~u', $str, null, PREG_SPLIT_NO_EMPTY);;

}

/**
 * переводит в русский текст написаного в анлийской раскладке
 * @param $str
 * @return mixed
 */
function key_eng2rus($str)
{
    $eng = "qwertyuiop[]asdfghjkl;'\\zxcvbnm,.QWERTYUIOP[]ASDFGHJKL;'\\ZXCVBNM,.";
    $rus = "йцукенгшщзхъфывапролджэёячсмитьбюЙЦУКЕНГШЩЗХЪФЫВАПРОЛДЖЭЁЯЧСМИТЬБЮ";
    return mb_strtr($str, $eng, $rus);
}

/**
 * @param string $string
 * @param bool $to_rus
 * @return string
 */
function translit($string, $to_rus = false)
{
    $eng_to_rus = [
        'yu' => 'ю',
        'ya' => 'я',
        'sch' => 'щ',
        'ts' => 'ц',
        'ch' => 'ч',
        'sh' => 'ш',
        'zh' => 'ж',
        'a' => 'а',
        'b' => 'б',
        'c' => 'ц',
        'd' => 'д',
        'e' => 'е',
        'f' => 'ф',
        'g' => 'г',
        'h' => 'х',
        'i' => 'и',
        'j' => 'дж',
        'k' => 'к',
        'l' => 'л',
        'm' => 'м',
        'n' => 'н',
        'o' => 'о',
        'p' => 'п',
        'q' => 'к',
        'r' => 'р',
        's' => 'с',
        't' => 'т',
        'u' => 'у',
        'v' => 'в',
        'w' => 'в',
        'x' => 'кс',
        'y' => 'й',
        'z' => 'з',
        'Yu' => 'Ю',
        'Ya' => 'Я',
        'Sch' => 'Щ',
        'Ts' => 'Ц',
        'Ch' => 'Ч',
        'Sh' => 'Ш',
        'Zh' => 'Ж',
        'A' => 'А',
        'B' => 'Б',
        'C' => 'Ц',
        'D' => 'Д',
        'E' => 'Е',
        'F' => 'Ф',
        'G' => 'Г',
        'H' => 'Х',
        'I' => 'И',
        'J' => 'ДЖ',
        'K' => 'К',
        'L' => 'Л',
        'M' => 'М',
        'N' => 'Н',
        'O' => 'О',
        'P' => 'П',
        'Q' => 'К',
        'R' => 'Р',
        'S' => 'С',
        'T' => 'Т',
        'U' => 'У',
        'V' => 'В',
        'W' => 'В',
        'X' => 'КС',
        'Y' => 'Й',
        'Z' => 'З',
    ];

    $rus_to_eng = [
        'а' => 'a',
        'б' => 'b',
        'в' => 'v',
        'г' => 'g',
        'д' => 'd',
        'е' => 'e',
        'ё' => 'e',
        'ж' => 'zh',
        'з' => 'z',
        'и' => 'i',
        'й' => 'y',
        'к' => 'k',
        'л' => 'l',
        'м' => 'm',
        'н' => 'n',
        'о' => 'o',
        'п' => 'p',
        'р' => 'r',
        'с' => 's',
        'т' => 't',
        'у' => 'u',
        'ф' => 'f',
        'х' => 'h',
        'ц' => 'ts',
        'ч' => 'ch',
        'ш' => 'sh',
        'щ' => 'sch',
        'ъ' => '\'',
        'ы' => 'y',
        'ь' => '',
        'э' => 'e',
        'ю' => 'yu',
        'я' => 'ya',
        'дж' => 'j',
        'А' => 'A',
        'Б' => 'B',
        'В' => 'V',
        'Г' => 'G',
        'Д' => 'D',
        'Е' => 'E',
        'Ж' => 'Zh',
        'З' => 'Z',
        'И' => 'I',
        'Й' => 'Y',
        'К' => 'K',
        'Л' => 'L',
        'М' => 'M',
        'Н' => 'N',
        'О' => 'O',
        'П' => 'P',
        'Р' => 'R',
        'С' => 'S',
        'Т' => 'T',
        'У' => 'U',
        'Ф' => 'F',
        'Х' => 'H',
        'Ц' => 'Ts',
        'Ч' => 'Ch',
        'Ш' => 'Sh',
        'Щ' => 'Sch',
        'Ъ' => '\'',
        'Ы' => 'Y',
        'Ь' => '',
        'Э' => 'E',
        'Ю' => 'Yu',
        'Я' => 'Ya',
        'ДЖ' => 'J'
    ];

    $output = $to_rus ? str_replace(array_keys($eng_to_rus), array_values($eng_to_rus), $string) : str_replace(
        array_keys($rus_to_eng),
        array_values($rus_to_eng),
        $string
    );

    return $output;
}

/**
 * @param string $ip
 * @param string $network
 * @return bool
 */
function cidr_contained_within_or_equals($ip, $network)
{
    if (strstr($ip, ':') || strstr($network, ':')) {
        //$ip = bin2hex(ip2bin($ip));
        //$net = bin2hex(ip2bin($network));
        // FIXME
        return true;
    } else {
        $dbo = ConnectionPool::getInstance()->get();
        $res = $dbo->getOne('SELECT ? <<= ?', array($ip, $network));
        return $res == 1;
    }
}

function ip2bin($ip)
{
    $s = $ipv4 = '';

    if (strpos($ip, ':') !== false) {
        if (!($l = preg_match_all('#(?|::|(?|\d{1,3}\.){3}\d{1,3}|[0-9a-f]{1,4})#i', $ip, $m)) || $l > 8)
            return false;
        foreach ($m[0] as $p) {
            if ($p == '::')
                $s .= str_repeat('0000', 8 - $l + (strpos($m[0][$l - 1], '.') === false));
            else if (strpos($p, '.') !== false)
                $ipv4 = ip2bin($p);
            else
                $s .= str_pad($p, 4, '0', STR_PAD_LEFT);
        }
        if ($ipv4 === false)
            return false;
        return pack('H*', $s) . $ipv4;
    } else if (strpos($ip, '.') !== false) {
        $x = explode('.', $ip);

        if (count($x) != 4)
            return false;
        foreach ($x as $i)
            $s .= chr((int)$i);
        return $s;
    }
    return false;
}


/**
 * @param string $ns_str
 * @throws CoreException
 */
function delete_objects_in_namespace($ns_str)
{

    if (HALK_RUNNING_MODE == HALK_MODE_WWW) {
        throw new CoreException('You cant use function ' . __FUNCTION__ . ' in WWW mode.');
    }

    $dbo = ConnectionPool::getInstance()->get();
    $ns = ObjectNamespace::getByField($ns_str, 'title');
    if (!is_null($ns)) {
        $dbo->query('DELETE FROM objects WHERE namespace_id = ?', array($ns->id));
    }
}

/**
 * @param array $array
 * @return array
 */
function halk_unhash_array(array $array)
{
    $input = $array;
    $result = array();
    foreach ($array as $k => $v) {
        if (!is_numeric($k)) {
            if (is_array($v)) {
                foreach ($v as $j) {
                    $result[] = array($k . '[]', $j);
                }
            } else {
                $result[] = array($k, $v);
            }
        } else {
            return $input;
        }
    }
    return $result;
}

function hstoreToArray($data)
{
    if (empty($data) || $data === 'NULL') {
        return null;
    }

    @eval(sprintf("\$hstore = array(%s);", $data));

    if (!(isset($hstore) and is_array($hstore))) {
        throw new CoreException(sprintf("Could not parse hstore string '%s' to array.", $data));
    }
    return $hstore;
}

function arrayToHstore($data)
{
    if (!is_array($data)) {
        throw new CoreException(sprintf(
            "HStore::toPg takes an associative array as parameter ('%s' given).",
            gettype($data)
        ));
    }

    $insert_values = array();

    foreach ($data as $key => $value) {
        if (is_null($value)) {
            $insert_values[] = sprintf('"%s" => NULL', $key);
        } else {
            $insert_values[] = sprintf('"%s" => "%s"', addcslashes($key, '\"'), addcslashes($value, '\"'));
        }
    }

    return sprintf("%s%s", null, join(', ', $insert_values));
}

/**
 * @param array $array
 * @return array
 */
function halk_hash_array(array $array)
{

    $result = array();
    foreach ($array as $v) {
        if (is_string($v)) {
            $v = halk_tuple_to_array($v);
        }
        list($key, $value) = $v;

        $key = preg_replace('/^(.+)(\[\])$/', '$1', $key, -1, $count);

        if ($count) {
            $result[$key][] = $value;
        } else {
            $result[$key] = $value;
        }
    }
    return $result;
}

function mail_utf8($to, $subject, $message)
{
    $subject = "=?UTF-8?B?" . base64_encode($subject) . "?=";

    $headers =
        "MIME-Version: 1.0" . "\r\n" .
        "Content-type: text/plain; charset=\"UTF-8\"" . "\r\n";

    return mail($to, $subject, $message, $headers);
}

/**
 * @param $to
 * @param $subject
 * @param $message
 * @param null $from
 * @param null $hostname
 * @param string $contentType
 * @param string $server
 * @return null
 * @throws CoreException
 */
function mail_manual_postfix($to, $subject, $message, $from = null, $hostname = null, $contentType = 'text/plain', $server = 'localhost', $message_id = null)
{

    $ini = Config::getInstance();

    if (!$hostname) {
        if ($ini->containsKey('mail_default_from')) {
            $addr = $ini->getValue('mail_default_from');
            $addr = explode('@', $addr);
            $hostname = $addr[1];
        }
    }

    if (is_numeric($server)) {

        /** @var integer $server */
        $server = (int)$server;

        if (!$server) {
            if ($ini->containsKey('white_relay_server')) {
                $server = $ini->getValue('white_relay_server');
            } else {
                throw new CoreException('Не указан сервер для "чистых" рассылок');
            }
        } else {
            if ($ini->containsKey('gray_relay_server')) {
                $server = $ini->getValue('gray_relay_server');
            } else {
                throw new CoreException('Не указан сервер для "грязных" рассылок');
            }
        }

    }

    if (empty($server)) {
        $server = 'localhost';
    }

    $subject = "=?UTF-8?B?" . base64_encode($subject) . "?=";

    if (preg_match("/^text\//", $contentType)) {
        $headers = array(
            "MIME-Version: 1.0" . "\r\n",
            "Content-type: {$contentType}; charset=\"UTF-8\"" . "\r\n"
        );
    } else {
        $headers = array(
            "MIME-Version: 1.0" . "\r\n",
            "Content-type: {$contentType}\r\n"
        );
    }

//    $headers[] = 'Content-Transfer-Encoding: base64' . "\r\n";
//    $message = rtrim(chunk_split(base64_encode($message)));

    $from = is_null($from) ? \Halk\Core\Config::getInstance()->getValue('mail_default_from') : $from;

    $sock = fsockopen($server, 25, $errno, $errstr, 2);
    if (!$sock) {
        throw new CoreException($errstr);
    } else {
        fwrite($sock, "EHLO $hostname\r\n");

        if (!is_array($to)) {
            $to = explode(',', $to);
        }

        foreach ($to as $_) {
            fwrite($sock, "MAIL FROM:$from\r\n");
            fwrite($sock, "RCPT TO:$_\r\n");
            fwrite($sock, "DATA\r\n");
            while ($line = fgets($sock)) {
                if (strpos($line, '354') !== false) {
                    break;
                }
            }

            fwrite($sock, "To: $_\r\n");
            fwrite($sock, "From: $from\r\n");
            fwrite($sock, "Date:". date("r") ."\r\n");
            fwrite($sock, "Subject: $subject\r\n");
            if (!empty($message_id)) {
                fwrite($sock, "Message-ID: <$message_id-mail-queue@halk.filanco.ru>\r\n");
                $dbo = ConnectionPool::getInstance()->get();
                $q =  "select mq.id 
                      from mail_queue mq 
                      where mq.to = ? and mq.object_id = (
                        select mq2.object_id from mail_queue mq2 where mq2.id = ?) order by mq.stamp limit 1";
                $mid = $dbo->getOne($q, [$_, $message_id]);
                if (!empty($mid)) {
                    fwrite($sock, "In-Reply-To: <$mid-mail-queue@halk.filanco.ru>\r\n");
                }
            }
            foreach ($headers as $header) {
                fwrite($sock, $header);
            }
            fwrite($sock, "\r\n");
            fwrite($sock, "$message\r\n");
            fwrite($sock, ".\r\n");

        }

        fwrite($sock, "QUIT\r\n");

        $out = '';
        while (!feof($sock)) {
            $out .= fgets($sock, 128);
        }
        fclose($sock);
        preg_match("/queued as ([a-z0-9]+)/i", $out, $r);

        if (count($r) == 2) {
            return $r[1];
        }
    }
    return null;
}

function set_postgresql_user_session(\Halk\Module\User\Model\User $user)
{
    $dbhost = \Halk\Core\Config::getInstance()->containsKey('db1_host') ? \Halk\Core\Config::getInstance()->getValue(
        'db1_host'
    ) : null;
    if ($dbhost != 'net.hoster.ru') { //там прописано. перезагузить сервер надо, чтобы подхватило изменения. но это проблема
        $dbo = ConnectionPool::getInstance()->get();
        if ($dbo->connection_id = 'main' && $dbo->engine == 'pgsql') {

            if (!($user instanceof \Halk\Module\User\Model\AnonymousUser)) {
                try {
                    $st = $dbo->query("select set_session_user_id(?)", array($user->id));
                    $res = $st->fetch();
                    if (!$res[0]) {
                        Logger::log(
                            'Надо прописать в конфиге постгреса user_session custom_variable_classes = \'user_session\' и user_session.user_id = 0 для корректного логирования',
                            Logger::LOG_WARNING
                        );
                    }
                } catch (Exception $e) {

                    Logger::log(
                        $e->getMessage() . 'Надо прописать в конфиге постгреса user_session custom_variable_classes = \'user_session\' и user_session.user_id = 0 для корректного логирования',
                        Logger::LOG_WARNING
                    );
                }
            }

        }
    }
}

function halk_object_to_array($obj)
{
    return get_object_vars($obj);
}

/**
 * @param $domain_name
 * @return string
 */
function punycode_encode($domain_name)
{

    require_once('Net/IDNA.php');

    try {
        $idna = Net_IDNA::singleton();
        $domain_name = $idna->encode($domain_name);
    } catch (Exception $e) {
        //trigger_error($e->getMessage(), E_USER_NOTICE);
        return $domain_name;
    }

    return $domain_name;
}

/**
 * @param $domain_name
 * @return string
 */
function punycode_decode($domain_name)
{

    require_once('Net/IDNA.php');

    try {
        $idna = Net_IDNA::singleton();
        $decoded_domain_name = $idna->decode($domain_name);
        !$decoded_domain_name && $decoded_domain_name = $domain_name;
        //$domain_name = $idna->decode($domain_name);
    } catch (Exception $e) {
        trigger_error($e->getMessage(), E_USER_NOTICE);
        $decoded_domain_name = $domain_name;
        //return $domain_name;
    }

    //return $domain_name;
    return $decoded_domain_name;
}

/**
 * Получение строки вида читаемое.имя (имя-в.паникоде)
 * @param $string
 * @return string
 */
function punycode_doublecode($string)
{
    $decoded = punycode_decode($string);
    if ($decoded != $string) {
        $readable = $decoded . ' (' . $string . ')';
        return $readable;
    }
    return $string;
}


/**
 * @param $str
 * @param bool $to_array
 * @return mixed
 * @throws \Halk\Core\Exception\InputException
 */
function halk_json_decode($str, $to_array = false)
{
    $error = null;
    $data = json_decode($str, $to_array);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            $error = 'Достигнута максимальная глубина стека';
            break;
        case JSON_ERROR_STATE_MISMATCH:
            $error = 'Некорректные разряды или не совпадение режимов';
            break;
        case JSON_ERROR_CTRL_CHAR:
            $error = 'Некорректный управляющий символ';
            break;
        case JSON_ERROR_SYNTAX:
            $error = 'Синтаксическая ошибка, не корректный JSON';
            break;
        case JSON_ERROR_UTF8:
            $error = 'Некорректные символы UTF-8, возможно неверная кодировка';
            break;
        default:
            $error = 'Неизвестная ошибка';
            break;
    }

    if (!is_null($error)) {
        throw new \Halk\Core\Exception\InputException('halk_json_decode_error: ' . $error . ' return_data: ' . $str);
    }
    return $data;
}

/**
 * Дебаг переменных в виде var_dump в файл tmp/_debug.log
 *
 * @return string
 */
function _d()
{
    $args = func_get_args();
    $log_file = rtrim(HALK_DOCUMENT_ROOT, '/') . "/../../tmp/_debug.log";
    if (!sizeof($args)) {
        file_put_contents($log_file, str_repeat('-=', 30) . PHP_EOL . PHP_EOL, FILE_APPEND);
        return;
    }
    $out = '';
    $i = 0;
    foreach ($args as $value) {
        if (!$i) {
            if (is_string($value)) {
                $out .= $value . PHP_EOL;
                file_put_contents($log_file, $value . PHP_EOL, FILE_APPEND);
                continue;
            }
        }
        $i = 1;
        ob_start();
        var_dump($value);
        $text = ob_get_clean();
        $text = preg_replace('~(\]=>\s*)~', '] => ', $text);
        $out .= $text;
        file_put_contents($log_file, $text, FILE_APPEND);
    }
    file_put_contents($log_file, PHP_EOL, FILE_APPEND);
    return $out;
}

/**
 * Shorthand для присвоения с проверкой значения
 * $a = isset($var) ? $var : null;
 * @param $var
 * @param null $default
 * @return null
 */
function ifset($var = null, $default = null)
{
    if (isset($var)) {
        return $var;
    } else {
        return $default;
    }
}

/**
 * Функция генерации @property-блоков из $fields обеъкта для подсказок в IDE
 * @param $model
 * @throws CoreException
 */
function generateFieldDoc($model)
{
    if (!$r = new ReflectionClass($model)) {
        throw new CoreException('Не удалось рефлексировать класс ' . $model);
    }
    if (!$filename = $r->getFileName()) {
        throw new CoreException('Не удалось получить имя файла');
    }
    if (!$code = file_get_contents($filename)) {
        throw new CoreException('Не удалось прочесть файл ' . $filename);
    }

    if (!isset($model::$fields)) {
        throw new CoreException('Нет полей для генерации');
    }
    $doc = "/**\n";
    if (is_array($model::$pk_field)) {
        foreach ($model::$pk_field as $i => $pk_field) {
            if (isset($model::$pk_type[$i])) {
                $pk_type = $model::$pk_type[$i];
            } else {
                $pk_type = 'integer';
            }
            $doc .= sprintf(" * @property %s $%s\n", $pk_type, $pk_field);
        }
    } else {
        $doc .= sprintf(" * @property %s $%s\n", $model::$pk_type, $model::$pk_field);
    }
    foreach ($model::$fields as $title => $field) {
        $t = str_replace('[]', '', $field['type'], $count);

        switch (preg_replace("/\(.*\)/", "", $t)) {
            case 'text':
            case 'string':
            case 'varchar':
            case 'interval':
            case 'inet':
            case 'cidr':
            case 'bit':
            case 'bytea':
            case 'character':
            case 'char':
            case 'enum':
            case 'date':
            case 'datetime':
            case 'timestamp':
            case 'timestamp with time zone':
                $type = 'string';
                break;

            case 'serial':
            case 'integer':
            case 'bigint':
            case 'smallint':
                $type = 'integer';
                break;

            case 'numeric':
            case 'float':
            case 'double':
            case 'double precision':
                $type = 'float';
                break;

            case 'boolean':
                $type = 'boolean';
                break;

            case 'tuple':
            case 'hstore':
                $type = 'array';
                break;

            default:
                $type = $field['type'];
                break;
        }
        $count && $type .= '[]';
        $comment = isset($field['comment']) ? ' ' . $field['comment'] : '';
        $doc .= sprintf(" * @property %s $%s%s\n", $type, $title, $comment);
    }
    $doc .= " */";
    if (preg_match('#/\*\*(((\r\n)|(\n))\s\*\s@property.+)+((\r\n)|(\n))\s\*/#', $code)) {
        $code = preg_replace('#/\*\*(((\r\n)|(\n))\s\*\s@property.+)+((\r\n)|(\n))\s\*/#', $doc, $code);
    } else {
        $code = preg_replace('/((\r\n)|(\n))class/', $doc . PHP_EOL . "class", $code);
    }
    file_put_contents($filename, $code);
}

/**
 * Shorthand для добавления в профайл чекпоинта
 * @param null $label имя метки
 * @param null $compareTo замерять относительно последней отметки с таким именем
 * @return bool
 */
function pf($label = null, $compareTo = null)
{
    return \Halk\Core\Profiler::check($label, $compareTo);
}

/**
 * Shorthand для вывода данных профайлера в лог
 * @return bool|null
 */
function pf_log()
{
    return \Halk\Core\Profiler::log();
}

function IPv4To6($Ip, $began = '')
{
    $IPv6 = (strpos($Ip, '::') === 0);
    $IPv4 = (strpos($Ip, '.') > 0);

    if (!$IPv4 && !$IPv6) return false;
    if ($IPv6 && $IPv4) $Ip = substr($Ip, strrpos($Ip, ':') + 1); // Strip IPv4 Compatibility notation
    elseif (!$IPv4) return $Ip; // Seems to be IPv6 already?
    $Ip = array_pad(explode('.', $Ip), 4, 0);
    if (count($Ip) > 4) return false;
    for ($i = 0; $i < 4; $i++) if ($Ip[$i] > 255) return false;
    $Part7 = base_convert(($Ip[0] * 256) + $Ip[1], 10, 16);
    $Part8 = base_convert(($Ip[2] * 256) + $Ip[3], 10, 16);
    return $began . $Part7 . ':' . $Part8;
}

/**
 * Переводит секунды в uptime
 * @return string
 */
function second2uptime($inputSeconds, $short = false)
{
    $now = $inputSeconds;
    $days = intval($now / (60 * 60 * 24 * 100));
    $remainder = $now % (60 * 60 * 24 * 100);
    $hours = intval($remainder / (60 * 60 * 100));
    $remainder = $remainder % (60 * 60 * 100);
    $minutes = intval($remainder / (60 * 100));
    if ($days == 1) {
        $writeDays = "day";
    } else {
        $writeDays = "days";
    }
    if ($hours == 1) {
        $writeHours = "hour";
    } else {
        $writeHours = "hours";
    }
    if ($minutes == 1) {
        $writeMins = "minute";
    } else {
        $writeMins = "minutes";
    }
    $rc = [];
    if ($days > 0) {
        $rc[] = "$days $writeDays";
    }
    if ($hours > 0) {
        $rc[] = "$hours $writeHours";
    }
    if ($minutes > 0) {
        $rc[] = "$minutes $writeMins";
    }
    if ($short) {
        return sprintf("%02d:%02d:%02d", $days, $hours, $minutes);
    } else {
        return implode(', ', $rc);
    }
}

function hex_dump($data, $newline = "\n")
{
    static $from = '';
    static $to = '';

    static $width = 16; # number of bytes per line

    static $pad = '.'; # padding for non-visible characters

    if ($from === '') {
        for ($i = 0; $i <= 0xFF; $i++) {
            $from .= chr($i);
            $to .= ($i >= 0x20 && $i <= 0x7E) ? chr($i) : $pad;
        }
    }

    $hex = str_split(bin2hex($data), $width * 2);
    $chars = str_split(strtr($data, $from, $to), $width);

    $offset = 0;
    foreach ($hex as $i => $line) {
        echo sprintf('%6X', $offset) . ' : ' . implode(' ', str_split($line, 2)) . ' [' . $chars[$i] . ']' . $newline;
        $offset += $width;
    }
}

function halk_tempnam($directory, $prefix, $sufix)
{
    $f = tempnam($directory, $prefix);
    if (!empty($sufix)) {
        unlink($f);
        touch("{$f}{$sufix}");
        $f .= $sufix;
    }
    return $f;
}

function nds($amount, $rate)
{
    if ($rate && $rate < 1) {
        $rate = $rate * 100;
    }
    return round($amount * $rate / (100 + $rate), 2);
}

function toCamelCase($name, $prefix = '')
{
    return $prefix . str_replace(' ', '', ucwords(str_replace('_', ' ', $name)));
}


/**
 * Конвертируем "машинную" дату в стандартную
 * @param $date
 * @return string
 */
function toDefaultDate($date)
{
    return substr($date, 8, 2) . '.' . substr($date, 5, 2) . '.' . substr($date, 0, 4);
}

/**
 * выводит переменные без верстки - plain text
 */
function dd()
{
    ini_set('html_errors', 0);
    echo '<pre>';
    foreach (func_get_args() as $arg) {
        var_dump($arg);
        echo '<br>';
    }
    echo '</pre>';
    exit;
}

/**
 * выводит переменные с html версткой
 */
function ddh()
{
    ini_set('html_errors', 1);
    echo '<pre>';
    foreach (func_get_args() as $arg) {
        var_dump($arg);
        echo '<br>';
    }
    echo '</pre>';
    exit;
}

/**
 * Пишем переменную в файл
 */
function df($var, $append = false, $file = 'dump.txt')
{
    if ($append) {
        file_put_contents($file, print_r($var, true) . PHP_EOL, FILE_APPEND);
    } else {
        file_put_contents($file, print_r($var, true));
    }
}

/**
 * Time from start - Время с момента запуска скрипта
 * @param bool $die
 */
function tfs($die = true)
{
    echo '<pre>';
    echo microtime_float() - HALK_START_TIME;
    echo '</pre>';

    if ($die) {
        exit;
    }
}

/**
 * id объектов массива в индекс массива
 * @param array $array
 * @param string $key
 * @return array | bool Вернет false, если некорректный массив
 */
function array2map(array $array, $key = 'id')
{
    $result = [];

    foreach ($array as $item) {
        if (is_array($item) && array_key_exists($key, $item)) {
            $result[$item[$key]] = $item;
        } elseif (is_object($item)) {
            $result[$item->$key] = $item;
        } else {
            return false;
        }
    }

    return $result;
}

/**
 * Окончания в зависимости от числа
 * @param $count Кол-во штук
 * @param $form1 Одна штука
 * @param $form2 Две штуки
 * @param $form3 Много штук
 * @return mixed
 */
function num_suffix($count, $form1, $form2, $form3)
{
    $count = abs($count) % 100;
    $lcount = $count % 10;
    if ($count >= 11 && $count <= 19) {
        return ($form3);
    }
    if ($lcount >= 2 && $lcount <= 4) {
        return ($form2);
    }
    if ($lcount == 1) {
        return ($form1);
    }
    return $form3;
}
