<?php
namespace Halk\Core;

/**
 * INI class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */

class Config extends Registry
{

    /**
     * Array of key => value data from INI file
     * @var array
     */
    public $data = array();

    /**
     * Custom getInstance() so we can have different INI files simultenously.
     *
     * @param null|string $file path to file (default=HALK_DEFAULT_CONF)
     * @param bool $process_sections (how to parse ini file)
     * @return self
     */
    public static function getInstance($file = HALK_DEFAULT_CONF, $process_sections = false) {

        static $instances = array();

        if(!isset($instances[$file])) {

            $instances[$file] = new self($file, $process_sections);
        }

        return $instances[$file];
    }//end getInstance()

    /**
     * Constructor.
     *
     * @param string $file path to file (default=HALK_DEFAULT_CONF)
     * @param bool $process_sections (how to parse ini file)
     * @throws \Halk\Core\Exception\INIException if cannot load file
     */
    public function __construct($file = HALK_DEFAULT_CONF, $process_sections = false) {

        $this->data = parse_ini_file($file, $process_sections);
        
        // -------------------------------------------------------------------
        if($this->data === false) {
            throw new \Halk\Core\Exception\INIException('Failed to load config. File is: ' . $file);
        }
    }//end __construct()

    /**
     * Get all key/value pairs.
     *
     * @return array
     */
    public function getAll() {

        return $this->data;
    }//end getAll()

    /**
     * Get value by key.
     *
     * @param string $key key
     *
     * @return string
     * @throws \Halk\Core\Exception\INIException if key is missing
     */
    public function getValue($key) {

        if(!isset($this->data[$key])) {
            throw new \Halk\Core\Exception\INIException('Missing key: ' . $key);
        }

        return $this->data[$key];
    }//end getValue()

    /**
     * Check if config contains key.
     *
     * @param string $key key
     *
     * @return boolean true if key exists
     */
    public function containsKey($key) {

        return isset($this->data[$key]);
    }//end containsKey()
}//end class
