<?php
namespace Halk\Core;

use Doctrine\DBAL\DriverManager;
use PDO;
use PDOException;

/**
 * Database framework.
 * Based on PDO.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
class DBO extends PoolElement {

    const PG_UNIQUE_VIOLATION = '23505';
    const MYSQL_UNIQUE_VIOLATION = '23000';

    const PG_FOREIGN_KEY_VIOLATION = '23503';
    const MYSQL_FOREIGN_KEY_VIOLATION = '23000'; // shit, mysql uses same error codes for different errors

    /**
     * PDO instance
     * @var PDO
     */
    protected $pdo;

    /**
     * DSN connection string
     * @var string
     */
    protected $dsn;

    /**
     * Type of the engine. Current supported engines: mysql, postgresql
     * @var string
     */
    public $engine;

    /**
     * Connection id, which is used to save/load object
     * @var string
     */
    public $connection_id;

    /**
     * @var array
     */
    public $debug = array('query_count' => 0, 'sql_time' => 0, 'fetch_time' => 0, 'queries' => array());

    /**
     * Constructor.
     *
     * @param string $db_name          database name
     * @param string $db_host          database host
     * @param string $db_user          database username
     * @param string $db_pass          database password
     * @param string $db_engine        database engine (mysql or postgresql)
     * @param string $db_connection_id unique id of the connection
     * @param integer $db_port unique id of the connection
     */
    public function __construct($db_name, $db_host, $db_user, $db_pass, $db_engine, $db_connection_id, $db_port = null) {

        $this->engine = $db_engine;
        $this->connection_id = $db_connection_id;

        if ($db_port) {
            $this->dsn = sprintf("%s:dbname=%s;host=%s;port=%d;", $this->engine, $db_name, $db_host, $db_port);
        }else {
            $this->dsn = sprintf("%s:dbname=%s;host=%s;", $this->engine, $db_name, $db_host);
        }

        $this->pdo = new PDO($this->dsn, $db_user, $db_pass);
        $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        try {
            $this->pdo->setAttribute(PDO::ATTR_TIMEOUT, 3);
        } catch (PDOException $e) {

        }

        if($this->engine == 'pgsql') {
            $this->pdo->query('SET DATESTYLE = ISO, European');
        } elseif($this->engine == 'mysql') {
            // @todo: UTF8 should not be magic constant
            $this->pdo->query('SET NAMES UTF8');
        }
    }//end __construct()

    /**
     * Start transaction.
     *
     * @return boolean true if OK
     *
     * @throws PDOException if transaction aren't supported
     */
    public function begin() {

        return $this->pdo->beginTransaction();
    }//end begin()

    /**
     * @return PDO
     */
    public function getPDO() {
        return $this->pdo;
    }

    /**
     * Commit transaction.
     *
     * @return boolean true if OK
     *
     * @throws PDOException if not OK
     */
    public function commit() {

        return $this->pdo->commit();
    }//end commit()

    /**
     * Rollback transaction.
     *
     * @return boolean if OK
     *
     * @throws PDOException if not OK
     */
    public function rollback() {

        return $this->pdo->rollBack();
    }//end rollback()

    /**
     * @return bool
     */
    public function inTransaction() {
        return $this->pdo->inTransaction();
    }

    /**
     * Execute query with params.
     *
     * @param string $Q      sql query string
     * @param array  $params [optional] sql params
     *
     * @throws \Halk\Core\Exception\DatabaseUniqueException
     * @throws \Halk\Core\Exception\DatabaseReferenceException
     * @throws \Halk\Core\Exception\DatabaseException
     * @return \PDOStatement if OK
     */
    public function query($Q, array $params = array()) {
        $_s = microtime(true);
        $params = !empty($params)&&is_array($params)? $params : array();
        $sth = $this->pdo->prepare($Q);

        foreach($params as $k=>$v) {
            $k = is_int($k)? $k+1 : $k;
            switch(true) {
                case is_bool($v):
                    $sth->bindValue($k, $v, PDO::PARAM_BOOL);
                    break;
                case is_integer($v):
                    $sth->bindValue($k, $v, PDO::PARAM_INT);
                    break;
                case is_null($v):
                    $sth->bindValue($k, $v, PDO::PARAM_NULL);
                    break;
                default:
                    $sth->bindValue($k, $v, PDO::PARAM_STR);
                    break;
            }
        }

        try {
            if (ini_get('pinba.enabled')) {
                $query_timer = pinba_timer_start([
                    'group' => 'sql_query',
                    'connection_id' => $this->connection_id,
                    'query' => $Q
                ]);
            }

            try {
                $sth->execute();
            } finally {
                if (ini_get('pinba.enabled')) {
                    isset($query_timer) && pinba_timer_stop($query_timer);
                }
            }
        } catch (PDOException $e) {
            switch ((string)$e->getCode()) {
                case self::PG_UNIQUE_VIOLATION:
                case self::MYSQL_UNIQUE_VIOLATION:
                    throw new \Halk\Core\Exception\DatabaseUniqueException($e->getMessage(), 0, $e);
                    break;
                case self::PG_FOREIGN_KEY_VIOLATION:
                case self::MYSQL_FOREIGN_KEY_VIOLATION:
                    throw new \Halk\Core\Exception\DatabaseReferenceException($e->getMessage(), 0, $e);
                    break;
                default:
                    throw new \Halk\Core\Exception\DatabaseException($e->getMessage(), 0, $e);

            }
        }

        if (Config::getInstance()->containsKey('sql_log')) {
            try {
                $sql_log = Config::getInstance()->getValue('sql_log');
                if (!empty($sql_log)) {
                    $_e = (microtime(true) - $_s);
                    $fp = fopen($sql_log, "a+");
                    if ($fp) {
                        fwrite($fp, var_export([
                                'queries' => array($sth->queryString, $params, $_e)
                            ], true));
                        fclose($fp);
                    }
                }
            } catch (\Exception $e) {

            }
        }
        return $sth;
    }//end query()

    /**
     * Execute query and return value of first column of first row.
     *
     * @param string $q      sql query string
     * @param array  $params [optional] sql params
     *
     * @return mixed or null
     */
    public function getOne($q, array $params = array()) {

        $sth = $this->query($q, $params);
        $row = $sth->fetch(PDO::FETCH_NUM);

        if(is_array($row)) {
            return $row[0];
        }

        return null;
    }//end getOne()

    /**
     * Execute query and return result as array.
     *
     * @param string  $q      sql query string
     * @param array   $params [optional] sql params
     * @param integer $mode   [optional] PDO fetch mode (default=PDO::FETCH_ASSOC)
     *
     * @return array
     */
    public function fetchAll($q, array $params = array(), $mode = PDO::FETCH_ASSOC) {

        $sth = $this->query($q, $params);

        return $sth ? $sth->fetchAll($mode) : array();
    }//end fetchAll()

    /**
     * Execute query and return array of first column values.
     *
     * @param string $q      sql query string
     * @param array  $params [optional] sql params
     *
     * @return array
     *
     * @todo ugly code
     */
    public function getAll($q, array $params = array()) {

        $rows = $this->fetchAll($q, $params, PDO::FETCH_BOTH);

        $result_array = array();
        foreach($rows as $row) {
            $result_array[] = $row[0];
        }

        return $result_array;
    }//end getAll()

    /**
     * Execute query anr return result as a single row.
     *
     * @param string  $q      sql query string
     * @param array   $params [optional] sql params
     * @param integer $mode   [optional] PDO fetch mode (default=PDO::FETCH_ASSOC)
     *
     * @return array
     */
    public function getRow($q, array $params = array(), $mode = PDO::FETCH_ASSOC) {
        $sth = $this->query($q, $params);
        return $sth ? $sth->fetch($mode) : array();
    }//end getRow()

    /**
     * Get last insert PK id.
     *
     * @param string $name [optional] name of the sequence object (default=NULL)
     *
     * @return string
     */
    public function getLastInsertId($name = null) {

        return $this->pdo->lastInsertId($name);
    }//end getLastInsertId()

    /**
     * Exec query
     * @param string $q sql query string
     * @return int affected rows
     * @throws \Halk\Core\Exception\DatabaseException
     * @throws \Halk\Core\Exception\DatabaseReferenceException
     * @throws \Halk\Core\Exception\DatabaseUniqueException
     */
    public function exec($q) {
        return $this->query($q)->rowCount();
    }

    public function getDBAL()
    {
        return DriverManager::getConnection([
            'pdo' => $this->pdo,
            'wrapperClass' => DBALConnectionWrapper::class
        ]);
    }
}
