<?php

namespace Halk\Core;

use Halk\Core\Exception\CoreException;

/**
 * SQL builder class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
class SQLBuilder extends Registry
{

    /**
     *
     * @var string
     */
    public $engine;

    /**
     *
     * @var string
     */
    private $_query_type; // one of INSERT|UPDATE|DELETE

    /**
     *
     * @var string
     */
    private $_sql;

    /**
     *
     * @var array
     */
    private $_params = array();

    /**
     * Constructor.
     *
     * @param string $engine database backend engine
     */
    public function __construct($engine)
    {

        $this->engine = $engine;
    }//end __construct()

    /**
     * Run query.
     *
     * @return \PDOStatement
     */
    public function query()
    {

        return DBO::getInstance()->query($this->_sql, $this->_params);
    }//end query()

    /**
     * Escape column name.
     *
     * @param string $column column name
     *
     * @param string $engine
     *
     * @return string
     */
    static function escapeColumnName($column, $engine)
    {

        $result = $column;

        switch ($engine) {

            case 'pgsql':
                //Обработка колонки для hstore
                if (strpos($column, '->')) {
                    $column = str_replace([' ', '\'', '"'], '', $column);
                    $parts = explode('->', $column);
                    if (count($parts) != 2) {
                        throw new \Halk\Core\Exception\UnexpectedSituation('Неверный hstore-синтаксис ' . $column);
                    }
                    $result = sprintf('"%s" -> \'%s\'', $parts[0], $parts[1]);

                } else {
                    $result = sprintf('"%s"', $column);
                }
                break;

            case 'mysql':
                $result = sprintf('`%s`', $column);
                break;
        }

        return $result;
    }//end escapeColumnName()

    /**
     * Escape column value.
     *
     * @param string $type database type
     * @param string $value value
     *
     * @throws CoreException
     * @return string
     */
    static function escapeColumnValue($type, $value, $engine = 'pgsql')
    {
        if (is_null($value)) {
            return $value;
        }

        $type = preg_replace("/\\(.*\\)/", "", $type);

        // array check
        $array = false;
        if (Utils::ends_with($type, '[]')) {
            $type = str_replace('[]', '', $type);
            $array = true;
        }

        switch ($type) {
            case 'text':
            case 'string':
            case 'varchar':
            case 'interval':
            case 'inet':
            case 'cidr':
            case 'bit':
            case 'bytea':
            case 'character':
                return (is_array($value) OR $array) ? Utils::toPgArray($value) : strval($value);
                break;

            case 'serial':
            case 'integer':
            case 'int2':
            case 'int4':
            case 'int8':
            case 'bigint':
            case 'smallint':
                return $array ? Utils::toPgArray($value) : intval($value);
                break;

            case 'numeric':
            case 'float':
            case 'double precision':
                return $array ? Utils::toPgArray($value) : floatval($value);
                break;

            case 'boolean':
                if ($array) {
                    return Utils::toPgArray($value);
                }
                if (is_bool($value)) {
                    return $value;
                } elseif (empty($value)) {
                    return false;
                } else {
                    return ($value != 'f');
                }
                break;

            case 'date':
            case 'timestamp':
                if ($engine === 'mysql') {
                    if (is_int($value)) {
                        return date('Y-m-d H:i:s', $value);
                    } elseif (is_string($value)) {
                        return date('Y-m-d H:i:s', strtotime($value));
                    }
                }
            case 'timestamp with time zone':
                if ($array) {
                    return Utils::toPgArray($value);
                }
                if ($value === null || $value === false) {
                    return $value;
                } elseif (is_int($value)) {
                    return date('r', $value);
                } elseif (is_string($value)) {
                    return date('r', strtotime($value));
                } else {
                    return $value;
                }
                break;

            case 'tuple':
                foreach ($value as $k => $v) {
                    $value[$k] = Utils::toPgType($v);
                }
                if ($array) {
                    $value = Utils::toPgArray($value);
                }

                return $value;
                break;

            case 'hstore' :
                if ($array) {
                    return Utils::arrayToHstore($value);
                } else {
                    throw new CoreException('$value not hstrore');
                }

                break;

            case 'array' :
                !is_array($value) && $value = [];

                return serialize($value);
                break;

            case 'json' :
                return json_encode($value);
                break;

            default:
                throw new CoreException(sprintf("[%s] Unknown type: '$type'", get_called_class()));
                break;
        }
    }//end escapeColumnValue()

    /**
     * Insert into table.
     *
     * @param string $table table name
     * @param array $columns column list
     * @param array $types type list
     * @param array $values value list
     *
     * @return void
     */
    public function insert($table, array $columns, array $types, array $values)
    {

        $this->_sql = sprintf("INSERT INTO %s (%s) VALUES (%s)",
            $table,
            implode(',', array_map(array($this, 'escapeColumnName'), $columns)),
            implode(',', array_fill(0, count($values) - 1, '?'))
        );

        //$this->_params = array_merge($this->_params, array_map(function));
    }

    public function update()
    {

    }

    public function delete()
    {

    }

    public function where()
    {

    }

    public function order()
    {

    }

    /**
     * Возвращает 0 если нет
     * Возвращает 1 если по колонке можно искать через LIKE
     * Возвращает 2 если по колонке можно искать только через =
     *
     * @param $type
     * @return int
     */
    public static function isSearchableType($type)
    {

        $type = preg_replace("/\\(.*\\)/", "", $type);

        switch ($type) {
            case 'date':
            case 'timestamp':
            case 'timestamp with time zone':
            case 'integer':
            case 'bigint':
                return 2;

            case 'text':
            case 'string':
            case 'varchar':
            case 'numeric':
            case 'character':
                return 1;
            case 'inet':
            case 'cidr':
                return 3;
            default:
                return 0;
        }
    }


}
