<?php
namespace Halk\Core;

/**
 * Basic controller class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
abstract class Controller extends Registry implements ControllerInterface {

    const BEFORE_RENDER = '\Halk\Core\Controller//BEFORE_RENDER';
    const AFTER_RENDER = '\Halk\Core\Controller//AFTER_RENDER';

    /**
     * Request to the website
     */
    const REQUEST_TYPE_WWW = 'WEB';

    /**
     * Request to the API
     */
    const REQUEST_TYPE_API = 'API';

    /**
     * 0 or 1
     * @var integer
     */
    public $request_type = self::REQUEST_TYPE_WWW;

    /**
     *
     * @var RequestData
     */
    public $request;

    /**
     * Contains template data, as key=>value map
     * @var array
     */
    protected $view = array();

    /**
     * Contains errors
     * @var array
     */
    protected $errors = array();

    /**
     * @var array
     */
    protected $warnings = array();

    /**
     * Smarty object
     * @var \Halk\Core\Smarty
     */
    protected $smarty;

    /**
     * Title of the page
     * @var string
     */
    protected $page_title;

    /**
     * Whether request is asynchronous or not (AJAX)
     * @var boolean
     */
    protected $is_ajax = false;

    /**
     * Requested method
     * @var string
     */
    public $requested_method = null;

    /**
     * @var null
     */
    protected $main_tpl = null;

    /**
     * @var null
     */
    protected $tpl = null;

    protected $template_dir = null;

    /**
     * @var null
     */
    static $current_controller = null;

    protected $base_url = null;

	protected $http_login = '';

	protected $http_pass = '';

	protected $http_realm = 'Auth required';

	protected $http_auth_type = 'Basic'; // Basic or Digest

    public static $use_default_action = false;

    /**
     * Show error page
     *
     * @param array $params controller params
     *
     * @return void
     */
    public function action_error(array $params)
    {
        $this->error($params['message'], $params['code']);
    }//end action_error()

    private function http_digest_parse($txt) {
            $needed_parts = array('nonce'=>1, 'nc'=>1, 'cnonce'=>1, 'qop'=>1, 'username'=>1, 'uri'=>1, 'response'=>1);
            $data = array();
            $keys = implode('|', array_keys($needed_parts));
            preg_match_all('@(' . $keys . ')=(?:([\'"])([^\2]+?)\2|([^\s,]+))@', $txt, $matches, PREG_SET_ORDER);
            foreach ($matches as $m) {
                $data[$m[1]] = $m[3] ? $m[3] : $m[4];
                unset($needed_parts[$m[1]]);
            }
            return (count($needed_parts) != 0 ? false : $data);
    }

    private function go_http_auth() {
            if($this->http_auth_type == 'Basic') {
                    header('WWW-Authenticate: Basic realm="'.$this->http_realm.'"');
                    header('HTTP/1.0 401 Unauthorized');
                    exit;
            } else {
                    header('WWW-Authenticate: Digest realm="'.$realm.'",qop="auth",nonce="'.uniqid().'",opaque="'.md5($realm).'"');
                    header('HTTP/1.1 401 Unauthorized');
                    exit;
            }
    }

    protected function check_http_auth() {
            if($this->http_auth_type == 'Basic') {
                if( (!isset($_SERVER['PHP_AUTH_USER'])) || ($_SERVER['PHP_AUTH_USER'] != $this->http_login)
                    || ($_SERVER['PHP_AUTH_PW'] != $this->http_pass )  ) {
                    $this->go_http_auth();
                }
            } else {
                if ( (empty($_SERVER['PHP_AUTH_DIGEST'])) || (!($data = $this->http_digest_parse($_SERVER['PHP_AUTH_DIGEST']))) ||
                        ($data['username'] != $this->http_login) )  {
                            $this->go_http_auth();
                } else {
                    $A1 = md5($data['username'] . ':' . $this->realm . ':' . $this->http_pass);
                    $A2 = md5($_SERVER['REQUEST_METHOD'].':'.$data['uri']);
                    $valid_response = md5($A1.':'.$data['nonce'].':'.$data['nc'].':'.$data['cnonce'].':'.$data['qop'].':'.$A2);
                    if ($data['response'] != $valid_response)
                            $this->go_http_auth();

                }
            }
    }


    /**
     * @static
     *
     * @param null $class_name
     *
     * @return static
     */
    public static function getInstance($class_name = null) {

        $result = parent::getInstance($class_name);
        static::$current_controller = $result;
        return $result;
    }

    /**
     * Constructor
     */
    public function __construct() {

        $this->request = RequestData::getInstance();
        $this->base_url = $_SERVER['REQUEST_URI'];

        if ((isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') ||
            isset($_REQUEST['json']) ||
            isset($_REQUEST['extjson']) ||
            isset($_REQUEST['groupjson'])) {

            $this->is_ajax = true;

        } else {

            $this->smarty = Smarty::getInstance();
        }
    }//end __construct()

    /**
     * @param $tpl
     */
    public function setTemplate($tpl) {
        $this->tpl = $tpl;
    }

    public function setBaseURL($base_url) {
        $this->base_url = $base_url;
    }

    public function getBaseURL() {
        return $this->base_url;
    }

    /**
     * @return null
     */
    public function getTemplate() {
        return $this->tpl;
    }

    /**
     * @param $tpl
     */
    public function setMainTemplate($tpl) {
        $this->main_tpl = $tpl;
    }

    /**
     * @return null
     */
    public function getMainTemplate() {
        return $this->main_tpl;
    }

    public function setTemplateDir($dir) {
        $this->template_dir = $dir;
    }

    public function getTemplateDir() {
        return $this->template_dir;
    }

    /**
     * Assign value to the template variable (equivalent to Smarty::assign).
     *
     * @param string $key variable key
     * @param mixed $value variable value
     *
     * @return void
     */
    public function assign($key, $value = null) {

        if(is_array($key)){
            $this->assignArray($key);
        } else {
            $this->view[$key] = $value;
        }
    }

    public function get($key) {
        return $this->view[$key];
    }

    public function assignArray(array $array) {

        foreach($array as $key => $value) {

            $this->assign($key, $value);
        }
    }

    /**
     * Report an error.
     *
     * @param string $text error message
     * @param int $code error code
     * @return void
     */
    public function error($text, $code = 404) {
        $this->errors = [
            'message' => $text,
            'code' => $code
        ];

    }//end error()

    /**
     * @param $text
     */
    public function warning($text) {

        $this->warnings[] = $text;
    }

    /**
     * Render template file, assign errors and data to the template variables.
     * Example: render('main', 'account/index')
     *
     * @internal param string $tpl template name without .tpl suffix
     * @internal param string $include smarty include of the file (without .tpl suffix)
     *
     * @return void
     */
    public function render() {
        if($this->is_ajax) {
            $this->_renderAjax();
        } else {

            $this->_postSignal($this, self::BEFORE_RENDER);
            $this->_renderSmarty();
            $this->_postSignal($this, self::AFTER_RENDER);
        }
    }//end render()

    /**
     *
     */
    protected function _renderSmarty(){
        if(!$this->smarty){
            $this->smarty = Smarty::getInstance();
        }
        $this->assign('PAGE_GENERATION_TIME', sprintf("%.3f", microtime(true) - HALK_STARTUP_TIME));
        $this->assign('HALK_APPLICATION_NAME', HALK_APPLICATION_NAME);
        $this->assign('HALK_PAGE_TITLE', $this->page_title);
        $this->assign('errors', $this->errors);
        $this->assign('warnings', $this->warnings);
        $this->assign('base_url', $this->getBaseURL());

        if(isset($_REQUEST['halk_command'])) {
            $tmp = explode('/', $_REQUEST['halk_command']);
            $controller = $tmp[0];
            $method = isset($tmp[1]) ? $tmp[1] : HALK_CONTROLLER_DEFAULT_METHOD_NAME;
        } else {
            $controller = HALK_CONTROLLER_DEFAULT_NAME;
            $method = HALK_CONTROLLER_DEFAULT_METHOD_NAME;
        }
        $this->assign('controller', $controller);
        $this->assign('method', stripslashes($method));

        $_dir = $this->getTemplateDir() ? ($this->getTemplateDir() . DIRECTORY_SEPARATOR) : '';
        $this->assign('LANG_DIR', $_dir);

        $this->smarty->assign($this->view);

        $this->smarty->display($_dir . $this->getTemplate() . HALK_SMARTY_TEMPLATE_POSTFIX);
    }

    /**
     *
     */
    protected function _renderAjax(){

        $content_type = 'json';

        if(isset($_REQUEST['json'])) {
            $content_type = 'json';
        } elseif(isset($_REQUEST['xml'])) {
            $content_type = 'xml';
        } elseif(array_key_exists('extjson', $_REQUEST)) {
            $content_type = 'extjson';
        } elseif(array_key_exists('groupjson', $_REQUEST)) {
            $content_type = 'groupjson';
        }

        switch($content_type) {

            case 'json':

                header('Content-Type: application/json; charset=' . HALK_INTERNAL_ENCODING);

                $json = new ResponseJSON();
                $json->request_time = $_SERVER['REQUEST_TIME'];
                $json->request_uri = $_SERVER['REQUEST_URI'];
                $json->errors = $this->errors;
                $json->success = count($this->errors) == 0;
                $json->data = $this->view;
                $json->debug = [
                    'GENERATION_TIME' => sprintf("%.3f", microtime(true) - HALK_STARTUP_TIME)
                ];

                print $json->serialize();
                
                break;

            case 'extjson':

                header('Content-Type: application/json; charset=' . HALK_INTERNAL_ENCODING);
                $json = new ResponseExtJSON();
                $json->request_time = $_SERVER['REQUEST_TIME'];
                $json->request_uri = $_SERVER['REQUEST_URI'];
                $json->errors = $this->errors;
                $json->success = count($this->errors) == 0;
                $json->data = $this->view;

                print $json->serialize();
                break;

            case 'groupjson':

                header('Content-Type: application/json; charset=' . HALK_INTERNAL_ENCODING);
                print json_encode($this->view['objects']);
                exit;

        }
    }

    /**
     * Этот код выполняется перед вызовом любого action-* метода любого контроллера.
     *
     * @param string $method который будет выполнен после.
     * @param array  $params
     *
     * @return void
     */
    public function before($method, $params = array()) {

    }

    /**
     * Этот код выполняется после вызова любого action-* метода любого контроллера.
     *
     * @param string $method метод который был выполнен (const)
     */
    public function after($method) {

    }

    public function ajax($flag)
    {
        $this->is_ajax = $flag;
    }

}//end class
