<?php

namespace Halk\Core;

use Halk\Core\Exception\CoreException;

/**
 * Connection pool class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
class ConnectionPool extends Pool
{
    /**
     *
     * @var string
     */
    public $entity = '\Halk\Core\DBO';

    /**
     * @var array
     */
    public $db_array = [];

    /**
     * Load all connections from config.
     *
     * @throws CoreException
     * @uses Config
     */
    public function load()
    {
        $ini = Config::getInstance();
        $n = 1;

        for (; ;) {

            if ($ini->containsKey("db{$n}_connection_id")) {
                $db_name = $ini->getValue("db{$n}_name");
                $db_host = $ini->getValue("db{$n}_host");
                $db_user = $ini->getValue("db{$n}_user");
                $db_pass = $ini->getValue("db{$n}_pass");
                $db_port = $ini->containsKey("db{$n}_port") ? $ini->getValue("db{$n}_port") : null;
                $db_engine = $ini->getValue("db{$n}_engine");
                $db_connection_id = $ini->getValue("db{$n}_connection_id");
            } else {
                break;
            }
            $this->db_array[$db_connection_id] = array($db_name, $db_host, $db_user, $db_pass, $db_engine, $db_port);

            $n++;
        }
    }

    /**
     * Get connection by id.
     *
     * @param string $id connection Id (default=HALK_DEFAULT_CONNECTION_ID)
     * @return DBO
     * @throws CoreException
     */
    public function get(string $id = HALK_DEFAULT_CONNECTION_ID): DBO
    {
        if ($this->contains($id)) {
            return parent::get($id);
        } else {
            if (!isset($this->db_array[$id])) {
                throw new CoreException('Connection not found: ' . $id);
            }

            list($db_name, $db_host, $db_user, $db_pass, $db_engine, $db_port) = $this->db_array[$id];

            $dbo = new DBO($db_name, $db_host, $db_user, $db_pass, $db_engine, $id, $db_port);

            $this->add($id, $dbo);

            return $dbo;
        }
    }
}
