<?php
namespace Halk\Core\Cache;

use Halk\Core\Config;
use Halk\Core\Exception\CoreException;

/**
 * Class Redis
 * @package Halk\Core\Cache
 */
class Redis implements Cacheable
{
    protected $host;
    protected $password;
    protected $port;
    protected $prefix;
    protected $connected = false;

    private static $instance = null;
    /**
     * @var \Redis|null
     */
    private $redis = null;

    private function __construct()
    {
        $config = Config::getInstance();
        if (!$config->containsKey('redis_host')) {
            throw new CoreException('No Redis credentials');
        }
        $this->host = $config->getValue('redis_host');
        $this->port = $config->getValue('redis_port');
        $this->prefix = $config->getValue('redis_prefix');
        if ($config->containsKey('redis_password')) {
            $this->password = $config->getValue('redis_password');
        }
        $this->redis = new \Redis();
        $this->connect();
    }

    private function redis()
    {
        if (!$this->connected) {
            $this->connect();
        }
        return $this->redis;
    }

    private function __clone()
    {

    }

    public static function getInstance()
    {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function setOption($key, $value)
    {
        $this->redis()->setOption($key, $value);
    }

    /**
     * @param $key
     * @param $value
     * @param int $expire
     * @return bool
     * @throws CoreException
     */
    public function set($key, $value, $expire = 0)
    {
        return $this->redis()->set($key, $value);
    }

    /**
     * @param $key
     * @return bool|string
     * @throws CoreException
     */
    public function get($key)
    {
        return $this->redis()->get($key);
    }

    /**
     * @param $key
     * @throws CoreException
     */
    public function delete($key)
    {
        $this->redis()->del($key);
    }

    public function deleteByMask($mask)
    {
        $command = "for _,k in ipairs(redis.call('keys', ARGV[1])) do redis.call('del', k) end";
        $this->redis()->eval($command, [$mask]);
    }

    public function exists($key)
    {
        return $this->redis()->exists($key);
    }

    public function connect()
    {
        $this->connected = $this->redis->pconnect($this->host, $this->port);

        if (!$this->connected) {
            throw new CoreException('Redis is not connected');
        }
        
        if (!is_null($this->password) && !$this->redis->auth($this->password)) {
            throw new CoreException('Redis password mismatch!');
        }

        $this->redis->setOption(\Redis::OPT_SERIALIZER, \Redis::SERIALIZER_PHP);
        $this->redis->setOption(\Redis::OPT_PREFIX, $this->prefix);
    }

    /**
     * @throws CoreException
     */
    public function disconnect()
    {
        $this->redis()->close();
    }

    /**
     * @return bool
     */
    public function isConnected()
    {
        return $this->connected;
    }
}
