<?php

namespace Halk\Core;

use Halk\Core\Exception\INIException;

/**
 * INI class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
class Config extends Registry implements ConfigInterface
{
    /**
     * Array of key => value data from INI file
     * @var array
     */
    public $data = [];

    /**
     * Custom getInstance() so we can have different INI files simultenously.
     *
     * @param null|string $file path to file (default=HALK_DEFAULT_CONF)
     * @param bool $process_sections (how to parse ini file)
     * @return self
     * @throws INIException
     */
    public static function getInstance($file = HALK_DEFAULT_CONF, $process_sections = false)
    {
        static $instances = [];

        if (!isset($instances[$file])) {
            $instances[$file] = new self($file, $process_sections);
        }

        return $instances[$file];
    }

    /**
     * Constructor.
     *
     * @param string $file path to file (default=HALK_DEFAULT_CONF)
     * @param bool $process_sections (how to parse ini file)
     * @throws INIException if cannot load file
     */
    public function __construct($file = HALK_DEFAULT_CONF, $process_sections = false)
    {
        $this->data = parse_ini_file($file, $process_sections);
        if ($this->data === false) {
            throw new INIException('Failed to load config. File is: ' . $file);
        }
    }

    /**
     * Get all key/value pairs.
     *
     * @return array
     */
    public function getAll(): array
    {
        return $this->data;
    }

    /**
     * Get value by key.
     *
     * @param string $key key
     *
     * @return string
     * @throws INIException if key is missing
     */
    public function getValue(string $key)
    {
        if (!isset($this->data[$key])) {
            throw new INIException('Missing key: ' . $key);
        }

        return $this->data[$key];
    }

    /**
     * Check if config contains key.
     *
     * @param string $key key
     *
     * @return boolean true if key exists
     */
    public function containsKey(string $key): bool
    {
        return isset($this->data[$key]);
    }
}
