<?php

namespace Halk\Core\File;

/**
 * FileObject.
 * Объектный интерфейс, для работы с файлами.
 * Расширяет стандартный SplFileObject.
 *
 * @link http://php.net/manual/ru/class.splfileobject.php Документация SplFileObject
 *
 * @package Halk
 * @subpackage File
 * @author Mikhail Levykin <zenf53@yandex.ru>
 *
 */
class FileObject extends \SplFileObject
{

    /**
     * Имя файла, отличающееся от исходного.
     * Может пригодиться, при копировании под новым именем
     *
     * @var String
     * @access protected
     */
    protected $real_name;

    /**
     * Описание файла
     *
     * @var String
     * @access protected
     */
    protected $description;

    /**
     * Пытается определить реальный mime - тип файла
     *
     * @access public
     * @return String
     */
    public function getMimeType()
    {
        //return mime_content_type($this->getRealPath());
        if(strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $finfo = new \finfo(FILEINFO_MIME);
        } else {
            $finfo = new \finfo(FILEINFO_MIME, "/usr/share/misc/magic");
        }

        $mime = $finfo->file($this->getRealPath());

        if(FALSE != ($pos_charset = strrpos($mime, ';')))
            $mime = substr($mime, 0, $pos_charset);

        return $mime;
    }

    /**
     * Возвращает в виде строки содержимое файла
     *
     * @access public
     * @return NULL|String
     */
    public function getContent()
    {
        $content = null;
        while(! $this->eof()) {
            $content .= $this->fgets();
        }

        return $content;
    }

    /**
     * Возвращает содержимое файла, закодированное в base64_encode
     *
     * @access public
     * @return NULL|String
     */
    public function getEncodedContent()
    {
        return base64_encode($this->getContent());
    }

    /**
     * Возвращает расширение файла
     *
     * @param Boolean $use_realname Если установлено "реальное имя",
     *     расширение будет браться из него.
     * @access public
     * @return String|NULL
     */
    public function getFileExtension($use_realname = false)
    {
        if($use_realname == true && ! empty($this->real_name)) {
            $filename = $this->real_name;
        } else {
            $filename = $this->getBasename();
        }

        if(NULL !== ($pos = strrpos($filename, '.'))) {
            return substr($filename, ++ $pos);
        }

        return null;
    }

    /**
     * Копирует текущий файл $destination.
     * Если файл, указанный в $destination существует, заменяет его,
     * если нет - создает.
     *
     * @param String $destination Целевой путь
     * @param resource $context Stream context - ресурс
     * @see http://php.net/manual/en/function.stream-context-create.php
     * @access public
     * @return Boolean
     */
    public function copy($destination, $context = null)
    {
        if(is_resource($context))
            return copy($this->getRealPath(), $destination, $context);

        return copy($this->getRealPath(), $destination);
    }

    /**
     * Позволяет установить "реальное имя" файла, в том случае,
     * если объект был создан, например, из временного файла.
     *
     * @param String $name
     * @access public
     * @return void
     */
    public function setRealName($name)
    {
        $this->real_name = (string) $name;
    }

    /**
     * Возвращает "реальное имя" файла.
     *
     * @access public
     * @return String|NULL
     */
    public function getRealName()
    {
        return $this->real_name;
    }

    /**
     * Устанавливает описание файла
     *
     * @param String $description
     * @access public
     * @return void
     */
    public function setDescription($description)
    {
        $this->description = $description;
    }

    /**
     * Возвращает описание файла
     *
     * @access public
     * @return String
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Проверяет, был-ли файл загружен по POST
     *
     * @access public
     * @return boolean
     */
    public function isUploaded()
    {
        return is_uploaded_file($this->getRealPath());
    }
}

