<?php
namespace Halk\Core;

use Gelf\Transport\TcpTransport;
use Halk\Core\Error\GelfSkipErrorHandler;
use Halk\Core\Exception\UnexpectedSituation;
use Halk\Core\Helper\ExceptionHelper;
use Monolog\Handler\AbstractProcessingHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\NativeMailerHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger as MonologLogger;
use Psr\Log\LoggerInterface;

class Logger extends Registry
{
    const LOG_DEBUG = 1;
    const LOG_INFO = 2;
    const LOG_NOTICE = 4;
    const LOG_WARNING = 8;
    const LOG_ERROR = 16;
    const LOG_EXCEPTION = 32;
    /**
     * Array of strings
     * @var array
     */
    public static $full_log = array();

    public static $errors = array();

    public static $log_level = 0;
    /**
     * Log message.
     *
     * @param string $message message
     * @param int    $mode    type of message
     *
     * @throws \Halk\Core\Exception\UnexpectedSituation
     * @return void
     */
    public static function log($message, $mode = self::LOG_INFO)
    {
        $ini = Config::getInstance();

        // установим лог_левел из конфига
        if ($ini->containsKey('debug_level') && empty(self::$log_level)) {
            $str = $ini->getValue('debug_level');
            $str_ar = explode('|', $str);
            foreach ($str_ar as $ar) {
                self::$log_level = self::$log_level | (int)$ar;
            }
        }

        $mode_str = null;
        $hash = md5($message);

        if (($mode & self::$log_level) !== $mode) {
            return null;
        }

        switch($mode) {
            case self::LOG_INFO:

                $message = sprintf("[%s] %s", "Info", $message);
                $mode_str = "info";
                break;

            case self::LOG_ERROR:

                $message = sprintf("[%s] %s", "Error", $message);
                $mode_str = "error";
                break;

            case self::LOG_DEBUG:

                $message = sprintf("[%s] %s", "Debug", $message);
                $mode_str = "debug";
                break;

            case self::LOG_NOTICE:
                $message = sprintf("[%s] %s", "Notice", $message);
                $mode_str = "notice";
                break;

            case self::LOG_WARNING:

                $message = sprintf("[%s] %s", "Warning", $message);
                $mode_str = "warning";
                break;

            case self::LOG_EXCEPTION:
                // message already formatted
                $mode_str = "exception";
                break;
        }

        $logger = static::getLogger();

        switch(HALK_RUNNING_MODE) {

            case HALK_MODE_WWW:

                /**
                 * @todo
                 * This is very slow.
                 */

                if ($ini->containsKey('error_threshold') &&
                    intval($ini->getValue('error_threshold')) > $mode) {
                    break;
                }

                // @todo: some loggin to database, emails, etc...
                if ($mode >= self::LOG_NOTICE &&
                    $ini->containsKey('print_errors') &&
                    $ini->getValue('print_errors') == 1) {
                    self::$errors[] = $message;
                }

                if ($mode === self::LOG_DEBUG || $ini->containsKey('not_send_errors_on_email')) {
                    break;
                }

                $hostname = $_SERVER['SERVER_NAME'];

                $message .= "\n\n";
                $message .= "From hostname: " . $hostname;

                $message .= "\n\n_GET:" . var_export($_GET, true);
                $message .= "\n\n_POST:" . var_export($_POST, true);
                $message .= "\n\n_COOKIE:" . var_export($_COOKIE, true);
                $message .= "\n\n_SERVER:" . var_export($_SERVER, true);

                $logger->log($mode_str === 'exception'? 'error': $mode_str, $message);
                break;

            case HALK_MODE_CLI:
                $logger->log($mode_str === 'exception'? 'error': $mode_str, $message);
                break;

            default:

                throw new UnexpectedSituation('Unknown HALK_RUNNING_MODE');
                break;
        }

        self::$full_log[$hash] = $message;
    }

    /**
     * Report an error.
     *
     * @param string  $file    file
     * @param integer $line    line
     * @param string  $message message
     * @param integer $level   log level
     *
     * @param string $trace
     * @return void
     */
    public static function error($file, $line, $message, $level = self::LOG_ERROR, $trace = '')
    {
        if (defined('HALK_DOCUMENT_ROOT')) {
            $file = str_replace(HALK_DOCUMENT_ROOT, '.', $file);
        }

        $message = sprintf("[%s:%s] %s\n\nTrace:\n%s", $file, $line, $message, $trace);
        self::log($message, $level);
    }

    public static function exception(\Exception $e)
    {
        $message = ExceptionHelper::convertExceptionToString($e);
        self::log($message, self::LOG_EXCEPTION);
    }

    /**
     * @return LoggerInterface
     */
    protected static function getLogger()
    {
        $config = Config::getInstance();
        $section_config = new Config(HALK_DEFAULT_CONF, true);

        $app_name = defined('APPLICATION_NAME') ? APPLICATION_NAME : 'Halk';

        $logger = new MonologLogger($app_name);

        $handlers = [];

        if ($section_config->containsKey('mail')) {
            $handlers[] = static::getMailHandler();
        }

        if ($section_config->containsKey('gray_log')) {
            $handlers[] = static::getGrayLogHandler();
        }

        if ($config->containsKey('logs')) {
            $handlers[] = static::getLogFileHandler();
        }

        if (HALK_RUNNING_MODE === HALK_MODE_CLI) {
            $handlers[] = static::getStdErrorHandler();
        }


        foreach ($handlers as $lh) {
            $logger->pushHandler($lh);
        }

        return $logger;
    }

    /**
     * @return NativeMailerHandler
     */
    protected static function getMailHandler()
    {
        $config = Config::getInstance();

        $recipients = $config->getValue('errors_destination');
        $recipients = (!is_array($recipients) ? explode(',', $recipients) : $recipients);
        $sender = $config->containsKey('mail_default_from')? $config->getValue('mail_default_from'): null;

        $app_name = defined('APPLICATION_NAME') ? APPLICATION_NAME : 'Halk';
        $subject = $app_name . ' %level_name%';
        $handler = new NativeMailerHandler($recipients, $subject, $sender);

        return $handler;
    }

    /**
     * @return AbstractProcessingHandler
     */
    protected static function getStdErrorHandler()
    {
        $stream_handler = new StreamHandler(STDERR);
        return $stream_handler;
    }

    /**
     * @return AbstractProcessingHandler
     */
    protected static function getLogFileHandler()
    {
        $config = Config::getInstance();
        $global_log_dir = $config->getValue('logs');

        $log_file = rtrim($global_log_dir, '/') . sprintf('/application.%s.log', date('Y-m-d', time()));

        $log_file_path_mas = explode('/', $log_file);
        $last_el = array_pop($log_file_path_mas);
        $log_file_path = implode('/', $log_file_path_mas);

        if (!file_exists($log_file_path)) {
            mkdir($log_file_path, 0775, true);
        }

        $log_file = $log_file_path . '/' . $last_el;

        $stream_handler = new StreamHandler($log_file);

        return $stream_handler;
    }

    /**
     * @return HandlerInterface
     */
    protected static function getGrayLogHandler()
    {
        $config = new Config(HALK_DEFAULT_CONF, true);
        $gray_log_transport = new TcpTransport($config->getValue('gray_log')['host'], $config->getValue('gray_log')['port']);

        return new GelfSkipErrorHandler($gray_log_transport, MonologLogger::DEBUG, false);
    }
}
