<?php namespace Halk\Core\File;

/**
 * Upload
 * Назначение: загрузка файлов на сервер.
 * Предоставляет возможность выбора адаптера, для сохранения файлов.
 * Включает встроенный механизм проверки файлов.
 *
 * @example ./example/upload.php
 *
 * @package Halk
 * @subpackage File
 * @author Mikhail Levykin
 *
 * <code>
 * $file_storage = new StorageFilesystem($this->uploads_dir);
 *
 * $uploader = new Upload($file_storage);
 * $uploader->setValidators(array(
 *             'Size' => 1,
 *             'Type' => array('image/jpeg'),
 *             'ExtensionAllowed' => array('pdf', 'jpeg', 'jpg',
 *                    'png', 'txt', 'xml', 'doc', 'xls', 'csv')));
 * $files = $uploader->getFiles('attached');
 * foreach($files as $file) {
 * 		if($uploader->validate($file)) {
 * 			$this->receive($file, $params);
 *      }
 * }
 * </code>
 */
class Upload extends ErrorsRegistry
{
    /**
     * массив объектов UploadObject
     *
     * @var Array
     * @access protected
     */
    protected $files = array();

    /**
     * массив, определяющий вызов валидаторов, для загруженных файлов.
     *
     * @var Array
     * @access protected
     */
    protected $validators = array();

    /**
     * адаптер, определяющий конечную локацию загруженных файлов
     *
     * @var UploadStorage
     * @access protected
     */
    protected $storage;

    /**
     *
     * @var array массив, содержащий "нормализованные данные",
     * извлеченные ранее из массива _POST.
     * Примерный вид такого массива: array (
     *      array (
     *          'name' => 'image1.jpg',
     *          'type' => 'application/octet-stream',
     *          'tmp_name' => '/tmp/phpWskoGN',
     *          'error' => 0,
     *          'size' => 248050,
     *      ),
     *      array (
     *          'name' => 'image2.jpg',
     *          'type' => 'application/octet-stream',
     *          'tmp_name' => '/tmp/phpYf0jDD',
     *          'error' => 0,
     *          'size' => 375591,
     *      ),)
     *
     * @access protected
     */
    protected $array_files_normalized;

    /**
     * Массив с описаниями файлов
     *
     * @var array
     */
    protected $array_descriptions;


    /**
     * стандартные сообщения об ошибках, получаемые PHP на этапе загрузке
     *
     * @var Array
     * @access protected
     */
    protected $messagesTemplates = array(
        UPLOAD_ERR_INI_SIZE => 'Размер файла "%s" превысил максимально допустимый размер, определенный настройками сервера',
        UPLOAD_ERR_FORM_SIZE => 'Размер загружаемого файла "%s" превысил допустимый',
        UPLOAD_ERR_PARTIAL => 'Загружаемый файл был получен только частично.',
        UPLOAD_ERR_NO_FILE => 'Файл не был загружен.'
    );

    /**
     * constructor
     *
     * @param \Halk\Module\Core\Model\FilesStorage $adapter
     * @param array $array_files_normalized
     */
    public function __construct(StorageInterface $storage,
                                array $array_files_normalized = null,
                                array $array_descriptions = null)
    {
        $this->storage = $storage;
        $this->array_descriptions = $array_descriptions;
        $this->array_files_normalized = $array_files_normalized;
    }

    public function getStorage()
    {
        return $this->storage;
    }

    /**
     * Возвращает массив объектов FileObject
     *
     * @param String $field_name
     * @return Array
     * @throws UploadException
     */
    public function getFiles($field_name = null, $field_description = null)
    {
        if($this->array_files_normalized !== NULL)
            return $this->getFilesFromNormalized($this->array_files_normalized, $field_description);

        if($field_name != null && isset($_FILES[$field_name])) {
            $files[] = $_FILES[$field_name];
        } else {
            $files = $_FILES;
            foreach($files as $field_name => &$fs) {
                foreach($fs as &$param) {
                    $param = (array)$param;
                }
            }
        }

        $uploaded_files = array();

        foreach($files as $info) {

            $i = 0;
            foreach($info['name'] as $name) {
                $error = $info['error'][$i];
                $tmp_name = $info['tmp_name'][$i];

                if(empty($tmp_name)) {
                    $i++;
                    continue;
                }

                $file = new FileObject($tmp_name);
                $file->setRealName($name);
                $file->setDescription($this->getFileDescription($i, $field_description));

                $uploaded_files[] = $file;

                if(isset($this->messagesTemplates[$error])) {
                    $msg = $this->messagesTemplates[$error];
                    $this->registerError($msg, $name);
                }
                $i++;
            }
        }

        if(count($this->errors) > 0)
            return false;

        return $uploaded_files;
    }

    /**
     * Возвращает массив с объектами FileObject из "нормализованного"
     * массива с загруженными файлами.
     *
     * @param array $normalized_array
     * @param string $field_description
     * @access protected
     * @return mixed FileObject
     */
    protected function getFilesFromNormalized(array $normalized_array,
                                            $field_description)
    {
        $uploaded_files = array();

        $i = 0;
        foreach($normalized_array as $info) {
            $name = $info['name'];
            $error = isset($info['error']) ? $info['error'] : 0;
            $tmp_name = $info['tmp_name'];

            if(empty($tmp_name)) {
                $i++;
                continue;
            }

            $file = new FileObject($tmp_name);
            $file->setRealName($name);
            $file->setDescription($this->getFileDescription($i, $field_description));

            $uploaded_files[] = $file;

            if(isset($this->messagesTemplates[$error])) {
                $msg = $this->messagesTemplates[$error];
                $this->registerError($msg, $name);
            }

            $i ++;
        }

        if(count($this->errors) > 0)
            return false;

        return $uploaded_files;
    }

    /**
     * Возвращает описание файла, если оно имеется в массиве $_REQUEST, или NULL.
     *
     * @param Int $num_file Порядковый номер файла в числе загружаемых
     * @param String $field_name Имя поля, содержащего описание
     * @access protected
     * @return String|NULL
     */
    protected function getFileDescription($num_file, $field_name)
    {
        if(NULL !== $this->array_descriptions &&
         isset($this->array_descriptions[$num_file])) {
            return $this->array_descriptions[$num_file];
        }

        if($field_name != null && isset($_REQUEST[$field_name])) {
            if(is_array($_REQUEST[$field_name]) &&
             isset($_REQUEST[$field_name][$num_file])) {

                $descr = $_REQUEST[$field_name][$num_file];

                return ! empty($descr) ? $descr : null;
            }
        }

        return null;
    }

    /**
     * Определяет используемые валидаторы.
     *
     * @param Array $validators Двумерный массив, содержащий в качестве
     * 	ключей - имена валидаторов, без префикса "validate",
     * 	в качестве значений ключей - параметры вызова валидаторов.
     * 	Пример: 'Type' => array('image/png', 'image/gif').
     * @return Upload
     *
     * <code>
     * $uploader->setValidators(array(
     *         'Size' => 1,
     *         'Type' => array('image/jpeg'),
     *         'ExtensionAllowed' => array('pdf', 'jpeg', 'jpg',
     *               'png', 'txt', 'xml', 'doc', 'xls', 'csv')));
     * </code>
     */
    public function setValidators(array $validators)
    {
        $this->validators = $validators;

        return $this;
    }

    /**
     * Устанавливает шаблоны сообщений.
     *
     * @param Array $templates
     * @return Upload
     */
    public function setMessagesTemplates(array $templates)
    {
        $this->messagesTemplates = array_merge($this->messagesTemplates, $templates);

        return $this;
    }

    /**
     * Производит валидацию загруженного файла
     *
     * @param FileObject $file
     * @throws \Exception
     * @return boolean
     */
    public function validate(FileObject $file)
    {
        $validator = new Validator($file);
        if(!in_array('isUploaded', $this->validators)) {
            $this->validators['isUploaded'] = null;
        }

        if(!$validator->validateQueue($this->validators)) {
            $this->errors = array_merge($this->getErrors(), $validator->getErrors());
            return false;
        }

        return true;
    }

    /**
     * Выполняет проверку, действительно - ли файл был загружен,
     * вызывает установленный класс Halk_Core_Model_FileStorage
     * и возвращает результаты работы eго метода "saveFile".
     *
     * @param  FileObject $file
     * @param Array $params
     * @access public
     * @return mixed
     */
    public function receive(FileObject $file, array $params)
    {
        return $this->storage->saveFile($file, $params);
    }

    public function unlink($file)
    {
        return $this->storage->unlink($file);
    }

}

