<?php
namespace Halk\Core;

/**
 * Class Error
 * @package Halk\Core
 */
class Error {

    /**
     * @var bool
     */
    private static $registered = false;

    protected static $levels = [
        E_ERROR,
        E_USER_ERROR,
        E_COMPILE_ERROR
    ];

    /**
     * Register error handlers
     */
    public static function register()
    {
        if (!self::$registered) {
            register_shutdown_function('Halk\Core\Error::fatal');
            set_exception_handler('Halk\Core\Error::exception');
            set_error_handler('Halk\Core\Error::warning');

            if (HALK_RUNNING_MODE == HALK_MODE_CLI) {
                error_reporting(E_ALL);
                ini_set("display_errors", "stderr");
            }
            self::$registered = true;
        }
    }

    public static function fatal()
    {
        if(HALK_RUNNING_MODE == HALK_MODE_CLI) {
            return true;
        }

        $error = error_get_last();

        if (in_array($error['type'], self::$levels)) {
            $error_message = var_export($error, true);
            error_log($error_message);
            header('HTTP/1.1 503 Service Unavailable');
            echo <<<EOF
<link href="/css/index/fullscreen_errors.css" rel="stylesheet">
<div class="blue-sky">
    <div class="body-content">
        <h1>Произошла внутренняя ошибка!</h1>
        <p>Наши специалисты уже знают об этой проблеме и работают над ее устранением.</p>
        <p>Приносим извинения за неудобства.</p>
    </div>
</div>
EOF;
            Logger::error($error['file'], $error['line'], "Halk FATAL ERROR\n" . $error_message, Logger::LOG_ERROR);
        }
    }

    public static function exception(\Exception $exception)
    {
        if(!defined('HALK_RUNNING_TESTS')) {
            Logger::exception($exception);
        }
    }

    public static function warning($code, $message, $file, $line)
    {
        switch ($code) {

            case E_WARNING:
            case E_USER_WARNING:
                $level = Logger::LOG_WARNING;
                break;
            case E_ERROR:
            case E_USER_ERROR:
                $level = Logger::LOG_ERROR;
                break;
            case E_USER_NOTICE:
            case E_NOTICE:
                $level = Logger::LOG_NOTICE;
                break;
            case E_STRICT:
                // TODO we cannot handle E_STRICT at the moment, Logger issues
                return;

            default:

                $level = Logger::LOG_ERROR;
                break;
        }

        if(error_reporting() !== 0) {

            $trace = array_reverse(debug_backtrace());
            $str_trace = '';

            foreach ($trace as $call) {
                if ($call['function'] == 'warning') {
                    continue;
                }
                $args = array();
                if (isset($call['args']) && is_array($call['args'])) {
                    foreach ($call['args'] as $arg) {
                        $args[] = is_scalar($arg) ? $arg : (is_object($arg) ? get_class($arg) : gettype($arg));
                    }
                }
                $args = implode(', ',$args);
                if (isset($call['class']) && $call['class']) {
                    $call['class'] .= $call['type'];
                } else {
                    $call['class'] = '';
                }
                if (!isset($call['line'])) {
                    $call['line'] = '<none>';
                }
                if (!isset($call['function'])) {
                    $call['function'] = '<none>';
                }
                $str_trace .= @$call['file'].':'.$call['line'].' '.$call['class'].$call['function'].'('.$args.')'."\n";
            }

            Logger::error($file, $line, $message, $level, $str_trace);
        }
    }
}