<?php

namespace Halk\Core\Db\Sql;

class Expression
{

    /**
     * предикат, определяющий использование OR в выражении
     *
     * @var String
     */
    const PREDICATE_OR = 'OR';

    /**
     * предикат, определяющий использование AND в выражении
     *
     * @var String
     */
    const PREDICATE_AND = 'AND';

    const ENGINE_MYSQL = 'mysql';
    const ENGINE_PGSQL = 'pgsql';

    /**
     * оператор =
     *
     * @var String
     */
    const OP_EQ = '=';

    /**
     * оператор !=
     *
     * @var String
     */
    const OP_NE = '!=';

    /**
     * оператор <
     *
     * @var String
     */
    const OP_LT = '<';

    /**
     * оператор >
     *
     * @var String
     */
    const OP_GT = '>';

    /**
     * оператор <=
     *
     * @var String
     */
    const OP_LTE = '<=';

    /**
     * оператор >=
     *
     * @var String
     */
    const OP_GTE = '>=';

    /**
     * оператор IN
     *
     * @var String
     */
    const OP_IN = 'IN';

    /**
     * оператор IS
     *
     * @var String
     */
    const OP_IS = 'IS';

    /**
     * оператор IS NOT
     *
     * @var String
     */
    const OP_IS_NOT = 'IS NOT';

    /**
     * оператор BETWEEN %s AND %s
     *
     * @var String
     */
    const OP_BETWEEN = 'BETWEEN %s AND %s';

    /**
     * оператор NOT BETWEEN %s AND %s
     *
     * @var String
     */
    const OP_NOT_BETWEEN = 'NOT BETWEEN %s AND %s';

    /**
     * оператор IS NOT IN
     *
     * @var String
     */
    const OP_IS_NOT_IN = 'NOT IN';

    /**
     * оператор ILIKE ?
     *
     * @var String
     */
    const OP_ILIKE = 'ILIKE %s';

    /**
     * оператор ILIKE '%%s
     *
     * @var String
     */
    const OP_ILIKE_LEFT = '%%%s';

    /**
     * оператор ILIKE '%s%'
     *
     * @var String
     */
    const OP_ILIKE_RIGHT = '%s%%';

    /**
     * опреатор ILIKE '%%s%'
     *
     * @var String
     */
    const OP_ILIKE_TWO = '%%%s%%';

    /**
     * оператор LIKE
     *
     * @var String
     */
    const OP_LIKE = 'LIKE %s';

    /**
     * опреатор LIKE '%%s%'
     *
     * @var String
     */
    const OP_LIKE_TWO = '%%%s%%';

    /**
     * опреатор LIKE '%s%'
     *
     * @var String
     */
    const OP_LIKE_RIGHT = '%s%%';

    /**
     * опреатор LIKE '%%s'
     *
     * @var String
     */
    const OP_LIKE_LEFT = '%%%s';

    const OP_REGEX = '~*';

    /**
     * дефолтный оператор
     *
     * @var String
     */
    const OP_DEFAULT = self::OP_EQ;

    const TYPE_SIMPLE = 'simple';
    const TYPE_GROUP = 'group';
    /**
     * placeholder symbol
     *
     * @var String
     */
    const PLACEHOLDER = '?';

    const TPL_SELF_OBJ = 'selfObj';
    const TPL_TYPE_NOTSUPP = 'notSqlExpr';
    const TPL_OPERATOR_NOTSUPP = 'opNotSupp';
    const TPL_VALUE_NOARR = 'valueNotArr';
    const TPL_VALUE_NULL = 'valueNull';

    protected $messageTemplates = array(
        self::TPL_SELF_OBJ => 'you can not pass an object to itself!',
        self::TPL_TYPE_NOTSUPP => 'object does not support the interface SqlExpression!',
        self::TPL_OPERATOR_NOTSUPP => '%s operator %s is not supported',
        self::TPL_VALUE_NOARR => '%s: value is not an array',
        self::TPL_VALUE_NULL => 'value can not be null'
    );

    /**
     * массив "простых" выражений.
     * пример: "worker_id" <= 1245546
     *
     * @var Array
     * @access protected
     */
    protected $expressions = array();

    /**
     * используемый тип БД
     *
     * @var String
     * @access protected
     */
    protected $engine;

    /**
     * определяет, будут-ли имена полей браться в кавычки
     *
     * @var Boolean
     * @access protected
     */
    protected $no_quoted_column = false;

    /**
     * плейсхолдер, для колонок
     *
     * @var String
     * @access protected
     */
    protected $column_placeholder;

    /**
     * массив всех разрешенных операторов,
     * для проверки правильности переданного оператора.
     *
     * @var Array
     * @access protected
     */
    protected $operators = array(
        self::OP_EQ,
        self::OP_NE,
        self::OP_LT,
        self::OP_GT,
        self::OP_LTE,
        self::OP_GTE,
        self::OP_IN,
        self::OP_IS,
        self::OP_IS_NOT,
        self::OP_BETWEEN,
        self::OP_NOT_BETWEEN,
        self::OP_IS_NOT_IN,
        self::OP_LIKE,
        self::OP_LIKE_LEFT,
        self::OP_LIKE_RIGHT,
        self::OP_LIKE_TWO,
        self::OP_ILIKE,
        self::OP_ILIKE_LEFT,
        self::OP_ILIKE_RIGHT,
        self::OP_ILIKE_TWO,
        self::OP_REGEX
    );

    /**
     * массив значений, используемых в запросе
     *
     * @var Array
     * @access protected
     */
    protected $values = array();

    /**
     *
     * @var Array
     * @access protected
     */
    protected $columns = array();

    /**
     *
     * @var Array
     * @access protected
     */
    protected $columns_replacements = array();

    /**
     *
     * @var Array
     * @access protected
     */
    protected $columns_variation = array();

    /**
     *
     * @var Boolean
     * @access protected
     */
    protected $character_varying = false;

    /**
     * constructor
     *
     * @param String $engine
     */
    public function __construct($engine = 'pgsql')
    {
        if ($engine != self::ENGINE_MYSQL && $engine != self::ENGINE_PGSQL) {
            $this->enhine = self::ENGINE_PGSQL;
        } else {
            $this->engine = $engine;
        }
    }

    /**
     * ПРостой фабричный метод
     * @param string $engine
     * @param string $engine
     * @return Expression
     */
    public static function factory($engine = 'pgsql')
    {
        return new self($engine);
    }

    /**
     * возвращает where - выражение, в виде строки
     *
     * @return string
     */
    public function __toString()
    {
        return $this->getExpressionString(true);
    }

    /**
     * Возвращает строку с готовым выражением, для подстановки в WHERE,
     * без оператора "WHERE".
     *
     * @param bool $wrap
     * @internal param bool $clearValues
     * @return string
     */
    public function getExpressionString($wrap = true)
    {
        $predicate = null;
        $this->clearValues();
        $self_expressions = '';

        foreach ($this->expressions as $expr) {
            if ($expr['type'] === self::TYPE_SIMPLE) {
                // обходим "простые выражения"
                $expr = $expr['obj'];
                if (isset($expr['string'])) {
                    $self_expressions .= $expr['predicate'];
                    $self_expressions .= '(' . $expr['string'] . ')' . PHP_EOL;
                    continue;
                }

                $c = &$expr['column'];
                $v = &$expr['value'];
                $o = &$expr['operator'];
                $p = &$expr['predicate'];

                if (array_key_exists($c, $this->columns_variation)) {
                    $col_var = $this->columns_variation[$c];
                    list($predicate, $array_columns) = each($col_var);

                    $sub_expr = $this->_buildUnit($c, $v, $o);
                    $sub_expr = $p . '(' . $sub_expr;
                    foreach ($array_columns as $cname) {
                        $sub_expr2 = $this->_buildUnit($cname, $v, $o);
                        $sub_expr .= $predicate;
                        $sub_expr .= '(' . $sub_expr2 . ')';
                    }
                    $sub_expr .= ')';

                } else {
                    $sub_expr = $this->_buildUnit($c, $v, $o, $p);
                }

                $self_expressions .= $sub_expr . PHP_EOL;
            } elseif ($expr['type'] === self::TYPE_GROUP) {
                // обходим группы выражений
                $data = $expr['obj'];
                $group = $data['object'];

                $group->setNoColumnQuote($this->no_quoted_column);
                $group->setCharacterVarying($this->character_varying);

                if (!empty($self_expressions)) {
                    $self_expressions .= $data['predicate'];
                }
                $self_expressions .= $group->getExpressionString();
            }
        }

        if (!empty($self_expressions)) {
            $self_expressions = '(' . $self_expressions . ')';
        }

        return $self_expressions;
    }

    /**
     * Очищает массив значений полей
     *
     * @access public
     * @return void
     */
    public function clearValues()
    {
        $this->values = array();
    }

    /**
     * Собирает отдельное выражение
     *
     * @param String $column
     * @param mixed $value
     * @param String $operator
     * @param String $predicate
     * @return string
     */
    protected function _buildUnit(
        $column,
        $value,
        $operator,
        $predicate = null
    ) {
        $column_name = $this->_getRealColumnName($column);

        $column = $this->_escapeColumn($column_name);
        $column = $this->_getColumnVarying($column, $operator);

        $operator = $this->_getOperator($operator);
        $value = $this->_prepareValuePlaceholder($value, $operator);

        return sprintf('%s%s %s %s', $predicate, $column, $operator, $value);
    }

    /**
     * Возвращает "реальное" имя колонки, с учетом массива $columns_replacements
     *
     * @param String $column_name
     * @return String
     */
    protected function _getRealColumnName($column_name)
    {
        if (array_key_exists($column_name, $this->columns_replacements)) {
            $column_name = $this->columns_replacements[$column_name];
        }

        return $column_name;
    }

    /**
     * экранирует имя колонки, для корректной подстановки в SQL - код
     *
     * @param String $column_name
     * @return String
     */
    protected function _escapeColumn($column_name)
    {
        $placeholder = $this->_getColumnPlaceholder();

        return sprintf($placeholder, $column_name);
    }

    /**
     * Возвращает строку для подстановки имени колонки,
     * с учетом особенностей БД.
     *
     * @access protected
     * @return string
     */
    protected function _getColumnPlaceholder()
    {
        if (null == $this->column_placeholder) {

            if ($this->no_quoted_column === true) {
                return $this->column_placeholder = '%s';
            }

            switch ($this->engine) {
                case self::ENGINE_MYSQL :
                    $this->column_placeholder = '`%s`';
                    break;
                case self::ENGINE_PGSQL :
                    $this->column_placeholder = '"%s"';
                    break;
                default:
                    $this->column_placeholder = '`%s`';
                    break;
            }
        }

        return $this->column_placeholder;
    }

    /**
     * Добавляет спецификатор "::varchar" к имени колонки
     *
     * @param string $column
     * @param $operator
     * @access protected
     * @return string
     */
    protected function _getColumnVarying($column, $operator)
    {
        if ($this->engine == self::ENGINE_PGSQL) {
            if ($operator == self::OP_ILIKE || $operator == self::OP_ILIKE_LEFT || $operator == self::OP_ILIKE_RIGHT || $operator == self::OP_ILIKE_TWO) {
                $column .= '::varchar';
            } else {

                if ($this->character_varying === true) {
                    return $column .= '::varchar';
                }
            }
        }

        return $column;
    }

    /**
     * Возвращает оператор, необходимый, для создания SQL - выражения
     *
     * @param String $operator
     * @throws \Exception
     * @return String
     */
    protected function _getOperator($operator)
    {
        $operator = $this->normalizeOperator($operator);

        if (false === $operator) {
            $tpl = $this->messageTemplates[self::TPL_OPERATOR_NOTSUPP];
            $msg = sprintf($tpl, __METHOD__, var_export($operator, true));

            throw new \Exception($msg);
        }

        return $operator;
    }

    /**
     * Приводит оператор к "нормальному" виду, определенному в константах класса
     *
     * @param string $op
     * @return unknown|boolean
     */
    public function normalizeOperator($op)
    {
        $op = trim($op);
        foreach ($this->operators as $operator) {
            if (strtoupper($op) == strtoupper($operator)) {
                return $operator;
            }
        }

        return false;
    }

    /**
     * экранирует значение
     *
     * @param mixed $value
     * @param $operator
     * @throws \Exception
     * @internal param String $type
     * @return String
     */
    protected function _prepareValuePlaceholder($value, &$operator)
    {
        switch ($operator) {
            case self::OP_BETWEEN:
            case self::OP_NOT_BETWEEN:
                if (is_array($value) && count($value) == 2) {
                    $sql = sprintf($operator, self::PLACEHOLDER, self::PLACEHOLDER);
                    $operator = '';

                    return $sql;
                } else {
                    $tpl = $this->messageTemplates[self::TPL_VALUE_NOARR];

                    throw new \Exception($tpl, __METHOD__);
                }
            case self::OP_ILIKE:
            case self::OP_ILIKE_LEFT:
            case self::OP_ILIKE_RIGHT:
            case self::OP_ILIKE_TWO:
            case self::OP_LIKE:
            case self::OP_LIKE_LEFT:
            case self::OP_LIKE_RIGHT:
            case self::OP_LIKE_TWO:
                if ($this->engine == self::ENGINE_MYSQL) {
                    $operator = '';

                    return sprintf(self::OP_LIKE, self::PLACEHOLDER);
                } else {
                    $operator = '';

                    return sprintf(self::OP_ILIKE, self::PLACEHOLDER);
                }
        }

        if ($value === null) {
            return 'NULL';
        }

        if (is_array($value)) {
            $vs = array_fill(0, count($value), self::PLACEHOLDER);

            return '(' . implode(',', $vs) . ')';
        }

        return self::PLACEHOLDER;
    }

    /**
     * определяет отсутствие ограничивающих кавычек в имени поля
     *
     * @param Boolean $flag
     * @return Expression
     */
    public function setNoColumnQuote($flag = true)
    {
        $this->no_quoted_column = $flag;
        foreach ($this->expressions as $expr) {
            if ($expr['type'] == self::TYPE_GROUP) {
                $expr_group = $expr['obj']['object'];
                $expr_group->setNoColumnQuote(true);
            }
        }

        return $this;
    }

    /**
     * Сигнализирует о необходимомсти приведения значений колонок к типу varchar
     *
     * @param Boolean $flag
     * @access public
     * @return Expression
     */
    public function setCharacterVarying($flag = true)
    {
        $this->character_varying = $flag;
        $this->no_quoted_column = $flag;

        return $this;
    }

    /**
     *
     * @access public
     * @return Array
     */
    public function getColumnsReplacements()
    {
        return $this->columns_replacements;
    }

    /**
     *
     * @param array $replacements
     * @access public
     * @return Expression
     */
    public function setColumnsReplacements(array $replacements)
    {
        $this->columns_replacements = $replacements;

        return $this;
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие
     *
     * @param String $column имя колонки
     * @param mixed $value значение
     * @param String $operator используемый оператор
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function add(
        $column,
        $value,
        $operator = '=',
        $predicate = 'AND'
    ) {
        return $this->addExpression($column, $value, $operator, $predicate);
    }

    /**
     * Добавляет условие
     *
     * @param String $column имя колонки
     * @param mixed $value значение
     * @param String $operator используемый оператор
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function addExpression(
        $column,
        $value,
        $operator = '=',
        $predicate = 'AND'
    ) {
        $predicate = $this->_getPredicate($predicate);

        if ($value === null && $operator != self::OP_IS_NOT) {
            $operator = self::OP_IS;
        }

        if ($operator == self::OP_ILIKE_TWO || $operator == self::OP_ILIKE_LEFT || $operator == self::OP_ILIKE_RIGHT || $operator == self::OP_LIKE_TWO || $operator == self::OP_LIKE_LEFT || $operator == self::OP_LIKE_RIGHT) {
            $value = sprintf($operator, $value);
        }

        $this->columns[] = $column;

        $this->expressions[] = [
            'type' => self::TYPE_SIMPLE,
            'obj' => [
                'column' => $column,
                'value' => $value,
                'operator' => $operator,
                'predicate' => $predicate
            ]
        ];

        return $this;
    }

    /**
     * Возвращает строку с используемым предикатом.
     * Если это первое значение, то предикат не добавляется
     * @param String $predicate 'OR'|'AND'|null
     * @return String
     */
    protected function _getPredicate($predicate)
    {
        if ($predicate === null || count($this->expressions) == 0) {
            return '';
        }

        $predicate = strtoupper(trim($predicate));

        if ($predicate === self::PREDICATE_AND) {
            return ' AND ';
        }

        return ' OR ';
    }

    /**
     * Добавляет условие <column> = <value>
     *
     * @param string $column
     * @param mixed $value
     * @param string $predicate
     * @return Expression
     */
    public function eq($column, $value, $predicate = 'AND')
    {
        return $this->addExpression($column, $value, self::OP_EQ, $predicate);
    }

    /**
     * Добавляет условие <column> != <value>
     *
     * @param string $column
     * @param mixed $value
     * @param string $predicate
     * @return Expression
     */
    public function ne($column, $value, $predicate = 'AND')
    {
        return $this->addExpression($column, $value, self::OP_NE, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpressionLike
     * добавляет условие ILIKE %<string>%
     *
     * @param String $column имя колонки
     * @param mixed $value значение
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function like($column, $value, $predicate = 'AND')
    {
        return $this->addExpressionLike($column, $value, $predicate);
    }

    /**
     * Добавляет условие ILIKE %<string>%
     *
     * @param String $column имя колонки
     * @param mixed $value значение
     * @param String $predicate предикат(OR|AND)
     * @throws \Exception
     * @return Expression
     */
    public function addExpressionLike($column, $value, $predicate = 'AND')
    {
        $predicate = $this->_getPredicate($predicate);

        if ($value === null) {
            throw new \Exception($this->messageTemplates[self::TPL_VALUE_NULL]);
        }

        if ($this->engine == self::ENGINE_PGSQL) {
            $this->addExpression($column, $value, self::OP_ILIKE_TWO, $predicate);
        } else {
            $this->addExpression($column, $value, self::OP_LIKE_TWO, $predicate);
        }

        return $this;
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие BETWEEN %s AND %s
     *
     * @param String $column имя колонки
     * @param mixed $values значения
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function between($column, array $values, $predicate = 'AND')
    {
        return $this->addExpression($column, $values, self::OP_BETWEEN, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие NOT BETWEEN %s AND %s
     *
     * @param String $column имя колонки
     * @param mixed $values значения
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function notBetween($column, array $values, $predicate = 'AND')
    {
        return $this->addExpression($column, $values, self::OP_NOT_BETWEEN, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие %s IS NULL
     *
     * @param String $column имя колонки
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function isNull($column, $predicate = 'AND')
    {
        return $this->addExpression($column, null, self::OP_IS, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие %s IS NOT NULL
     *
     * @param String $column имя колонки
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function isNotNull($column, $predicate = 'AND')
    {
        return $this->addExpression($column, null, self::OP_IS_NOT, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие %s IN ()
     *
     * @param String $column имя колонки
     * @param mixed $values значения
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function in($column, array $values, $predicate = 'AND')
    {
        return $this->addExpression($column, $values, self::OP_IN, $predicate);
    }

    /**
     * псевдоним Halk\Core\Db\Sql\Expression::addExpression
     * добавляет условие %s IN ()
     *
     * @param String $column имя колонки
     * @param mixed $values значения
     * @param String $predicate предикат(OR|AND)
     * @return Expression
     */
    public function notIn($column, array $values, $predicate = 'AND')
    {
        return $this->addExpression($column, $values, self::OP_IS_NOT_IN, $predicate);
    }

    /**
     * Псевдоним Halk\Core\Db\Sql\Expression::addExpressionsGroup
     * добавляет группу условий
     *
     * @param array $expressions массив, содержащий набор объектов SqlExpression
     * @param array $predicates массив предикатов, для каждого из объектов
     * @throws \Exception
     * @return Expression
     */
    public function group(array $expressions, array $predicates = array())
    {
        return $this->addExpressionsGroup($expressions, $predicates);
    }

    /**
     * Добавляет группу условий
     *
     * @param array $expressions массив, содержащий набор объектов SqlExpression
     * @param array $predicates массив предикатов, для каждого из объектов
     * @throws \Exception
     * @return Expression
     */
    public function addExpressionsGroup(
        array $expressions,
        array $predicates = array()
    ) {
        foreach ($expressions as $index => $obj) {
            if (!$obj instanceof Expression) {
                throw new \Exception($this->messageTemplates[self::TPL_TYPE_NOTSUPP]);
            }

            if ($obj === $this) {
                throw new \Exception($this->messageTemplates[self::TPL_SELF_OBJ]);
            }

            if (!isset($predicates[$index])) {
                $predicate = $this->_getPredicate(self::PREDICATE_AND);
            } else {
                $predicate = $this->_getPredicate($predicates[$index]);
            }

            $this->columns = array_merge($this->columns, $obj->getColumnNames());

            $this->expressions[] = [
                'type' => self::TYPE_GROUP,
                'obj' => [
                    'object' => $obj,
                    'predicate' => $predicate
                ]
            ];
        }

        return $this;
    }

    /**
     * Возвращает имена колонок, на тот случай, если они потребуются отдельно
     *
     * @access public
     * @return Array
     */
    public function getColumnNames()
    {
        return $this->columns;
    }

    /**
     * Метод, для добавления в результирующее выражение подвыражения в виде строки.
     * Это для сложных случаев.
     * <code>
     * $expr->addExpressionString('? = ANY(groups_id)', $id, 'and');
     * $expr->string('group_id IN(?,?,?)', array(1,2,3), 'or');
     * </code>
     *
     * @param string $string
     * @param mixed $values
     * @param string $predicate
     * @return Expression
     *
     * Работа метода нуждается в проверке.
     */
    public function string($string, $values = null, $predicate = 'AND')
    {
        return $this->addExpressionString($string, $values, $predicate);
    }

    /**
     * Метод, для добавления в результирующее выражение подвыражения в виде строки.
     * Это для сложных случаев.
     * <code>
     * $expr->addExpressionString('? = ANY(groups_id)', $id, 'and');
     * $expr->string('group_id IN(?,?,?)', array(1,2,3), 'or');
     * </code>
     *
     * @param string $string
     * @param mixed $values
     * @param string $predicate
     * @return Expression
     *
     * Работа метода нуждается в проверке.
     */
    public function addExpressionString($string, $values, $predicate = 'AND')
    {
        $predicate = $this->_getPredicate($predicate);

        $this->expressions[] = [
            'type' => self::TYPE_SIMPLE,
            'obj' => [
                'string' => $string,
                'value' => $values,
                'predicate' => $predicate
            ]
        ];

        return $this;
    }

    public function addExpressionRegex($column, $value, $predicate = 'AND')
    {
        $predicate = $this->_getPredicate($predicate);
        if ($value === null) {
            throw new \Exception($this->messageTemplates[self::TPL_VALUE_NULL]);
        }
        $this->addExpression($column, $value, self::OP_REGEX, $predicate);

        return $this;
    }

    /**
     * Позволяет указать несколько полей, имеющих сходное значение с полем $column.
     * Т.е. позволяет построить аналог выражения
     * "$column_name = 'x' AND column2 = 'x' AND column3 = 'x'".
     * вместо предиката "AND" может использоваться "OR".
     *
     * @param String $column_name Имя колонки,
     * которая добавляется с помощью метода "addExpression"
     * @param Array $variation_columns Массив имен полей, имеющих сходное значение
     * @param String $predicate Предикат AND|OR
     * @access public
     * @return Expression
     */
    public function addVariationColumns(
        $column_name,
        array $variation_columns,
        $predicate = 'AND'
    ) {
        $predicate = $this->_getPredicate($predicate);

        $this->columns_variation[$column_name] = array(
            $predicate => $variation_columns
        );

        return $this;
    }

    /**
     * Возвращает одномерный массив всех значений, используемых в запросе,
     * с учетом их подстановки в плейсхолдеры.
     *
     * @access public
     * @return Array
     */
    public function getValues()
    {
        $this->clearValues();

        foreach ($this->expressions as $expr) {
            if ($expr['type'] == self::TYPE_SIMPLE) {
                $expr = $expr['obj'];
                if (is_null($expr['value'])) {
                    continue;
                }

                if (isset($expr['column']) && array_key_exists($expr['column'], $this->columns_variation)) {

                    $col_var = $this->columns_variation[$expr['column']];
                    list($predicate, $array_columns) = each($col_var);
                    $count = count($array_columns);

                    if (is_array($expr['value'])) {
                        for ($i = 0; $i < $count; $i++) {
                            $this->values = array_merge($this->values, $expr['value']);
                        }
                    } else {
                        for ($i = 0; $i < $count; $i++) {
                            $this->values[] = $expr['value'];
                        }
                    }
                }

                if (is_array($expr['value'])) {
                    $this->values = array_merge($this->values, $expr['value']);
                } else {
                    $this->values[] = $expr['value'];
                }
            } elseif ($expr['type'] == self::TYPE_GROUP) {
                $this->values = array_merge($this->values, $expr['obj']['object']->getValues());
            }
        }


        if ($this->character_varying) {
            foreach ($this->values as &$val) {
                $val = (string)$val;
            }
        }
        return $this->values;
    }

    /**
     * Возвращает число всех выражений
     *
     * @access public
     * @return Integer
     */
    public function getExpressionCount()
    {
        $c = 0;
        foreach ($this->expressions as $expr) {
            if ($expr['type'] == self::TYPE_SIMPLE) {
                $c++;
            } elseif ($expr['type'] == self::TYPE_GROUP) {
                $c += $expr['obj']['object']->getExpressionCount();
            }
        }

        foreach ($this->columns_variation as $v) {
            list(, $arr) = each($v);
            $c += count($arr);
        }

        return $c;
    }


}
