<?php

namespace Halk\Core\Helper\XLSXWriter;

use Exception;
use ZipArchive;

class XLSXWriter
{
    const EXCEL_2007_MAX_ROW = 1048576;
    const EXCEL_2007_MAX_COL = 16384;

    protected $author = 'author';
    protected $sheets = array();
    protected $shared_strings = array();
    protected $shared_string_count = 0;
    protected $temp_files = array();

    protected $current_sheet = '';

    protected $file;
    protected $row_num;
    protected $header_offset;
    protected $cell_formats_arr;
    protected $style_cols = '<col collapsed="false" hidden="false" max="1025" min="1" style="0" width="41.5"/>';
    protected $filename = 'Отчет.xlsx';
    protected $styles = [
        'color_header' => '74869A',
        'border' => TRUE,
    ];

    public function __construct()
    {
        if (!class_exists('ZipArchive')) {
            throw new Exception('ZipArchive not found');
        }
    }

    public function setAuthor($author='')
    {
        $this->author=$author;
    }

    public function __destruct()
    {
        if (!empty($this->temp_files)) {
            foreach($this->temp_files as $temp_file) {
                @unlink($temp_file);
            }
        }
    }

    protected function tempFilename()
    {
        $filename = tempnam("/tmp", "xlsx_writer_");
        $this->temp_files[] = $filename;
        return $filename;
    }

    public function writeToString()
    {
        $temp_file = $this->tempFilename();
        self::writeToFile($temp_file);
        $string = file_get_contents($temp_file);
        return $string;
    }

    public function writeToFile($filename)
    {
        foreach($this->sheets as $sheet_name => $sheet) {
            self::finalizeSheet($sheet_name);
        }

        @unlink($filename);
        $zip = new ZipArchive();
        if (!$zip->open($filename, ZipArchive::CREATE)) {
            throw new Exception('Не удалось открыть архив');
        }

        $zip->addEmptyDir("docProps/");
        $zip->addFromString("docProps/app.xml" , self::buildAppXML() );
        $zip->addFromString("docProps/core.xml", self::buildCoreXML());

        $zip->addEmptyDir("_rels/");
        $zip->addFromString("_rels/.rels", self::buildRelationshipsXML());

        $zip->addEmptyDir("xl/worksheets/");
        foreach($this->sheets as $sheet) {
            $zip->addFile($sheet->filename, "xl/worksheets/".$sheet->xmlname );
        }
        if (!empty($this->shared_strings)) {
            $zip->addFile($this->writeSharedStringsXML(), "xl/sharedStrings.xml" );
        }
        $zip->addFromString("xl/workbook.xml", self::buildWorkbookXML() );
        $zip->addFile($this->writeStylesXML(), "xl/styles.xml" );
        $zip->addFromString("[Content_Types].xml", self::buildContentTypesXML() );

        $zip->addEmptyDir("xl/_rels/");
        $zip->addFromString("xl/_rels/workbook.xml.rels", self::buildWorkbookRelsXML() );
        $zip->close();
    }

    public function writeToOut()
    {
        header('Content-Type: application/vnd.ms-excel;charset=utf-8');
        header('Content-Disposition: attachment; filename="'. $this->filename .'"');
        header("Content-Type: application/force-download");
        header("Content-Type: application/octet-stream");
        header("Content-Type: application/download");
        header("Content-Transfer-Encoding: binary");
        echo $this->writeToString();
        exit;
    }

    protected function initializeSheet($sheet_name)
    {
        if ($this->current_sheet==$sheet_name || isset($this->sheets[$sheet_name]))
            return;

        $sheet_filename = $this->tempFilename();
        $sheet_xmlname = 'sheet' . (count($this->sheets) + 1).".xml";
        $this->sheets[$sheet_name] = (object)array(
            'filename' => $sheet_filename,
            'sheetname' => $sheet_name,
            'xmlname' => $sheet_xmlname,
            'header_offset' => 0,
            'row_count' => 0,
            'file_writer' => new XLSXWriterBuffererWriter($sheet_filename),
            'cell_formats' => array(),
            'max_cell_tag_start' => 0,
            'max_cell_tag_end' => 0,
            'finalized' => false,
        );
        $sheet = &$this->sheets[$sheet_name];
        $tabselected = count($this->sheets) == 1 ? 'true' : 'false';//only first sheet is selected
        $max_cell = XLSXWriter::xlsCell(self::EXCEL_2007_MAX_ROW, self::EXCEL_2007_MAX_COL);
        $sheet->file_writer->write('<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' . "\n");
        $sheet->file_writer->write('<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">');
        $sheet->file_writer->write(  '<sheetPr filterMode="false">');
        $sheet->file_writer->write(    '<pageSetUpPr fitToPage="false"/>');
        $sheet->file_writer->write(  '</sheetPr>');
        $sheet->max_cell_tag_start = $sheet->file_writer->ftell();
        $sheet->file_writer->write('<dimension ref="A1:' . $max_cell . '"/>');
        $sheet->max_cell_tag_end = $sheet->file_writer->ftell();
        $sheet->file_writer->write(  '<sheetViews>');
        $sheet->file_writer->write(    '<sheetView colorId="64" defaultGridColor="true" rightToLeft="false" showFormulas="false" showGridLines="true" showOutlineSymbols="true" showRowColHeaders="true" showZeros="true" tabSelected="' . $tabselected . '" topLeftCell="A1" view="normal" windowProtection="false" workbookViewId="0" zoomScale="100" zoomScaleNormal="100" zoomScalePageLayoutView="100">');
        $sheet->file_writer->write(      '<selection activeCell="A1" activeCellId="0" pane="topLeft" sqref="A1"/>');
        $sheet->file_writer->write(    '</sheetView>');
        $sheet->file_writer->write(  '</sheetViews>');
        $sheet->file_writer->write(  '<cols>');
        $sheet->file_writer->write(    $this->style_cols);
        $sheet->file_writer->write(  '</cols>');
        $sheet->file_writer->write(  '<sheetData>');
    }

    public function writeSheetHeader($sheet_name, array $header_types)
    {
        if (empty($sheet_name) || empty($header_types) || !empty($this->sheets[$sheet_name]))
            return;

        self::initializeSheet($sheet_name);
        $sheet = &$this->sheets[$sheet_name];
        $sheet->cell_formats = array_values($header_types);
        $header_row = array_keys($header_types);

        $sheet->file_writer->write('<row collapsed="false" customFormat="false" customHeight="false" hidden="false" ht="12.1" outlineLevel="0" r="' . (1) . '">');
        foreach ($header_row as $k => $v) {
            $this->writeCell($sheet->file_writer, 0, $k, $v, $cell_format = 'header');
        }
        $sheet->file_writer->write('</row>');
        $sheet->header_offset = 1;
        $sheet->row_count++;
        $this->current_sheet = $sheet_name;
    }

    public function writeSheetRow($sheet_name, array $row=null)
    {
        if (is_array($sheet_name) && empty($row))
            return $this->writeSheetCurrentRow($row=$sheet_name);

        if (empty($sheet_name) || empty($row))
            return;

        self::initializeSheet($sheet_name);
        $sheet = &$this->sheets[$sheet_name];
        if (empty($sheet->cell_formats))
        {
            $sheet->cell_formats = array_fill(0, count($row), 'string');
        }

        $sheet->file_writer->write('<row collapsed="false" customFormat="false" customHeight="false" hidden="false" ht="12.1" outlineLevel="0" r="' . ($sheet->row_count + $sheet->header_offset) . '">');
        foreach ($row as $k => $v) {
            $this->writeCell($sheet->file_writer, $sheet->row_count + $sheet->header_offset -1, $k, $v, $sheet->cell_formats[$k]);
        }
        $sheet->file_writer->write('</row>');
        $sheet->row_count++;
        $this->current_sheet = $sheet_name;
    }

    protected function finalizeSheet($sheet_name)
    {
        if (empty($sheet_name) || $this->sheets[$sheet_name]->finalized)
            return;

        $sheet = &$this->sheets[$sheet_name];

        $sheet->file_writer->write(    '</sheetData>');
        $sheet->file_writer->write(    '<printOptions headings="false" gridLines="false" gridLinesSet="true" horizontalCentered="false" verticalCentered="false"/>');
        $sheet->file_writer->write(    '<pageMargins left="0.5" right="0.5" top="1.0" bottom="1.0" header="0.5" footer="0.5"/>');
        $sheet->file_writer->write(    '<pageSetup blackAndWhite="false" cellComments="none" copies="1" draft="false" firstPageNumber="1" fitToHeight="1" fitToWidth="1" horizontalDpi="300" orientation="portrait" pageOrder="downThenOver" paperSize="1" scale="100" useFirstPageNumber="true" usePrinterDefaults="false" verticalDpi="300"/>');
        $sheet->file_writer->write(    '<headerFooter differentFirst="false" differentOddEven="false">');
        $sheet->file_writer->write(        '<oddHeader>&amp;C&amp;&quot;Times New Roman,Regular&quot;&amp;12&amp;A</oddHeader>');
        $sheet->file_writer->write(        '<oddFooter>&amp;C&amp;&quot;Times New Roman,Regular&quot;&amp;12Page &amp;P</oddFooter>');
        $sheet->file_writer->write(    '</headerFooter>');
        $sheet->file_writer->write('</worksheet>');

        $max_cell = self::xlsCell($sheet->row_count - 1, count($sheet->cell_formats) - 1);
        $max_cell_tag = '<dimension ref="A1:' . $max_cell . '"/>';
        $padding_length = $sheet->max_cell_tag_end - $sheet->max_cell_tag_start - strlen($max_cell_tag);
        $sheet->file_writer->fseek($sheet->max_cell_tag_start);
        $sheet->file_writer->write($max_cell_tag.str_repeat(" ", $padding_length));
        $sheet->file_writer->close();
        $sheet->finalized=true;
    }

    public function writeSheet(array $data, $sheet_name='', array $header_types=array() )
    {
        $data = empty($data) ? array(array('')) : $data;
        if (!empty($header_types)) {
            $this->writeSheetHeader($sheet_name, $header_types);
        }
        foreach($data as $i=>$row) {
            $this->writeSheetRow($sheet_name, $row);
        }
        $this->finalizeSheet($sheet_name);
    }

    public function writeSheetHead($row_count, $column_count, $sheet_name='', array $header_types=array() )
    {
        trigger_error ( __FUNCTION__ . " is deprecated and will be removed in future releases.", E_USER_DEPRECATED);
        $this->writeSheetHeader($this->current_sheet=$sheet_name, $header_types);
    }

    public function writeSheetCurrentRow($row) //formerly named writeSheetRow
    {
        trigger_error ( __FUNCTION__ . " is deprecated and will be removed in future releases.", E_USER_DEPRECATED);
        $this->writeSheetRow($this->current_sheet, $row);
    }

    public function writeSheetFooter()
    {
        trigger_error ( __FUNCTION__ . " is deprecated and will be removed in future releases.", E_USER_DEPRECATED);
        $this->finalizeSheet($this->current_sheet);
    }

    protected function writeCell(XLSXWriterBuffererWriter &$file, $row_number, $column_number, $value, $cell_format)
    {
        static $styles = ['string' => 1, 'money' => 2, 'header' => 3];
        $cell = self::xlsCell($row_number, $column_number);
        $s = isset($styles[$cell_format]) ? $styles[$cell_format] : '0';

        if (!is_scalar($value) || $value=='') {
            $file->write('<c r="'.$cell.'" s="'.$s.'"/>');
        }
        elseif (!is_string($value) || $s == 2) {
            $file->write('<c r="' . $cell . '" s="' . $s . '" ><v>' . $value . '</v></c>');
        }
        elseif ($value{0} == '=') {
            $file->write('<c r="'.$cell.'" s="'.$s.'" t="s"><v>'.self::xmlspecialchars($this->setSharedString($value)).'</v></c>');
        }
        elseif ($value !== '') {
            $file->write('<c r="'.$cell.'" s="'.$s.'" t="s"><v>'.self::xmlspecialchars($this->setSharedString($value)).'</v></c>');
        }
    }

    protected function writeStylesXML()
    {
        $temporary_filename = $this->tempFilename();
        $file = new XLSXWriterBuffererWriter($temporary_filename);
        $file->write('<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'."\n");
        $file->write('<styleSheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">');
        $file->write('<numFmts count="2">');
        $file->write(		'<numFmt formatCode="GENERAL" numFmtId="164"/>');
        //$file->write(		'<numFmt formatCode="[$$-1009]#,##0.00;[RED]\-[$$-1009]#,##0.00" numFmtId="165"/>');
        $file->write(		'<numFmt numFmtId="167" formatCode="#,##0.00;" />');
        $file->write('</numFmts>');
        $file->write('<fonts count="4">');
        $file->write(		'<font><name val="Arial"/><charset val="1"/><family val="2"/><sz val="10"/></font>');
        $file->write(		'<font><name val="Arial"/><family val="0"/><sz val="10"/></font>');
        $file->write(		'<font><name val="Arial"/><family val="0"/><sz val="10"/></font>');
        $file->write(		'<font><name val="Arial"/><family val="0"/><sz val="10"/></font>');
        $file->write(		'<font><b val="true"/><name val="Arial"/><family val="0"/><sz val="10"/></font>');
        $file->write('</fonts>');
        $file->write('<fills count="3">');
        $file->write('    <fill><patternFill patternType="none"/></fill>');
        $file->write('    <fill><patternFill patternType="gray125"/></fill>');
        $file->write('    <fill><patternFill patternType="solid"><fgColor rgb="FF' . $this->styles['color_header'] .'"/><bgColor rgb="FF000000"/></patternFill></fill>');
        $file->write('</fills>');
        $file->write('<borders count="2">');
        $file->write('    <border><left /><right /><top /><bottom /><diagonal /></border>');
        $file->write('        <border><left style="thin"><color indexed="64" /></left><right style="thin"><color indexed="64" /></right><top style="thin"><color indexed="64" /></top><bottom style="thin"><color indexed="64" /></bottom><diagonal /></border>');
        $file->write('</borders>');
        $file->write(	'<cellStyleXfs count="20">');
        $file->write(		'<xf applyAlignment="true" applyBorder="true" applyFont="true" applyProtection="true" borderId="0" fillId="0" fontId="0" numFmtId="164">');
        $file->write(		'<alignment horizontal="general" indent="0" shrinkToFit="false" textRotation="0" vertical="bottom" wrapText="false"/>');
        $file->write(		'<protection hidden="false" locked="true"/>');
        $file->write(		'</xf>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="4" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="2" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="2" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="43"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="41"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="44"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="42"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="true" applyProtection="false" borderId="0" fillId="0" fontId="1" numFmtId="9"/>');
        $file->write(	'</cellStyleXfs>');
        $file->write(	'<cellXfs count="5">');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="false" applyProtection="false" borderId="0" fillId="0" fontId="0" numFmtId="165" xfId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="false" applyProtection="false" borderId="' . $this->getBorder() . '" fillId="0" fontId="0" numFmtId="164" xfId="0"/>');
        $file->write(		'<xf applyNumberFormat="1" applyAlignment="false" applyBorder="false" applyFont="false" applyProtection="false" borderId="' . $this->getBorder() . '" fillId="0" fontId="0" numFmtId="167" xfId="0"/>');
        $file->write(		'<xf applyAlignment="false" applyBorder="false" applyFont="false" applyProtection="false" borderId="' . $this->getBorder() . '" fillId="2" fontId="4" numFmtId="168" xfId="0"/>');
        $file->write(	'</cellXfs>');
        $file->write(	'<cellStyles count="6">');
        $file->write(		'<cellStyle builtinId="0" customBuiltin="false" name="Normal" xfId="0"/>');
        $file->write(		'<cellStyle builtinId="3" customBuiltin="false" name="Comma" xfId="15"/>');
        $file->write(		'<cellStyle builtinId="6" customBuiltin="false" name="Comma [0]" xfId="16"/>');
        $file->write(		'<cellStyle builtinId="4" customBuiltin="false" name="Currency" xfId="17"/>');
        $file->write(		'<cellStyle builtinId="7" customBuiltin="false" name="Currency [0]" xfId="18"/>');
        $file->write(		'<cellStyle builtinId="5" customBuiltin="false" name="Percent" xfId="19"/>');
        $file->write(	'</cellStyles>');
        $file->write('</styleSheet>');
        $file->close();

        return $temporary_filename;
    }

    protected function setSharedString($v)
    {
        if (isset($this->shared_strings[$v])) {
            $string_value = $this->shared_strings[$v];
        }
        else {
            $string_value = count($this->shared_strings);
            $this->shared_strings[$v] = $string_value;
        }
        $this->shared_string_count++;
        return $string_value;
    }

    protected function writeSharedStringsXML()
    {
        $temporary_filename = $this->tempFilename();
        $file = new XLSXWriterBuffererWriter($temporary_filename, $fd_flags='w', $check_utf8=true);
        $file->write('<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'."\n");
        $file->write('<sst count="'.($this->shared_string_count).'" uniqueCount="'.count($this->shared_strings).'" xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">');
        foreach($this->shared_strings as $s=>$c) {
            $file->write('<si><t>'.self::xmlspecialchars($s).'</t></si>');
        }
        $file->write('</sst>');
        $file->close();

        return $temporary_filename;
    }

    protected function buildAppXML()
    {
        $app_xml="";
        $app_xml.='<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'."\n";
        $app_xml.='<Properties xmlns="http://schemas.openxmlformats.org/officeDocument/2006/extended-properties" xmlns:vt="http://schemas.openxmlformats.org/officeDocument/2006/docPropsVTypes"><TotalTime>0</TotalTime></Properties>';
        return $app_xml;
    }

    protected function buildCoreXML()
    {
        $core_xml="";
        $core_xml.='<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'."\n";
        $core_xml.='<cp:coreProperties xmlns:cp="http://schemas.openxmlformats.org/package/2006/metadata/core-properties" xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:dcmitype="http://purl.org/dc/dcmitype/" xmlns:dcterms="http://purl.org/dc/terms/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
        $core_xml.='<dcterms:created xsi:type="dcterms:W3CDTF">'.date("Y-m-d\TH:i:s.00\Z").'</dcterms:created>';
        $core_xml.='<dc:creator>'.self::xmlspecialchars($this->author).'</dc:creator>';
        $core_xml.='<cp:revision>0</cp:revision>';
        $core_xml.='</cp:coreProperties>';
        return $core_xml;
    }

    protected function buildRelationshipsXML()
    {
        $rels_xml="";
        $rels_xml.='<?xml version="1.0" encoding="UTF-8"?>'."\n";
        $rels_xml.='<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">';
        $rels_xml.='<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>';
        $rels_xml.='<Relationship Id="rId2" Type="http://schemas.openxmlformats.org/package/2006/relationships/metadata/core-properties" Target="docProps/core.xml"/>';
        $rels_xml.='<Relationship Id="rId3" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/extended-properties" Target="docProps/app.xml"/>';
        $rels_xml.="\n";
        $rels_xml.='</Relationships>';
        return $rels_xml;
    }

    protected function buildWorkbookXML()
    {
        $workbook_xml="";
        $workbook_xml.='<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'."\n";
        $workbook_xml.='<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">';
        $workbook_xml.='<fileVersion appName="Calc"/><workbookPr backupFile="false" showObjects="all" date1904="false"/><workbookProtection/>';
        $workbook_xml.='<bookViews><workbookView activeTab="0" firstSheet="0" showHorizontalScroll="true" showSheetTabs="true" showVerticalScroll="true" tabRatio="212" windowHeight="8192" windowWidth="16384" xWindow="0" yWindow="0"/></bookViews>';
        $workbook_xml.='<sheets>';
        foreach($this->sheets as $i=>$sheet) {
            $workbook_xml.='<sheet name="'.self::xmlspecialchars($sheet->sheetname).'" sheetId="'.($i+1).'" state="visible" r:id="rId'.($i+2).'"/>';
        }
        $workbook_xml.='</sheets>';
        $workbook_xml.='<calcPr iterateCount="100" refMode="A1" iterate="false" iterateDelta="0.001"/></workbook>';
        return $workbook_xml;
    }

    protected function buildWorkbookRelsXML()
    {
        $wkbkrels_xml="";
        $wkbkrels_xml.='<?xml version="1.0" encoding="UTF-8"?>'."\n";
        $wkbkrels_xml.='<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">';
        $wkbkrels_xml.='<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/styles" Target="styles.xml"/>';
        foreach($this->sheets as $i=>$sheet) {
            $wkbkrels_xml.='<Relationship Id="rId'.($i+2).'" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/'.($sheet->xmlname).'"/>';
        }
        if (!empty($this->shared_strings)) {
            $wkbkrels_xml.='<Relationship Id="rId'.(count($this->sheets)+2).'" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/sharedStrings" Target="sharedStrings.xml"/>';
        }
        $wkbkrels_xml.="\n";
        $wkbkrels_xml.='</Relationships>';
        return $wkbkrels_xml;
    }

    protected function buildContentTypesXML()
    {
        $content_types_xml="";
        $content_types_xml.='<?xml version="1.0" encoding="UTF-8"?>'."\n";
        $content_types_xml.='<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">';
        $content_types_xml.='<Override PartName="/_rels/.rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>';
        $content_types_xml.='<Override PartName="/xl/_rels/workbook.xml.rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>';
        foreach($this->sheets as $i=>$sheet) {
            $content_types_xml.='<Override PartName="/xl/worksheets/'.($sheet->xmlname).'" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>';
        }
        if (!empty($this->shared_strings)) {
            $content_types_xml.='<Override PartName="/xl/sharedStrings.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml"/>';
        }
        $content_types_xml.='<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>';
        $content_types_xml.='<Override PartName="/xl/styles.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml"/>';
        $content_types_xml.='<Override PartName="/docProps/app.xml" ContentType="application/vnd.openxmlformats-officedocument.extended-properties+xml"/>';
        $content_types_xml.='<Override PartName="/docProps/core.xml" ContentType="application/vnd.openxmlformats-package.core-properties+xml"/>';
        $content_types_xml.="\n";
        $content_types_xml.='</Types>';
        return $content_types_xml;
    }

    public static function xlsCell($row_number, $column_number)
    {
        $n = $column_number;
        for($r = ""; $n >= 0; $n = intval($n / 26) - 1) {
            $r = chr($n%26 + 0x41) . $r;
        }
        return $r . ($row_number+1);
    }

    public static function sanitize_filename($filename)
    {
        $nonprinting = array_map('chr', range(0,31));
        $invalid_chars = array('<', '>', '?', '"', ':', '|', '\\', '/', '*', '&');
        $all_invalids = array_merge($nonprinting,$invalid_chars);
        return str_replace($all_invalids, "", $filename);
    }

    public static function xmlspecialchars($val)
    {
        return str_replace("'", "&#39;", htmlspecialchars($val));
    }

    public static function array_first_key(array $arr)
    {
        reset($arr);
        $first_key = key($arr);
        return $first_key;
    }

    public function setStyleCol(array $style)
    {
        $this->style_cols = '';
        $max = 1;
        $min = 1;
        foreach($style as $st) {
            $this->style_cols .= "<col collapsed=\"false\" hidden=\"false\" max=\"{$max}\" min=\"{$min}\" style=\"0\" width=\"{$st['width']}\"/>";
            $min++;
            $max++;
        }
    }

    public function setFilename($name)
    {
        $this->filename = $name;
    }

    public function setColorHeader($color)
    {
        $this->styles['color_header'] = $color;
    }

    public function setBorder($border)
    {
        $this->styles['color_header'] = (bool)$border;
    }

    public function getBorder()
    {
        return (int)$this->styles['border'];
    }

}
