<?php
namespace Halk\Core\Cache;

use Halk\Core\Config;
use Halk\Core\Exception\CoreException;

class Memcache implements Cacheable
{

    protected $host;
    protected $port;
    protected $salt;
    protected $conn;

    protected static $instance;
    protected $memcache;

    /**
     * @return Memcache
     */
    public static function getInstance()
    {
        if (!self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __clone()
    {

    }

    private function __construct()
    {
        $ini = Config::getInstance();

        if ($ini->containsKey('memcache_host')) {
            $host = $ini->getValue('memcache_host');
            $port = $ini->getValue('memcache_port');
            $salt = null;

            if ($ini->containsKey('memcache_salt')) {
                $salt = $ini->getValue('memcache_salt');
            }

            $this->host = $host;
            $this->port = $port;
            $this->salt = $salt;

            $this->memcache = new \Memcache;
            $this->connect($host, $port);
        } else {
            throw new CoreException('No Memcache credentials');
        }
    }

    public function __destruct()
    {
        $this->disconnect();
    }

    protected function connect($host, $port)
    {
        $connect = null;

        if (!$connect = $this->memcache->connect($host, $port)) {
            $this->memcache = null;
            throw new CoreException(sprintf('Failed to connect to memcached at %s:%s', $host, $port));
        }

        $this->conn = $connect;
    }

    public function disconnect()
    {
        if ($this->memcache) {
            $this->memcache->close();
        }
    }

    public function get($key)
    {
        if (!$this->memcache) {
            return null;
        }
        $key = $this->salt . $key;

        return $this->memcache->get($key);
    }

    public function delete($key)
    {
        if (!$this->memcache) {
            return null;
        }
        $key = $this->salt . $key;
        $this->memcache->delete($key);
    }

    public function set($key, $value, $expire = 0)
    {
        if (!$this->memcache) {
            return null;
        }
        $key = $this->salt . $key;

        return $this->memcache->set($key, $value, 0, $expire);
    }

    public function exists($key)
    {
        return count(array_filter($this->getKeyList(), function ($k) use ($key) {
            return $k === $key;
        })) > 0;
    }

    public function getKeyList()
    {
        $list = array();
        $allSlabs = $this->memcache->getExtendedStats('slabs');
        foreach ($allSlabs as $server => $slabs) {
            foreach ($slabs as $slabId => $slabMeta) {
                $cdump = @$this->memcache->getExtendedStats('cachedump', (int)$slabId);
                foreach ($cdump as $keys => $arrVal) {
                    if (!is_array($arrVal)) {
                        continue;
                    }
                    foreach ($arrVal as $k => $v) {
                        $list[] = $k;
                    }
                }
            }
        }

        return $list;
    }

    public function deleteKeyByMask($mask)
    {
        $list = $this->getKeyList();
        foreach ($list as $key) {
            if (strpos($key, $mask) !== false) {
                $this->memcache->connect($this->host, $this->port);
                $this->memcache->delete($key);
            }
        }
    }

    public function isConnected()
    {
        return !!$this->conn;
    }
}
