<?php

/**
 * @namespace Halk\Core\File
 */
namespace Halk\Core\File;

/**
 * Validator
 * Класс, предоставляющий интерфейс, для валидации файлов.
 *
 * @package core
 * @subpackage file
 * @author Mikhail Levykin
 *
 */
class Validator extends ErrorsRegistry
{

    /**
     * запрошенный валидатор не существует
     *
     * @var String
     */
    const BAD_VALIDATOR = 'BadValidator';

    /**
     * ключ не пройдена валидация IsImage
     *
     * @var String
     */
    const IS_IMAGE_ERROR = 'IsImageError';

    /**
     * не пройдена валидация по допустимым расширения
     *
     * @var String
     */
    const EXT_ALLOW_ERROR = 'ExtAllowError';

    /**
     * не пройдена валидация по mime-type
     *
     * @var String
     */
    const TYPE_ERROR = 'TypeError';

    /**
     * размер файла превышает допустимый
     *
     * @var String
     */
    const MAX_SIZE_ERROR = 'MaxSizeError';

    /**
     * файл не был загружен по POST
     *
     * @var String
     */
    const IS_UPLOADED_ERROR = 'IsUploadedError';

    /**
     * файл пустой
     *
     * @var String
     */
    const IS_EMPTY = 'IsEmpty';

    /**
     * массив всех шаблонов сообщений валидаторов о том, что валидация не пройдена
     *
     * @var Array
     * @access protected
     */
    protected $messagesTemplates = array(
        self::BAD_VALIDATOR => 'Валидатор "%s" не существует!',
        self::IS_IMAGE_ERROR => 'Файл "%s" не является изображением! Тип файла - "%s"',
        self::EXT_ALLOW_ERROR => 'Файл "%s" имеет недопустимое расширение "%s". Допустимые типы файлов: "%s"',
        self::TYPE_ERROR => 'Файл "%s" имеет недопустимый тип(%s)!',
        self::MAX_SIZE_ERROR => 'Размер файла "%s" превышает допустимый (%s Mb)',
        self::IS_UPLOADED_ERROR => 'Файл "%s" не был загружен',
        self::IS_EMPTY => 'Файл "%s" пустой!'
    );

    /**
     * Валидируемый файл
     *
     * @var FileObject
     * @access protected
     */
    protected $file;

    /**
     * Имена всех доступных методов - валидаторов
     *
     * @var Array
     * @access protected
     */
    protected $validators = array(
        'isImage',
        'extensionsAllowed',
        'extensionsNotAllowed',
        'type',
        'size',
        'isNotEmpty',
        'isUploaded'
    );

    /**
     * Конструктор класса.
     *
     * @param FileObject $file Валидируемый файл
     */
    public function __construct(FileObject $file)
    {
        $this->file = $file;
    }

    /**
     * Возвращает имя валидируемого файла
     *
     * @return String
     */
    protected function getFilename()
    {
        if($this->file->getRealName() != null) {
            return $this->file->getRealName();
        } else {
            return $this->file->getBasename();
        }
    }

    /**
     * Позволяет установить шаблоны сообщений
     *
     * @param array $templates
     * @access public
     * @return void
     */
    public function setMessagesTemplates(array $templates)
    {
        $this->messagesTemplates = array_merge($this->messagesTemplates, $templates);
    }

    /**
     * Возвращает массив с именами всех доступных валидаторов
     *
     * @return Array
     * @access public
     */
    public function getValidators()
    {
        return $this->validators;
    }

    /**
     * Проверяет, является - ли загружаемый файл изображением
     *
     * @param Array $image_types
     * @throws UploadException
     * @return Boolean
     */
    public function isImage($image_types = array())
    {
        $default_types = array(
                'image/gif',
                'image/jpg',
                'image/jpeg',
                'image/pjpeg',
                'image/png',
                'image/tiff'
        );

        $mime_type = $this->file->getMimeType();
        $allowed_types = array_merge($default_types, (array) $image_types);

        if(! in_array($mime_type, $allowed_types)) {
            $filename = $this->getFilename();
            $msg = $this->messagesTemplates[self::IS_IMAGE_ERROR];

            $this->registerError($msg, $filename, $mime_type);

            return false;
        }

        return true;
    }

    /**
     * Производит валидацию загружаемого файла по расширению,
     * среди допустимых расширений.
     *
     * @param array $extensions
     * @return boolean
     */
    public function extensionsAllowed(array $extensions)
    {
        $file_extension = strtolower($this->file->getFileExtension(true));
        $allowed_extensions = array_map(
            function ($ext)
            {
                return strtolower($ext);
            }, $extensions);

        if(! in_array($file_extension, $allowed_extensions)) {
            $filename = $this->getFilename();
            $msg = $this->messagesTemplates[self::EXT_ALLOW_ERROR];

            $this->registerError($msg, $filename, $file_extension,
                implode(', ', $extensions));

            return false;
        }

        return true;
    }

    /**
     * Производит валидацию загружаемого файла по расширению,
     * производя поиск среди НЕ допустимых расширений.
     *
     * @param array $extensions
     * @return boolean
     */
    public function extensionsNotAllowed(array $extensions)
    {
        $file_extension = $this->file->getFileExtension(true);
        $not_allowed_extensions = array_map(
            function ($ext)
            {
                return strtolower($ext);
            }, $extensions);

        if(in_array($file_extension, $not_allowed_extensions)) {
            $filename = $this->getFilename();
            $msg = $this->messagesTemplates[self::EXT_ALLOW_ERROR];

            $this->registerError($msg, $filename, $file_extension);

            return false;
        }

        return true;
    }

    /**
     * Проверяет загруженный файл по Mime-Type
     *
     * @param Array $allowed_types
     * @return Boolean
     */
    public function type(array $allowed_types)
    {
        $mime_type = $this->file->getMimeType();

        if(! in_array($mime_type, $allowed_types)) {
            $filename = $this->getFilename();
            $msg = $this->messagesTemplates[self::TYPE_ERROR];

            $this->registerError($msg, $filename, $mime_type);

            return false;
        }

        return true;
    }

    /**
     * Производит валидацию по размеру файла.
     *
     * @param Int $max_size максимальный размер файла в мегабайтах
     * @return boolean
     */
    public function size($max_size)
    {
        // размер файла в байтах
        $filesize = $this->file->getSize();
        $max_size_bytes = $max_size * 1024 * 1024;

        if($filesize > $max_size_bytes) {
            $filename = $this->getFilename();
            $msg = $this->messagesTemplates[self::MAX_SIZE_ERROR];

            $this->registerError($msg, $filename, $max_size);

            return false;
        }

        return true;
    }

    /**
     * Производит проверку, что файл не пустой.
     *
     * @return boolean
     */
    public function isNotEmpty()
    {
        // размер файла в байтах
        $filesize = $this->file->getSize();

        if($filesize == 0) {
            $msg = $this->messagesTemplates[self::IS_EMPTY];
            $this->registerError($msg, $this->getFilename());

            return false;
        }

        return true;
    }

    /**
     * Проверяет, был-ли этот файл загружен по POST.
     *
     * @return boolean
     */
    public function isUploaded()
    {
        if($this->file->isUploaded()) {
            return true;
        } else {
            $msg = $this->messagesTemplates[self::IS_UPLOADED_ERROR];
            $this->registerError($msg, $this->getFilename());

            return false;
        }
    }

    /**
     * Позволяет вызывать сразу несколько валидаторов по очереди.
     *
     * @param array $validators Массив, содержащий имена методов-валидаторов
     *     в качестве ключей и параметры валидации - в качестве значений.
     * @throws \Exception
     * @return boolean
     */
    /*
     $validator->validateQueue(array(
         'IsNotEmpty' => null,
         'Size' => $this->max_filesize,
         'Type' => array('image/jpeg'),
         'ExtensionsAllowed' => array('pdf', 'jpeg', 'jpg',
             'png', 'txt', 'xml', 'doc', 'xls', 'csv')));
     */
    public function validateQueue(array $validators)
    {
        $result = true;

        foreach($validators as $method => $params) {

            if(! in_array($method, $this->getValidators())) {
                $msg = $this->messagesTemplates[self::BAD_VALIDATOR];
                throw new \Exception(sprintf($msg, $method));
            }

            if(!$this->$method($params)) {
                $result = false;
            }
        }

        if(count($this->errors) > 0)
            $result = false;

        return $result;
    }

}

/* Возможные типы, для проверки по mime **
 application/pdf
application/xhtml+xml
application/zip
application/x-gzip

image/gif
image/jpeg
image/pjpeg
image/png
image/svg+xml
image/tiff

text/css: CSS
text/csv: CSV
text/html: HTML
text/plain: текстовые данные
text/xml: XML

application/vnd.oasis.opendocument.text: OpenDocument
application/vnd.oasis.opendocument.spreadsheet: OpenDocument
application/vnd.oasis.opendocument.presentation: OpenDocument
application/vnd.oasis.opendocument.graphics: OpenDocument
application/vnd.ms-excel: Microsoft Excel файлы
application/vnd.openxmlformats-officedocument.spreadsheetml.sheet: Microsoft Excel 2007 файлы
application/vnd.ms-powerpoint: Microsoft Powerpoint файлы
application/vnd.openxmlformats-officedocument.presentationml.presentation: Microsoft Powerpoint 2007 файлы
application/msword: Microsoft Word файлы
application/vnd.openxmlformats-officedocument.wordprocessingml.document: Microsoft Word 2007 файлы

application/x-rar-compressed: RAR
application/x-tar: Tarball
*/
