<?php

namespace Halk\Core\Helper;

use DateTime;

class HalkDate extends DateTime
{
    const
        FORMAT_HUMAN = 'd-m-Y',
        TYPE_TIMESTAMP = 'timestamp',
        TYPE_POSIX = 'posix';

    public static function dayInPeriod($start, $end, $prc = 2)
    {
        $d1 = new DateTime($start);
        $d2 = new DateTime($end);
        $interval = $d2->diff($d1);
        $d = round(($interval->h * 3600 + $interval->i * 60 + $interval->s) / 86400, $prc);
        return intval($interval->format('%a')) + $d;
    }

    public static function addDay($stamp, $days)
    {
        $seconds = $days * 86400;
        $d = new DateTime($stamp);
        $d->modify("+$seconds second");
        return $d->format('Y-m-d H:i:s');
    }

    public static function addYear($stamp, $year)
    {
        $d = new DateTime($stamp);
        $d->modify("+$year year");
        return $d->format('Y-m-d H:i:s');
    }

    public static function addSec($stamp, $seconds=1)
    {
        $d = new DateTime($stamp ?? 'now');
        $d->modify("+$seconds second");
        return $d->format('Y-m-d H:i:s');
    }

    public static function delSec($stamp, $seconds=1)
    {
        $d = new DateTime($stamp);
        $d->modify("-$seconds second");
        return $d->format('Y-m-d H:i:s');
    }

    public static function addMonth($stamp, $month)
    {
        $stamp = strtotime($stamp);
        $begin_month = strtotime(date('Y-m-01 00:00:00', $stamp));
        $start_total_seconds = date('t', $stamp) * 86400;
        $usage = ($stamp - $begin_month) / $start_total_seconds;
        $result = strtotime('+ ' . floor($month) . ' month', $begin_month);
        $left_diff = $usage*date('t', $result)*86400;
        $result += $left_diff;

        $part = ($month - floor($month));
        $result_month_seconds = date('t', $result) * 86400;
        $month_offset_seconds = $part*$result_month_seconds;
        $end_month_timestamp = strtotime(date('Y-m-t 23:59:59', $stamp));
        $month_stay_seconds = ($end_month_timestamp - $stamp) + 1;

        if ($month_offset_seconds > $month_stay_seconds) {
            $result = strtotime('+ 1 month', strtotime(date('Y-m-01 00:00:00', $result)));
            $unused_period = ($month_offset_seconds - $month_stay_seconds)/$result_month_seconds;
            $result += $unused_period*(date('t', $result)*86400);
        } else {
            $result += $month_offset_seconds;
        }

        return date('Y-m-d H:i:s', round($result));
    }

    public static function delMonth($stamp, $month)
    {
        // ВСЕГДА ПРИБАВЛЯЕМ ЧЕРЕЗ ДЕЛЕНИЕ МЕСЯЦА на 100

        // определим, сколько осталось до конца месяца
        $stay_days = HalkDate::dayLeftInMonth($stamp);      // к примеру 28,28
        $day_in_mounth = HalkDate::daysInMonth($stamp);     // к прмиеру 31
        $day_use = round($stay_days / $day_in_mounth, 2);     // столько еще до конца месяца

        $month = $month - (1 - $day_use);
        $int = floor($month); // целая часть для вычитания
        $end = $month - $int; // дробная часть для вычитания
        // возмем начало месяца
        $begin_date = self::getBeginMonth($stamp);
        $d = new DateTime($begin_date);
        $d->modify("-$int month");

        if ($end) {
            $d->modify("-1 second"); // т.к. отнимаем - нам важно знать кол-во дней в предыдущем месяце !
            $day_in_mounth = HalkDate::daysInMonth($d->format('Y-m-d'));
            // для точности работаем с секундами
            $seconds = $day_in_mounth * 24 * 60 * 60;
            $seconds_add = round($seconds * $end);
            $d->modify("-$seconds_add second");
        }

        return $d->format('Y-m-d H:i:s');
    }

    public static function dayLeftInMonth($stamp, $prc = 2)
    {
        $d1 = new DateTime($stamp);
        $d2 = new DateTime($stamp);
        $d2->modify("first day of next month midnight");
        $d2->modify("-1 second");
        $interval = $d2->diff($d1);
        $d = round(($interval->h * 3600 + $interval->i * 60 + $interval->s) / 86400, $prc);
        $r = intval($interval->format('%a')) + $d;

        return $r;
    }

    public static function daysInMonth($stamp)
    {
        $d = new DateTime($stamp);

        return $d->format('t');
    }

    public static function getBeginMonth($now = '', $format = '')
    {
        if (!$now) {
            $now = self::Now();
        }
        if ($format == 'Y-m-d') {
            return self::strtotime($now, 'Y-m') . "-01";
        } else {
            return self::strtotime($now, 'Y-m') . "-01 00:00:00";
        }

    }

    public static function Now($full = true)
    {
        $d = new DateTime('now');

        return $full ? $d->format('Y-m-d H:i:s') : $d->format('Y-m-d');
    }

    public static function strtotime($str, $format = 'Y-m-d H:i:s')
    {
        $d = new DateTime($str);

        return $d->format($format);
    }

    public static function getEndMonth($stamp)
    {
        $d = new DateTime($stamp);
        $d->modify("first day of next month midnight");
        $d->modify("-1 second");

        return $d->format('Y-m-d H:i:s');
    }

    public static function getTimeEndPrevDay($stamp)
    {
        $d = new DateTime($stamp);
        $d->modify("-1 day");

        return $d->format('Y-m-d') . ' 23:59:59';
    }

    public static function parseStamp($stamp)
    {
        /*
         * функция возвращает время как YYYY-MM-DD HH:mm:ss или то чего пришло на вход
         * на входе ловим варианты
         * 2015-07-01 23:45:34(YYYY-MM-DD)
         * 2015-07-01 23:45:34(YYYY-MM-DD HH:mm)
         * 2015-07-01 23:45:34(YYYY-MM-DD HH:mm:ss)
         * 01-07-2015 23:45:34(DD-MM-YYYY)
         * 01-07-2015 23:45:34(DD-MM-YYYY HH:mm)
         * 01-07-2015 23:45:34(DD-MM-YYYY HH:mm:ss)
         * 01.07.2015 23:45:34(DD.MM.YYYY)
         * 01.07.2015 23:45:34(DD.MM.YYYY HH:mm)
         * 01.07.2015 23:45:34(DD.MM.YYYY HH:mm:ss)
         */
        if (preg_match("/^([0-9]{2})-([0-9]{2})-([0-9]{4})( (.*))?$/", $stamp, $m)) {
            if (!isset($m[5])) {
                $m[4] = '00:00:00';
            } elseif (preg_match('/^[0-9]{2}:[0-9]{2}$/', $m[5])) {
                $m[5] .= ':00';
                $m[4] = $m[5];
            } elseif (isset($m[5])) {
                $m[4] = $m[5];
            }
            return $m[3] . '-' . $m[2] . '-' . $m[1] . ' ' . $m[4];
        } elseif (preg_match("/^([0-9]{2})\.([0-9]{2})\.([0-9]{4})( (.*))?$/", $stamp, $m)) {
            if (!isset($m[5])) {
                $m[4] = '00:00:00';
            } elseif (preg_match('/^[0-9]{2}:[0-9]{2}$/', $m[5])) {
                $m[5] .= ':00';
                $m[4] = $m[5];
            } elseif (isset($m[5])) {
                $m[4] = $m[5];
            }
            return $m[3] . '-' . $m[2] . '-' . $m[1] . ' ' . $m[4];
        } elseif (preg_match("/^([0-9]{4})-([0-9]{2})-([0-9]{2})( (.*))?$/", $stamp, $m)) {
            if (!isset($m[5])) {
                $m[4] = '00:00:00';
            } elseif (preg_match('/^[0-9]{2}:[0-9]{2}$/', $m[5])) {
                $m[5] .= ':00';
                $m[4] = $m[5];
            } elseif (isset($m[5])) {
                $m[4] = $m[5];
            }
            return $m[1] . '-' . $m[2] . '-' . $m[3] . ' ' . $m[4];
        } else {
            return $stamp;
        }
    }

    /**
     * Сравнение 2 дат
     */
    public static function isOlder($start, $end)
    {
        $d1 = new DateTime($start);
        $d2 = new DateTime($end);

        // PHP 5.2.2 version
        if ($d2 > $d1) {
            return 1;
        } elseif ($d2 < $d1) {
            return -1;
        }

        return 0;
    }

    public static function humanFormat($date)
    {
        return static::getByFormat($date, static::FORMAT_HUMAN);
    }

    /**
     * @param string|int $date текстовый формат или posix timestamp
     * @param bool|false $format
     * @return bool|string
     */
    public static function getByFormat($date, $format = false)
    {
        $is_text = preg_match('/[^0-9]+/u', (string)$date);
        return date(
            (($format !== false)? $format: static::FORMAT_HUMAN),
            (($is_text)? strtotime($date): $date)
        );
    }

    /**
     * @param $quarter
     * @param $year
     * @return array
     */
    public static function getQuarter($quarter, $year)
    {
        $m = $quarter * 3;
        switch($m) {
            case $m >= 1 && $m <= 3:
                $start = $year.'-01-01 00:00:00';
                $end = $year.'-03-31 23:59:59';
                break;
            case $m >= 4 && $m <= 6:
                $start = $year.'-04-01 00:00:00';
                $end = $year.'-06-30 23:59:59';
                break;
            case $m >= 7 && $m <= 9:
                $start = $year.'-07-01 00:00:00';
                $end = $year.'-09-30 23:59:59';
                break;
            case $m >= 10 && $m <= 12:
                $start = $year.'-10-01 00:00:00';
                $end = $year.'-12-31 23:59:59';
                break;
        }
        return array(
            'start' => $start,
            'end' => $end,
            'start_nix' => strtotime($start),
            'end_nix' => strtotime($end)
        );
    }

    /**
     * @param $num_month
     * @param bool $genitive - возвратить в род. п.
     * @return string
     */
    public static function getRussianMonth($num_month, $genitive = false)
    {
        if ($genitive) {
            $months =  [
                1  => 'Января',
                2  => 'Февраля',
                3  => 'Марта',
                4  => 'Апреля',
                5  => 'Мая',
                6  => 'Июня',
                7  => 'Июля',
                8  => 'Августа',
                9  => 'Сентября',
                10 => 'Октября',
                11 => 'Ноября',
                12 => 'Декабря'
            ];
        } else {
            $months =  [
                1  => 'Январь',
                2  => 'Февраль',
                3  => 'Март',
                4  => 'Апрель',
                5  => 'Май',
                6  => 'Июнь',
                7  => 'Июль',
                8  => 'Август',
                9  => 'Сентябрь',
                10 => 'Октябрь',
                11 => 'Ноябрь',
                12 => 'Декабрь'
            ];
        }

        return isset($months[$num_month]) ? $months[$num_month] : '';
    }

    /**
     * @param string $left_border
     * @param string $right_border
     * @param string $current_time
     * @return bool входит ли указанная дата в период
     */
    public static function inPeriod($left_border, $right_border, $current_time = null)
    {
        if (is_null($current_time)) {
            $current_time = new DateTime();
        }
        $left_border = new DateTime($left_border);
        $right_border = new DateTime($right_border);
        return $current_time >= $left_border && $current_time <= $right_border;
    }

    /**
     * Форматирует полученные минуты в массив с днями, часами и минутами
     * @param $minutes
     * @return mixed
     */
    public static function minToArray($minutes){
        $h = floor($minutes / 60);

        $result['all_min'] = $minutes;
        $result['days'] = floor($h / 24);
        $result['hours'] = $h - ($result['days'] * 24);
        $result['minutes'] = $minutes - ($h * 60);

        return $result;
    }
}
