<?php
namespace Filanco;

use Filanco\Whois\Exception\ConnectionErrorException;
use Filanco\Whois\Exception\WhoisServerForTldNotFoundException;
use TrueBV\Punycode;

/**
 * Новый класс для работы с whois
 * Создан, тк воозникла необходимость в правке исходного класса.
 */
class Whois
{
    /** @var string */
    protected $domain;

    /** @var string */
    protected $sld;

    /** @var string */
    protected $tld;
    
    /** @var array */
    protected $servers;

    /**
     * @param string $domain full domain name (without trailing dot)
     * @param string $base_file_path
     */
    public function __construct($domain, $base_file_path = null)
    {
        $puny_code = new Punycode();
        $this->domain = $puny_code->encode(rtrim(trim($domain), '.'));
        list($this->sld, $this->tld) = array_pad(explode('.', $this->domain, 2), 2, null);
        $base_file_path = $base_file_path?:__DIR__ . '/../whois.servers.json';
        $this->servers = json_decode(file_get_contents($base_file_path), true);
    }

    public function info()
    {
        if (empty($this->servers[$this->tld][0])) {
            throw new WhoisServerForTldNotFoundException($this->tld);
        }
        /** @var string $whois_server */
        $whois_server = $this->servers[$this->tld][0];

        if (preg_match("/^https?:\/\//i", $whois_server)) {
            $ch = curl_init();
            $url = $whois_server . $this->domain;
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);

            $data = curl_exec($ch);

            if (curl_error($ch)) {
                throw new ConnectionErrorException();
            } else {
                $string = strip_tags($data);
            }
            curl_close($ch);

        } else {
            // Getting whois information
            $fp = fsockopen($whois_server, 43);
            if (!$fp) {
                throw new ConnectionErrorException();
            }

            fputs($fp, $this->domain . "\r\n");

            // Getting string
            $string = '';

            // Checking whois server for .com and .net
            if ($this->tld == 'com' || $this->tld == 'net') {
                while (!feof($fp)) {
                    $line = trim(fgets($fp, 128));

                    $string .= $line;

                    $lineArr = explode(":", $line);

                    if (strtolower($lineArr[0]) == 'whois server') {
                        $whois_server = trim($lineArr[1]);
                    }
                }
                // Getting whois information
                $fp = fsockopen($whois_server, 43);
                if (!$fp) {
                    throw new ConnectionErrorException();
                }

                fputs($fp, $this->domain . "\r\n");

                // Getting string
                $string = '';

                while (!feof($fp)) {
                    $string .= fgets($fp, 128);
                }

                // Checking for other tld's
            } else {
                while (!feof($fp)) {
                    $string .= fgets($fp, 128);
                }
            }
            fclose($fp);
        }

        return htmlspecialchars($string);
    }

    public function isAvailable()
    {
        $whois_string = $this->info();
        $not_found_string = '';
        if (isset($this->servers[$this->tld][1])) {
            $not_found_string = $this->servers[$this->tld][1];
        }

        $whois_string2 = @preg_replace('/' . $this->domain . '/', '', $whois_string);
        $whois_string = @preg_replace("/\s+/", ' ', $whois_string);

        $array = explode(":", $not_found_string);
        if ($array[0] == "MAXCHARS") {
            if (strlen($whois_string2) <= $array[1]) {
                return true;
            } else {
                return false;
            }
        } else {
            if (preg_match("/" . $not_found_string . "/i", $whois_string)) {
                return true;
            } else {
                return false;
            }
        }
    }

    public function htmlInfo()
    {
        return nl2br($this->info());
    }

    /**
     * @return string full domain name
     */
    public function getDomain()
    {
        return $this->domain;
    }
}
