<?php
namespace Filanco;

use Filanco\Whois\Exception\ConnectionErrorException;
use Filanco\Whois\Exception\WhoisServerForTldNotFoundException;
use TrueBV\Punycode;

/**
 * Новый класс для работы с whois
 * Создан, тк воозникла необходимость в правке исходного класса.
 */
class Whois
{
    /** @var string */
    protected $domain;

    /** @var string */
    protected $sld;

    /** @var string */
    protected $tld;

    /** @var array */
    protected $servers;

    /** @var array */
    protected $expKeys;

    /** @var string */
    protected $data;

    /**
     * Whois constructor.
     * @param string $domain
     * @param null|string $base_file_path
     * @throws ConnectionErrorException
     * @throws WhoisServerForTldNotFoundException
     */
    public function __construct($domain, $base_file_path = null)
    {
        // получаем данные для дальнейшей работы
        $puny_code = new Punycode();
        $this->domain = $puny_code->encode(rtrim(trim($domain), '.'));
        list($this->sld, $this->tld) = array_pad(explode('.', $this->domain, 2), 2, null);
        $base_file_path = $base_file_path?:__DIR__ . '/../whois.servers.json';
        $this->servers = json_decode(file_get_contents($base_file_path), true);
        $this->expKeys = json_decode(file_get_contents(__DIR__ . '/../whois.expiration.json'), true)['keys'];

        // получаем данные whois
        $this->data = $this->getDataFromServers();
    }

    /**
     * Получаем данные whois с сервера
     *
     * @return string
     * @throws ConnectionErrorException
     * @throws WhoisServerForTldNotFoundException
     */
    protected function getDataFromServers()
    {
        if (empty($this->servers[$this->tld][0])) {
            throw new WhoisServerForTldNotFoundException($this->domain);
        }
        /** @var string $whois_server */
        $whois_server = $this->servers[$this->tld][0];

        if (preg_match("/^https?:\/\//i", $whois_server)) {
            $ch = curl_init();
            $url = $whois_server . $this->domain;
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);

            $data = curl_exec($ch);

            if (curl_error($ch)) {
                throw new ConnectionErrorException($this->tld, $whois_server);
            } else {
                $string = strip_tags($data);
            }
            curl_close($ch);

        } else {

            // Getting whois information
            $fsoc_er = null;
            $fsoc_erstr = null;
            $time_out = 10; // устанавливаем 10 сек.
            $fp = @fsockopen($whois_server, 43, $fsoc_er, $fsoc_erstr, $time_out);
            if (!$fp) {
                throw new ConnectionErrorException($this->tld, $whois_server);
            }

            fputs($fp, $this->domain . "\r\n");

            // Getting string
            $string = '';

            // Checking whois server for .com and .net
            if ($this->tld == 'com' || $this->tld == 'net') {
                while (!feof($fp)) {
                    $line = trim(fgets($fp, 128));

                    $string .= $line;

                    $lineArr = explode(":", $line);

                    if (strtolower($lineArr[0]) == 'whois server') {
                        $whois_server = trim($lineArr[1]);
                    }
                }
                // Getting whois information
                $fp = fsockopen($whois_server, 43);
                if (!$fp) {
                    throw new ConnectionErrorException($this->tld, $whois_server);
                }

                fputs($fp, $this->domain . "\r\n");

                // Getting string
                $string = '';

                while (!feof($fp)) {
                    $string .= fgets($fp, 128);
                }

                // Checking for other tld's
            } else {
                while (!feof($fp)) {
                    $string .= fgets($fp, 128);
                }
            }
            fclose($fp);
        }

        return htmlspecialchars($string);
    }

    /*
     * Проверяем доступность домена
     * */
    public function isAvailable()
    {
        $whois_string = $this->data;
        $not_found_string = '';
        if (isset($this->servers[$this->tld][1])) {
            $not_found_string = $this->servers[$this->tld][1];
        }

        $whois_string2 = @preg_replace('/' . $this->domain . '/', '', $whois_string);
        $whois_string = @preg_replace("/\s+/", ' ', $whois_string);

        $array = explode(":", $not_found_string);
        if ($array[0] == "MAXCHARS") {
            if (strlen($whois_string2) <= $array[1]) {
                return true;
            } else {
                return false;
            }
        } else {
            if (preg_match("/" . $not_found_string . "/i", $whois_string)) {
                return true;
            } else {
                return false;
            }
        }
    }

    /*
     * Отдаем данные о домене
     * */
    public function info()
    {
        return $this->data;
    }

    /*
    * Отдаем данные о домене в виде HTML
    * */
    public function htmlInfo()
    {
        return nl2br($this->data);
    }

    /*
    * Отдаем имя домена
    * */
    public function getDomain()
    {
        return $this->domain;
    }

    /*
    * Парсим и отдаем дату окончания регистрации домена
    * */
    public function getExpired()
    {
        // переменные для результата
        $result = false;
        $results = [];

        // получаем данные
        $info = $this->data;

        // разбиваем на строки
        $lines = preg_split('/[\r\n]{1,2}/', $info);

        // все строки
        foreach ($lines as $line) {
            // все ключи
            foreach ($this->expKeys as $key) {
                // обработка
                $key = str_replace(' ', '[\s\t]+', $key);
                // ищем совпадения
                $count = preg_match('/^[\s\t]*' . $key .  '[:\s\t]*([^\s\t]+)/iu', $line, $matches);
                // если совпадения найдены
                if ($count) {
                    // добавляем результаты
                    $results[] = $matches;
                }
            }
        }

        // если результат был найден
        if (count($results)) {
            // выбираем первый результат (в целом он и будет только один единственный)
            $result_parse = array_shift($results)[1];
            // правим странные места в формате даты
            $result_parse = str_replace('/', '.', $result_parse);
            // получаем результат
            $result = date('d.m.Y', strtotime($result_parse));
        }

        return $result;
    }
}
