<?php namespace Halk\Core;

/**
 * Class Profiler
 * Простой профайлер, способный замерять время между контрольными точками,
 * устанавливаемыми при помощи функции pf() и пищущий результат измерений в лог.
 * @package Halk\Core
 */

class Profiler
{
    public static $checkpoints = [];
    public static $enabled;
    public static $filename = '/logs/profiler.log';

    /**
     * Добавляет отметку в массив
     * @param null $label имя метки
     * @param null $compareTo замерять относительно последней отметки с таким именем
     * @return bool
     */
    public static function check($label = null, $compareTo = null)
    {
        self::init();
        if (self::$enabled) {
            empty($label) && $label = 'mark_' . count(self::$checkpoints);
            self::$checkpoints[] = ['label' => $label, 'time' => microtime(true), 'compareTo' => $compareTo];

            return true;
        } else {
            return false;
        }
    }

    /**
     * Вывод данных в лог-файл
     * @return bool|null
     */
    public static function log()
    {
        if (!self::$enabled || empty(self::$checkpoints)) {
            return null;
        }

        $dir = Config::getInstance()->getValue('home');
        $filename = $dir . self::$filename;
        self::rotateLog($filename);
        if (!$file = fopen($filename, 'a')) {
            trigger_error('Не удалось открыть файл' . $filename . ' для записи, профайлинг отключен');
            self::$enabled = false;

            return null;
        }

        fwrite($file, sprintf("\n\n==== Profiler log at %s =====", date(DATE_RSS, intval(HALK_STARTUP_TIME))));
        fwrite($file, sprintf("\n==== Request: %s %s =====", $_SERVER['REQUEST_METHOD'], $_SERVER['QUERY_STRING']));
        fwrite($file, sprintf("\n|Index:Time\t|Relative time\t|Label\t|"));
        $prev = self::$checkpoints[0];
        foreach (self::$checkpoints as $i => $v) {
            if (isset($v['compareTo']) && ($chp = self::findCheckpoint($i, $v['compareTo']))) {
                $diff = $v['time'] - $chp['time'];
                $rel = '(relative to ' . $v['compareTo'] . ')';
            } else {
                $diff = $v['time'] - $prev['time'];
                $rel = '';
            }
            $time = $v['time'] - HALK_STARTUP_TIME;
            $prev = $v;
            $str = sprintf("\n %2d:%5.2f sec\t %6.3f sec\t %s%s", $i, $time, $diff, $v['label'], $rel);
            fwrite($file, $str);
        }
        fwrite($file, sprintf("\n====Log over at %s=====", date(DATE_RSS)));
        fclose($file);

        return true;
    }

    /**
     * Ищем ближайший к указанному индексу чекпоинт с заданным именем
     * @param $index
     * @param $label
     * @return null
     */
    private static function findCheckpoint($index, $label)
    {
        foreach (array_reverse(array_slice(self::$checkpoints, 0, $index, true)) as $i => $v) {
            if ($v['label'] == $label) {
                return $v;
            }
        }

        return null;
    }

    /**
     * Если лог больше заданного размера - переименовываем его в .old
     * @param $filename
     * @return bool
     */
    private function rotateLog($filename)
    {
        if (file_exists($filename) && filesize($filename) > 1048576 * 10) {
            if (!rename($filename, $filename . '.old')) {
                trigger_error('Не удалось пересоздать файл лога профайлера ' . $filename);
            }

            return true;
        }

        return false;
    }

    /**
     * Инициализация
     */
    private function init()
    {
        if (is_null(self::$enabled)) {
            $ini = Config::getInstance();
            if ($ini->containsKey('profiler_enabled') && $ini->getValue('profiler_enabled')) {
                self::$enabled = true;
            } else {
                self::$enabled = false;

            }
        }
    }
}
