<?php namespace Halk\Core;

/**
 * Class Error
 * @package Halk\Core
 */
class Error {

    /**
     * @var bool
     */
    private static $registered = false;

    protected static $levels = [
        E_ERROR,
        E_USER_ERROR,
        E_COMPILE_ERROR
    ];

    /**
     * Register error handlers
     */
    public static function register()
    {
        if (!self::$registered) {
            register_shutdown_function('Halk\Core\Error::fatal');
            set_exception_handler('Halk\Core\Error::exception');
            set_error_handler('Halk\Core\Error::warning');

            if (HALK_RUNNING_MODE == HALK_MODE_CLI) {
                error_reporting(E_ALL);
                ini_set("display_errors", "stderr");
            }
            self::$registered = true;
        }
    }

    public static function fatal()
    {
        if(HALK_RUNNING_MODE == HALK_MODE_CLI) {
            return true;
        }

        $error = error_get_last();
        if (in_array($error['type'], self::$levels))
        {
            if (class_exists('\Halk\Core\Config') && Config::getInstance()->containsKey('errors_destination')) {
                $mail_boxes = Config::getInstance()->getValue('errors_destination');
                $rcpt = (is_array($mail_boxes) ? implode(',', $mail_boxes) : $mail_boxes);
            } else {
                $rcpt = 'halk_errors@filanco.ru';
            }

            error_log(var_export($error, true));

            //when soap can't parse wsdl
            if (stristr($error['message'], 'SOAP-ERROR')) {
                mail($rcpt, 'Halk SOAP-ERROR', var_export($error, true));
                return null;
            }

            $output = ob_get_clean();
            # Here we handle the error, displaying HTML, logging, ...
            header('HTTP/1.1 503 Service Unavailable');

            $debug_info = $_SERVER['REMOTE_ADDR'] == '217.65.14.16' ? ('<p><pre>' . htmlspecialchars(var_export($error, true)) . '</pre></p>') : '';
            echo <<<EOF
<link href="/css/index/fullscreen_errors.css" rel="stylesheet">

<div class="blue-sky">
    <div class="body-content">
        <h1>Произошла внутренняя ошибка!</h1>
        <p>Наши специалисты уже знают об этой проблеме и работают над ее устранением.</p>
        <p>Приносим извинения за неудобства.</p>
        $debug_info
    </div>
</div>
EOF;

            $headers = sprintf("Content-type: text/plain; charset=%s\r\n", HALK_INTERNAL_ENCODING);

            if(class_exists('\Halk\Core\Config') && Config::getInstance()->containsKey('mail_default_from')) {
                $headers .= sprintf("From: %s \r\n", Config::getInstance()->getValue('mail_default_from'));
            }

            $debug = var_export($error, true) . PHP_EOL;

            $debug .= "\n\n";
            $debug .= "From hostname: " . $_SERVER['SERVER_NAME'];

            $debug .= "\n\n_GET:" . var_export($_GET, true);
            $debug .= "\n\n_POST:" . var_export($_POST, true);
            $debug .= "\n\n_COOKIE:" . var_export($_COOKIE, true);
            $debug .= "\n\n_SERVER:" . var_export($_SERVER, true);

            mail($rcpt, 'Halk FATAL ERROR', $debug, $headers);
        }
    }

    public static function exception(\Exception $exception)
    {
        if(!defined('HALK_RUNNING_TESTS')) {
            Logger::exception($exception);
        }
    }

    public static function warning($code, $message, $file, $line)
    {
        $level = '';
        switch ($code) {

            case E_WARNING:
            case E_USER_WARNING:
                $level = Logger::LOG_WARNING;
                break;
            case E_ERROR:
            case E_USER_ERROR:
                $level = Logger::LOG_ERROR;
                break;
            case E_USER_NOTICE:
            case E_NOTICE:
                $level = Logger::LOG_NOTICE;
                break;
            case E_STRICT:
                // TODO we cannot handle E_STRICT at the moment, Logger issues
                return;

            default:

                $level = Logger::LOG_ERROR;
                break;
        }

        if(error_reporting() !== 0) {

            $trace = array_reverse(debug_backtrace());
            $str_trace = '';

            foreach ($trace as $call) {
                if ($call['function'] == 'warning') {
                    continue;
                }
                $args = array();
                if (isset($call['args']) && is_array($call['args'])) {
                    foreach ($call['args'] as $arg) {
                        $args[] = is_scalar($arg) ? $arg : (is_object($arg) ? get_class($arg) : gettype($arg));
                    }
                }
                $args = implode(', ',$args);
                if (isset($call['class']) && $call['class']) {
                    $call['class'] .= $call['type'];
                } else {
                    $call['class'] = '';
                }
                if (!isset($call['line'])) {
                    $call['line'] = '<none>';
                }
                if (!isset($call['function'])) {
                    $call['function'] = '<none>';
                }
                $str_trace .= @$call['file'].':'.$call['line'].' '.$call['class'].$call['function'].'('.$args.')'."\n";
            }

            Logger::error($file, $line, $message, $level, $str_trace);
        }
    }
}