<?php namespace Halk\Core;

use Halk\Core\Exception\UnexpectedSituation;
use Halk\Core\Helper\ExceptionHelper;
use Halk\Core\Config;

/**
 * Logger class.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
class Logger extends Registry
{
    const LOG_DEBUG = 1;
    const LOG_INFO = 2;
    const LOG_NOTICE = 4;
    const LOG_WARNING = 8;
    const LOG_ERROR = 16;
    const LOG_EXCEPTION = 32;
    /**
     * Array of strings
     * @var array
     */
    public static $full_log = array();

    public static $errors = array();

    public static $log_level = 0;
    /**
     * Log message.
     *
     * @param string $message message
     * @param int    $mode    type of message
     *
     * @throws \Halk\Core\Exception\UnexpectedSituation
     * @return void
     */
    public static function log($message, $mode = self::LOG_INFO)
    {
        $ini = Config::getInstance();

        // установим лог_левел из конфига
        if ($ini->containsKey('debug_level') && empty(self::$log_level)) {
            $str = $ini->getValue('debug_level');
            $str_ar = explode('|', $str);
            foreach ($str_ar as $ar) {
                self::$log_level = self::$log_level | (int)$ar;
            }
        }

        $mode_str = null;
        $hash = md5($message);

        if (($mode & self::$log_level) !== $mode) {
            return null;
        }

        switch($mode) {
            case self::LOG_INFO:

                $message = sprintf("[%s] %s", "Info", $message);
                $mode_str = "info";
                break;

            case self::LOG_ERROR:

                $message = sprintf("[%s] %s", "Error", $message);
                $mode_str = "error";
                break;

            case self::LOG_DEBUG:

                $message = sprintf("[%s] %s", "Debug", $message);
                $mode_str = "debug";
                break;

            case self::LOG_NOTICE:
                $message = sprintf("[%s] %s", "Notice", $message);
                $mode_str = "notice";
                break;

            case self::LOG_WARNING:

                $message = sprintf("[%s] %s", "Warning", $message);
                $mode_str = "warning";
                break;

            case self::LOG_EXCEPTION:
                // message already formatted
                $mode_str = "exception";
                break;
        }

        switch(HALK_RUNNING_MODE) {

            case HALK_MODE_WWW:

                /**
                 * @todo
                 * This is very slow.
                 */

                if ($ini->containsKey('error_threshold') &&
                    intval($ini->getValue('error_threshold')) > $mode) {
                    break;
                }

                // @todo: some loggin to database, emails, etc...
                if ($mode >= self::LOG_NOTICE &&
                    $ini->containsKey('print_errors') &&
                    $ini->getValue('print_errors') == 1) {
                    self::$errors[] = $message;
                }

                if ($mode === self::LOG_DEBUG || $ini->containsKey('not_send_errors_on_email')) {
                    break;
                }

                $hostname = $_SERVER['SERVER_NAME'];

                $message .= "\n\n";
                $message .= "From hostname: " . $hostname;

                $message .= "\n\n_GET:" . var_export($_GET, true);
                $message .= "\n\n_POST:" . var_export($_POST, true);
                $message .= "\n\n_COOKIE:" . var_export($_COOKIE, true);
                $message .= "\n\n_SERVER:" . var_export($_SERVER, true);

                $app_name = defined('APPLICATION_NAME') ? APPLICATION_NAME : HALK_APPLICATION_NAME;
                $subject = sprintf('%s %s', $app_name, $mode_str);

                if ($ini->containsKey('errors_destination')) {
                    $mail_boxes = $ini->getValue('errors_destination');
                    $rcpt = (is_array($mail_boxes) ? implode(',', $mail_boxes) : $mail_boxes);
                } else {
                    $rcpt = 'halk_errors@filanco.ru';
                }

                $headers = sprintf("Content-type: text/plain; charset=%s\r\n", HALK_INTERNAL_ENCODING);

                if ($ini->containsKey('mail_default_from')) {
                    $headers .= sprintf("From: %s \r\n", $ini->getValue('mail_default_from'));
                }

                mail($rcpt, $subject, $message, $headers);
                break;

            case HALK_MODE_CLI:
                fputs(STDERR, $message . PHP_EOL);
                break;

            default:

                throw new UnexpectedSituation('Unknown HALK_RUNNING_MODE');
                break;
        }

        self::$full_log[$hash] = $message;
    }

    /**
     * Report an error.
     *
     * @param string  $file    file
     * @param integer $line    line
     * @param string  $message message
     * @param integer $level   log level
     *
     * @param string $trace
     * @return void
     */
    public static function error($file, $line, $message, $level = self::LOG_ERROR, $trace = '')
    {
        if (defined('HALK_DOCUMENT_ROOT')) {
            $file = str_replace(HALK_DOCUMENT_ROOT, '.', $file);
        }

        $message = sprintf("[%s:%s] %s\n\nTrace:\n%s", $file, $line, $message, $trace);
        self::log($message, $level);
    }

    /**
     * Report an exception.
     *
     * @param \Exception $e
     *
     * @return void
     */
    public static function exception(\Exception $e)
    {
        $message = ExceptionHelper::convertExceptionToString($e);
        self::log($message, self::LOG_EXCEPTION);
    }
}
