<?php
namespace Halk\Core;

use Generator;
use Halk\Core\Cache\CacheAdapter;
use Halk\Core\Cache\CacheFactory;
use Halk\Core\Exception\DatabaseException;
use Halk\Core\Exception\LoadException;
use Halk\Core\Exception\NotFoundException;
use Halk\Core\Exception\UnexpectedSituation;
use Halk\Core\Exception\ValidationError;
use Halk\Core\Exception\CoreException;
use PDO;

/**
 * Класс, реализующий паттерн ActiveRecord.
 *
 * @category   PHP
 * @package    Halk
 * @subpackage Core
 * @author     Kolombet Ivan <i.kolombet@hoster.ru>
 * @copyright  2011 Filanco
 * @license    Proprietary http://www.filanco.ru
 * @version    Release: 2.0
 * @link       http://halk.filanco.ru
 */
abstract class ActiveRecord extends Basic implements ActiveRecordInterface, LoadInterface
{

    /**
     * Signals
     */
    const BEFORE_DELETE = 'AR//BEFORE_DELETE';

    /**
     *
     */
    const BEFORE_UPDATE = 'AR//BEFORE_UPDATE';

    /**
     *
     */
    const BEFORE_INSERT = 'AR//BEFORE_INSERT';

    /**
     *
     */
    const AFTER_DELETE = 'AR//AFTER_DELETE';

    /**
     *
     */
    const AFTER_UPDATE = 'AR//AFTER_UPDATE';

    /**
     *
     */
    const AFTER_INSERT = 'AR//AFTER_INSERT';

    /**
     * Сценарий обработки модели
     * @var string
     */
    public $scenario;

    /**
     * Map of fields and their values and properties.
     * @var array
     */
    private $__map = array();

    /**
     * Current PK value.
     * @var mixed
     */
    protected $_pk_value = null;

    /**
     * Previous PK value (if changed).
     * @var mixed
     */
    protected $_pk_value_old = null;

    /**
     * Last query type. 'update' or 'insert'.
     * @var string
     */
    protected $_last_op = null;

    /**
     * Top class; this is basically the result of get_called_class() call.
     * @var string
     */
    protected $_top_class = null;

    /**
     * Primary key field.
     * @var string|array
     */
    static $pk_field = 'id';

    static $title_field = null;

    /**
     * Primary key type.
     * @var string
     */
    static $pk_type = 'integer';

    /**
     * Allowed sort by primary key.
     * @var boolean
     */
    static $pk_sortable = false;

    /**
     * Default sort field.
     * @var string
     */
    static $default_sort = 'id';

    /**
     * Table where data is stored.
     * @var string
     */
    static $table = null;

    /**
     * Connection id.
     * See DBO::$connection_id.
     * @var string
     */
    static $connection_id = HALK_DEFAULT_CONNECTION_ID;

    /**
     * Array of fields.
     * @var array
     */
    static $fields = array();

    /**
     * Cache of reflected classes.
     * @var array
     */
    static $__reflections = array();

    /**
     * Is loaded?
     * @var boolean
     */
    protected $_loaded = false;

    /**
     * Is changed?
     * @var boolean
     */
    protected $_changed = false;
    protected $_was_changed = false;

    /**
     * @var ActiveRecord
     */
    protected $_old_copy = null;

    /**
     * Table name, to override table for certain instance of class.
     * Used in lb.filanco.ru
     * @var string
     */
    public $table_override = null;

    /**
     * Disable caching
     * @var boolean
     */
    static $__disable_cache = false;

    /**
     * @var bool
     */
    static $__foreign_extending = false;
    static $__forcing_extending = false;

    static $__ignore_tables = array();

    /**
     * @var \Halk\Core\Cache\Cacheable
     */
    private static $cache_driver = false;

    public static $cache = false;

    /**
     * @static
     * @return array
     */
    public static function validators()
    {
        return array();
    }

    /**
     * @param $attributes
     */
    public function setAttributes($attributes)
    {
        foreach ($attributes as $key => $value) {
            if (isset($this->__map[$key])) {
                if ($this->__map[$key]['value'] != $value) {
                    $this->setChanged(true);
                }

                $this->__map[$key]['value'] = $value;
            }

            if (static::$pk_field == $key) {
                $this->_pk_value = $value;
            }
        }
    }

    /**
     *
     * @param bool $change
     */
    protected function setChanged($change)
    {
        if (!$this->wasChanged()) {
            $this->_changed = $change;
            $this->_was_changed = true;
        }
    }

    /**
     * Initialize class.
     * Get field and validator data from the class.
     *
     * @param string $class class name to initialize
     *
     * @return void
     */
    public static function reflect($class = null)
    {
        if (is_null($class)) {
            $class = get_called_class();
        }

        if (!isset(self::$__reflections[$class])) {

            $reflection = new \ReflectionClass($class);

            self::$__reflections[$class] = array(
                'fields' => $class::$fields,
                'table' => $class::$table,
                'pk_field' => $class::$pk_field,
                'class' => $class,
                'abstract' => false,
            );

            $reflect_fields = array();

            /**
            Going all the way down from top class to ActiveRecord class.
            Iterate over all parent classes and merge their stuff all together to get resulting
            fields and validators.
             */

            $child_class = null;

            do {

                $reflection_class = $reflection->getName();

                // Merge fields
                if (in_array('fields', array_keys($reflection->getStaticProperties()))) {
                    $reflect_fields = array_merge($reflect_fields, $reflection->getStaticPropertyValue('fields'));
                }

                $parent_class = $reflection->getParentClass();

                if ($parent_class !== false) {

                    $parent_class_name = $parent_class->getName();

                    if ($parent_class_name !== __CLASS__) {
                        static::reflect($parent_class_name);
                    }

                    if (!isset(self::$__reflections[$parent_class_name])) {
                        self::$__reflections[$parent_class_name] = array();
                    }

                    if (!isset(self::$__reflections[$parent_class_name]['fields'])) {
                        self::$__reflections[$parent_class_name]['fields'] = $parent_class_name::$fields;
                    }

                    self::$__reflections[$parent_class_name]['table'] = $parent_class_name::$table;
                    self::$__reflections[$parent_class_name]['pk_field'] = $parent_class_name::$pk_field;
                    self::$__reflections[$parent_class_name]['class'] = $parent_class_name;
                    self::$__reflections[$parent_class_name]['abstract'] = $parent_class->isAbstract();

                    self::$__reflections[$reflection_class]['parent_class'] = self::$__reflections[$parent_class_name];

                    if ($parent_class->isAbstract()) {
                        if (!$reflection->isAbstract()) {
                            if (!is_null($child_class)) {
                                self::$__reflections[$child_class]['bottom_table'] = $reflection->getStaticPropertyValue(
                                    'table'
                                );
                            }
                            self::$__reflections[$class]['bottom_table'] = $reflection->getStaticPropertyValue('table');
                        }
                    }

                    if ($parent_class_name::$table === $class::$table && !empty($class::$table) && !$class::$__forcing_extending) {
                        self::$__reflections[$class] =& self::$__reflections[$parent_class_name];
                    }
                }

                $child_class = $reflection_class;

            } while (($reflection = $reflection->getParentClass()) !== false && $reflection->name != __CLASS__);

            $class::$fields = $reflect_fields;
        }

    }

    /**
     * @static
     * @return mixed
     */
    public static function getBottomTable()
    {

        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            self::reflect($class);
        }

        if (!isset(self::$__reflections[$class]['bottom_table'])) {

            self::$__reflections[$class]['bottom_table'] = self::$__reflections[$class]['table'];

            foreach (self::getClassFamily() as $i_class) {
                if (isset(self::$__reflections[$i_class]['bottom_table']) && !is_null(
                        self::$__reflections[$class]['bottom_table']
                    )
                ) {
                    self::$__reflections[$class]['bottom_table'] = self::$__reflections[$i_class]['bottom_table'];
                    break;
                }
            }
        }

        return self::$__reflections[$class]['bottom_table'];
    }

    /**
     * @static
     *
     * @param bool $include_abstract
     *
     * @return mixed
     */
    public static function getClassFamily($include_abstract = true)
    {

        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            self::reflect($class);
        }

        $i_class = self::$__reflections[$class];

        $s = $include_abstract ? '-abs' : '';

        if (!isset(self::$__reflections[$i_class['class']]['family' . $s])) {

            $cache = array();

            do {

                if ($include_abstract || (!$include_abstract && !$i_class['abstract'])) {
                    $cache[] = $i_class['class'];
                }

                $i_class = self::$__reflections[$i_class['parent_class']['class']];

            } while (isset($i_class['parent_class']));

            self::$__reflections[$i_class['class']]['family' . $s] = $cache;
        }

        return self::$__reflections[$i_class['class']]['family' . $s];
    }

    /**
     * @static
     * @param null $class
     * @return array
     */
    public static function getColumnFamily($class = null)
    {

        $columns = array();

        if (is_null($class)) {

            foreach (static::getClassFamily() as $class) {

                $fields = self::$__reflections[$class]['fields'];
                $columns = array_merge($columns, array_keys($fields));
            }
            $class = get_called_class();

        } else {

            if (!isset(self::$__reflections[$class])) {
                self::reflect($class);
            }

            $columns = array_keys(self::$__reflections[$class]['fields']);
        }

        if (is_array($class::$pk_field)) {
            $columns = array_merge($columns, $class::$pk_field);
        } else {
            $columns[] = $class::$pk_field;
        }

        return $columns;
    }

    /**
     * @static
     * @param $table
     * @return array|null
     */
    public static function getTableColumns($table)
    {

        foreach (static::getClassFamily(true) as $class) {
            if (self::$__reflections[$class]['table'] == $table) {
                return static::getColumnFamily($class);
            }
        }

        return null;
    }

    /**
     * Constructor
     * @param array $values
     */
    public function __construct(array $values = array())
    {
        // Detect called class
        $this->_top_class = get_called_class();

        // Initialize class (detect fields)
        static::reflect($this->_top_class);

        // Initialize field map
        foreach (static::$fields as $field => $params) {

            $this->__map[$field] = array();
            $this->__map[$field]['type'] = $params['type'];
            $this->__map[$field]['is_array'] = Utils::ends_with($params['type'], '[]');
            $this->__map[$field]['value'] = null;
            isset($params['default']) && ($this->__map[$field]['default'] = $params['default']);
            isset($params['default']) && ($this->__map[$field]['value'] = $params['default']);
            isset($params['readonly']) && ($this->__map[$field]['readonly'] = $params['readonly']);
        }

        $this->setAttributes($values);
    }

    /**
     * @param ActiveRecord $parent_class
     * @return bool
     * @throws \Halk\Core\Exception\SaveException
     * @throws CoreException
     */
    public function extend(ActiveRecord $parent_class)
    {

        $reflection = new \ReflectionClass($this);

        if (!($reflection = $reflection->getParentClass())) {
            throw new CoreException(sprintf(
                'You cannot extend record of class %s from record of class %s',
                $this->_top_class,
                get_class($parent_class)
            ));
        }

        if ($reflection->getName() !== get_class($parent_class)) {
            throw new CoreException(sprintf(
                'You cannot extend record of class %s from record of class %s',
                $this->_top_class,
                get_class($parent_class)
            ));
        }

        if (!$parent_class->isLoaded()) {
            throw new CoreException(sprintf('You cannot extend record from unsaved record'));
        }

        $this->setPKValue($parent_class->getPKValue());

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        $stmt = $this->__buildInsertQueryExtending($dbo, $this->_top_class);
        $values = $this->__getInsertValuesExtending($dbo, $this->_top_class);

        $obj = $this;

        try {
            $dbo->query($stmt, array_values($values));
        } catch (\Halk\Core\Exception\DatabaseUniqueException $ue) {
            throw new \Halk\Core\Exception\SaveException(sprintf('%s уже существует', (string)$obj));
        } catch (DatabaseException $pdo_e) {
            throw new \Halk\Core\Exception\SaveException(sprintf(
                'Failed to extend record of class %s from record of class %s. Error was: %s. SQL was: %s (%s)',
                $this->_top_class,
                get_class($parent_class),
                $pdo_e->getMessage(),
                $stmt,
                implode(', ', $values)
            ));
        }

        $this->load();

        return true;
    }

    /**
     * Get primary key value.
     *
     * @see ActiveRecord::$_pk_value
     * @see ActiveRecord::$pk_type
     *
     * @return string
     */
    public function getPKValue()
    {

        return $this->_pk_value;
    }

    /**
     * Get old (previous) primary key value.
     *
     * @see ActiveRecord::$_old_pk_value
     *
     * @return string
     */
    public function getOldPKValue()
    {
        return $this->_pk_value_old;
    }

    /**
     * Set primary key value.
     *
     * @param string $value new value
     *
     * @see ActiveRecord::$_pk_value
     * @see ActiveRecord::$pk_type
     *
     * @return void
     */
    public function setPKValue($value)
    {
        if (is_array(static::$pk_field)) {
            $this->_pk_value = $value;
        } else {
            // Setting it via __set() setter
            $this->{self::getPKField()} = $value;
        }
    }

    /**
     * Get primary key field name.
     *
     * @return string
     */
    public static function getPKField()
    {
        return static::$pk_field;
    }

    //end getPKField()

    /**
     * @param $name
     * @return bool
     */
    protected function _isPKField($name)
    {
        if (is_array(static::$pk_field)) {
            return in_array($name, static::$pk_field);
        } else {
            return $name == static::$pk_field;
        }
    }

    /**
     * Clone object. PHP5 magic method.
     *
     * @link http://php.net/manual/ru/language.oop5.cloning.php
     *
     * @return mixed|object
     */
    public function __clone()
    {
        // Cloned object is not loaded, it is cloned, so it doesn't exist in database
        $this->_loaded = false;
        // And its primary key is null because it doesn't exist in database
        $this->_pk_value = null;
    }

    /**
     * Getter. PHP5 magic method.
     *
     * @param string $name name of the field to get
     *
     * @throws CoreException
     * @link http://ru.php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
     *
     * @return mixed
     */
    public function __get($name)
    {
        if ($name == static::$pk_field) {

            return $this->_pk_value;

        } elseif (is_array(static::$pk_field) && in_array($name, static::$pk_field)) {

            /**
            Composite PK
             */

            return $this->_pk_value[$name];

        } elseif (!in_array($name, array_keys($this->__map))) {

            throw new CoreException(sprintf("[%s:get] Field does not exist: '$name'", $this->_top_class));
        }

        return $this->__map[$name]['value'];
    }

    /**
     * PHP5 magic method.
     *
     * @param $name
     * @return mixed
     */
    public function __isset($name)
    {

        if ($name == static::$pk_field) {

            return isset($this->_pk_value);

        } elseif (is_array(static::$pk_field) && in_array($name, static::$pk_field)) {

            return isset($this->_pk_value[$name]);
        }

        return isset($this->__map[$name]['value']);
    }

    /**
     * Setter. PHP5 magic method.
     *
     * @param string $name  name of the field to set
     * @param mixed $value new value
     *
     * @throws CoreException
     * @link http://ru.php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
     *
     * @return void
     */
    public function __set($name, $value)
    {

        if ($this->isLoaded() && is_null($this->_old_copy) && !$this->isChanged() && $this->getLastOperation(
            ) != 'insert'
        ) {
            $this->_old_copy = clone $this;
            $this->_old_copy->setPKValue($this->getPKValue());
        }

        // ML
        if (static::hasServiceParam('props')) {
            if ($name == 'owner_id') {
                return;
            }
            $props_params = static::getServiceParam('props');
            if (isset($props_params['names'], $props_params['prefix'], $props_params['array'])) {
                foreach ($props_params['names'] as $prop) {
                    if ($name == $props_params['prefix'] . $prop['key']) {

                        if ($props_params['array'] === false) {
                            $this->{$props_params['prefix'] . $prop['key']} = $value;
                        } else {
                            if (!isset($this->__map['properties'])) {
                                $this->__map['properties'] = array();
                            }

                            $this->__map['properties'][$prop['key']] = $value;
                        }

                        return;
                    }
                }
            }
        }

        if ($value instanceof ActiveRecord) {
            $props = static::$fields[$name];
            if (isset($props['foreign_key'])) {
                $foreign_key = $props['foreign_key'];
                list($class, $key) = explode('.', $foreign_key, 2);
                if (!($value instanceof $class)) {
                    throw new CoreException(sprintf(
                        "[%s:set] Field %s expected to be of class %s, but %s given",
                        $this->_top_class,
                        $name,
                        $class,
                        get_class($value)
                    ));
                }
                $value = $value->$key;
            } else {
                throw new CoreException(sprintf(
                    "[%s:set] Invalid object assignment to field %s",
                    $this->_top_class,
                    $name
                ));
            }
        }

        if ($name == static::$pk_field) {

            $this->_pk_value_old = $this->_pk_value;
            $this->_pk_value = $value;

            settype($this->_pk_value, static::$pk_type);
            if ($this->isLoaded()) {
                if ($this->_pk_value_old !== $this->_pk_value) {
                    $this->_changed = true;
                }
            }
            return;

        } elseif (is_array(static::$pk_field) && in_array($name, static::$pk_field)) {

            /**
            Composite PK
             */

            if (!is_array($this->_pk_value)) {
                $this->_pk_value = array();
            }

            $this->_pk_value_old[$name] = isset($this->_pk_value[$name])? $this->_pk_value[$name]: null;
            $this->_pk_value[$name] = $value;

            // Get type for field
            // @todo ugly code
            $field_idx = 0;
            foreach (static::$pk_field as $i => $f) {
                if ($f == $name) {
                    $field_idx = $i;
                    break;
                }
            }

            if (!isset(static::$pk_type[$field_idx])) {
                throw new CoreException(sprintf(
                    '[%s:set] Invalid field- and type-array for composite PK',
                    $this->_top_class
                ));
            }

            $type = static::$pk_type[$field_idx];
            $type = preg_replace("/\(.*\)/", "", $type);
            $type = str_replace('varchar', 'string', $type);
            settype($this->_pk_value[$name], $type);

            if ($this->isLoaded()) {
                if ($this->_pk_value_old[$name] !== $value) {
                    $this->_changed = true;
                }
            }
            return;

        } elseif (!in_array($name, array_keys($this->__map))) {
            throw new CoreException(sprintf(
                "[%s:set] Field does not exist: '$name', value: $value",
                $this->_top_class
            ));
        }

        if (is_null($value) && isset($this->__map[$name]['not_null']) && $this->__map[$name]['not_null']) {
            throw new CoreException(sprintf("[%s:set] Field cannot be NULL: '$name'", $this->_top_class));
        }

        if (empty($value) && $value !== 0 && isset($this->__map[$name]['not_empty']) && $this->__map[$name]['not_empty']) {
            throw new CoreException(sprintf("[%s:set] Field cannot be empty: '$name'", $this->_top_class));
        }

        if (isset($this->__map[$name]['choices']) && !in_array($value, $this->__map[$name]['choices'])) {
            throw new CoreException(sprintf(
                "[%s:set] Constraint failed for field %s, allowed one of %s",
                $this->_top_class,
                $name,
                implode(', ', $this->__map[$name]['choices'])
            ));
        }

        if ($this->__map[$name]['is_array'] && !is_array($value) && $this->__map[$name]['type'] !== 'tuple') {
            $value = Utils::pgArray($value);
        }

        if ($this->__map[$name]['type'] === 'json' && is_string($value)) {
            $value = json_decode($value, true);
        }

        $old_value = $this->__map[$name]['value'];
        $this->__map[$name]['value'] = $value;

        if ($this->isLoaded()) {
            if ($old_value !== $value) {
                $this->_changed = true;
            }
        }
    }

    //end __set()

    /**
     * Load object from array.
     *
     * @param array $row db-row to load from
     *
     * @param bool $ignore_missing_fields
     *
     * @return bool true if okay
     */
    public function loadByRow(array $row, $ignore_missing_fields = false)
    {

        foreach ($row as $key => $value) {

            if ($ignore_missing_fields && !isset($this->__map[$key]) && !$this->_isPKField($key)) {
                continue;
            }

            $this->$key = $value;
        }

        $this->_loaded = true;
        $this->_afterLoad();
        return true;
    }

    //end loadByRow()

    /**
     *
     */
    protected function _afterLoad()
    {

    }

    public function cast($new_model)
    {
        $obj = new $new_model();
        $obj->loadByRow($this->asArray());
        return $obj;
    }

    public function isFieldChanged($field)
    {

        if (!$this->isChanged()) {
            return false;
        }

        return ($this->$field !== $this->_old_copy->$field);
    }

    /**
     * @static
     * @param DBO $dbo
     * @param         $from
     * @return string
     */
    protected static function __joinExtendingTables(DBO $dbo, $from, $as_array = false)
    {
        self::reflect(get_called_class());

        if (is_array(static::$pk_field)) {

            $pk = array_map(
                function ($x) use ($dbo) {
                    return SQLBuilder::escapeColumnName($x, $dbo->engine);
                },
                static::$pk_field
            );

            $pk = implode(', ', $pk);

        } else {

            $pk = SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine);
        }

        $_tables_used = array($from);
        foreach (self::getClassFamily(true) as $class) {

            if ($class == get_called_class()) {
                continue;
            }

            if (in_array(self::$__reflections[$class]['table'], static::$__ignore_tables)) {
                continue;
            }

            if (!is_null(self::$__reflections[$class]['table']) && !in_array(
                    self::$__reflections[$class]['table'],
                    $_tables_used
                )
            ) {
                $from .= sprintf(' LEFT JOIN %s USING (%s)', self::$__reflections[$class]['table'], $pk);
                $_tables_used[] = self::$__reflections[$class]['table'];
            }
        }

        if ($as_array === true) {
            $ext_tables['pk'] = & $pk;

            if (count($_tables_used) > 0) {
                $ext_tables['tables'] = & $_tables_used;
            } else {
                $ext_tables['tables'] = array();
            }

            return $ext_tables;
        }

        return $from;
    }

    /**
     * @static
     * @param DBO $dbo
     * @param bool $join
     * @return null|string
     */
    protected static function __getTableForSelect(DBO $dbo, $join = false)
    {

        $table = static::$table;

        //if(static::$__foreign_extending && $join) {
        $table = static::__joinExtendingTables($dbo, $table);
        //}

        return $table;
    }

    /**
     * @return mixed
     */
    protected function _loadByCompositePK()
    {

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        $field_list = implode(
            ', ',
            array_map(
                function ($x) use ($dbo) {
                    return SQLBuilder::escapeColumnName($x, $dbo->engine);
                },
                static::$pk_field
            )
        );
        $placeholder_list = implode(', ', array_fill(0, count($this->_pk_value), '?'));
        $from = static::__getTableForSelect($dbo, true);

        $column_list = array_map(
            function ($x) use ($dbo) {
                return SQLBuilder::escapeColumnName($x, $dbo->engine);
            },
            static::getColumnFamily()
        );

        $column_list = implode(', ', $column_list);

        $row = $dbo->getRow(
            sprintf('SELECT %s FROM %s WHERE (%s) = (%s)', $column_list, $from, $field_list, $placeholder_list),
            array_values($this->_pk_value),
            PDO::FETCH_ASSOC
        );

        return $row;
    }

    /**
     * @return mixed
     */
    protected function _loadByBasicPK()
    {

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        $from = static::__getTableForSelect($dbo, true);

        $column_list = array_map(
            function ($x) use ($dbo) {
                return SQLBuilder::escapeColumnName($x, $dbo->engine);
            },
            static::getColumnFamily()
        );

        $column_list = implode(', ', $column_list);

        $row = $dbo->getRow(
            sprintf(
                'SELECT %s FROM %s WHERE %s = ?',
                $column_list,
                $from,
                SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine)
            ),
            array(SQLBuilder::escapeColumnValue(static::$pk_type, $this->_pk_value, $dbo->engine)),
            PDO::FETCH_ASSOC
        );

        return $row;
    }

    /**
     * Load object by its primary key.
     * Primary key must be set before.
     *
     * @return boolean true if success
     *
     * @throws LoadException if PK not exists
     */
    public function load()
    {

        $row = is_array(static::$pk_field) ? $this->_loadByCompositePK() : $this->_loadByBasicPK();

        if (empty($row)) {

            throw new LoadException(sprintf(
                "Failed to load %s with pk=%s",
                $this->_top_class,
                is_array($this->_pk_value) ?
                    implode(', ', $this->_pk_value) :
                    $this->_pk_value
            ));
        }

        return $this->loadByRow($row);
    }

    //end load()

    /**
     * @param bool $only
     */
    protected function _deleteByCompositePK($only = false)
    {

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        $field_list = implode(
            ', ',
            array_map(
                function ($x) use ($dbo) {
                    return SQLBuilder::escapeColumnName($x, $dbo->engine);
                },
                static::$pk_field
            )
        );
        $placeholder_list = implode(', ', array_fill(0, count($this->_pk_value), '?'));
        $table = $only ? static::$table : self::getBottomTable();

        $dbo->query(
            sprintf('DELETE FROM %s WHERE (%s) = (%s)', $table, $field_list, $placeholder_list),
            array_values($this->_pk_value)
        );
    }

    /**
     * @param bool $only
     */
    protected function _deleteByBasicPK($only = false)
    {

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $table = $only ? static::$table : self::getBottomTable();

        $dbo->query(
            sprintf(
                "DELETE FROM %s WHERE %s = ?",
                $table,
                SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine)
            ),
            array(SQLBuilder::escapeColumnValue(static::$pk_type, $this->_pk_value, $dbo->engine))
        );
    }

    /**
     * Delete object by its primary key.
     * Primary key must be set before.
     *
     * @param bool $only
     *
     * @throws \Halk\Core\Exception\DeleteException
     * @return bool true if success
     */
    public function delete($only = false)
    {

        if ($this->isLoaded() && is_null($this->_old_copy)) {
            $this->_old_copy = clone $this;
            $this->_old_copy->setPKValue($this->getPKValue());
        }

        $this->_postSignal($this, self::BEFORE_DELETE);

        try {

            if (is_array(static::$pk_field)) {

                // Composite PK
                $this->_deleteByCompositePK($only);

            } else {

                // Basic PK
                $this->_deleteByBasicPK($only);
            }

        } catch (\Halk\Core\Exception\DatabaseReferenceException $re) {

            throw new \Halk\Core\Exception\DeleteException($re->getMessage());

        } catch (DatabaseException $e) {

            throw new \Halk\Core\Exception\DeleteException(sprintf(
                "Failed to delete %s with pk=%s. Message was: %s",
                $this->_top_class,
                $this->_pk_value,
                $e->getMessage()
            ));
        }

        $this->_loaded = false;
        $this->setLastOperation('delete');

        $this->_postSignal($this, self::AFTER_DELETE);

        if (static::$cache === true) {
            /* delete in redis */
            try {
                if (!is_array(static::$pk_field) && $this->getPKValue()) {
                    $cd = self::cache();
                    $cd->delete('class:' . get_called_class() . ':' . $this->getPKValue());
                }
            } catch (\Exception $e) {

            }
        }

        return true;
    }

    //end delete()

    /**
     *
     * @return bool
     */
    public function deleteOnly()
    {

        return $this->delete(true);
    }

    /**
     * @throws ValidationError
     * @return void
     */
    private function __validate()
    {

        foreach (static::validators() as $field => $closure) {

            if (!$closure($this->__map[$field]['value'], $this)) {
                throw new ValidationError(sprintf(
                    'Can\'t save class %s: invalid value %s of field %s',
                    $this->_top_class,
                    $this->__map[$field]['value'],
                    $field
                ));
            }
        }

        foreach (static::$fields as $key => $prop) {
            if (isset($prop['not_null']) && $prop['not_null'] && is_null($this->$key) && !isset($prop['default'])) {
                throw new ValidationError(sprintf('[%s] Field `%s` cant be null', $this->_top_class, $key));
            }

            $v = $this->$key;
            if (isset($prop['not_empty']) && $prop['not_empty'] && $v === "") {
                throw new ValidationError(sprintf('[%s] Field `%s` cant be empty', $this->_top_class, $key));
            }

            /* @todo unique check */
            /* @todo foreign key check */
        }
    }

    /**
     * @return bool
     */
    protected function _checkState()
    {
        return true;
    }

    /**
     * @param DBO $dbo
     * @param string $only_class
     * @return array
     */
    private function __updateGetFieldList(DBO $dbo, $only_class = null)
    {

        $non_empty_default_fields = array_filter(
            $this->__map,
            function ($x) {
                return (isset($x['default']) || $x['type'] == 'serial') && !is_null($x['value']);
            }
        );

        $field_list = ActiveRecordManager::getUpdateFieldList(
            is_null($only_class) ? $this->_top_class : $only_class,
            array_flip(array_keys($non_empty_default_fields)),
            !is_null($only_class)
        );

        // Add PK field if its changed
        if (is_array(static::$pk_field)) {
            /** @todo На скорую руку поправил работу с составным первичным ключом надо сделать элегантнее */
            $null_finded = false;

            foreach ($this->_pk_value_old as $old_value) {
                if (is_null($old_value)) {
                    $null_finded = true;
                    break;
                }
            }

            if (!$null_finded) {
                $old_pk_values = array_values($this->_pk_value_old);
                $pk_values = array_values($this->_pk_value);
                $diff = array_diff($old_pk_values, $pk_values);
                if (!empty($diff)) {
                    foreach (static::$pk_field as $tmp_field) {
                        $field_list[] = sprintf('%s = ?', SQLBuilder::escapeColumnName($tmp_field, $dbo->engine));
                    }
                }
            }

        } else {
            if (!is_null($this->_pk_value_old) && $this->_pk_value_old != $this->_pk_value) {
                $field_list[] = sprintf(
                    '%s = ?',
                    SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine)
                );
            }
        }
        return $field_list;
    }

    /**
     * @param DBO $dbo
     * @param string $only_class
     * @param bool $pk_force
     * @return mixed
     */
    private function __insertGetFieldList(DBO $dbo, $only_class = null, $pk_force = false)
    {

        $field_list = ActiveRecordManager::getInsertFieldList(
            is_null($only_class) ? $this->_top_class : $only_class,
            !is_null($this->_pk_value) || $pk_force,
            !is_null($only_class)
        );

        return $field_list;
    }

    /**
     * @param DBO $dbo
     * @param string $only_class
     * @param bool $pk_force
     * @return mixed
     */
    private function __insertGetPlaceholders(DBO $dbo, $only_class = null, $pk_force = false)
    {

        $non_empty_default_fields = array_filter(
            $this->__map,
            function ($x) {
                return (isset($x['default']) || $x['type'] == 'serial') && !is_null($x['value']);
            }
        );

        $placeholders = ActiveRecordManager::getInsertPlaceholders(
            is_null($only_class) ? $this->_top_class : $only_class,
            array_flip(array_keys($non_empty_default_fields)),
            !is_null($this->_pk_value) || $pk_force,
            !is_null($only_class)
        );

        return $placeholders;
    }

    /**
     * @param DBO $dbo
     * @return string
     */
    private function __buildUpdateQueryCompositePK(DBO $dbo)
    {

        return sprintf(
            'UPDATE %s SET %s WHERE (%s) = (%s) %s',
            empty($this->table_override) ? static::$table : $this->table_override,
            implode(', ', $this->__updateGetFieldList($dbo)),
            implode(
                ', ',
                array_map(
                    function ($x) use ($dbo) {
                        return SQLBuilder::escapeColumnName($x, $dbo->engine);
                    },
                    array_keys($this->_pk_value)
                )
            ),
            implode(
                ', ',
                array_fill(0, count(static::$pk_field), '?')),
            $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
        );
    }

    /**
     * @param DBO $dbo
     * @return string
     */
    private function __buildUpdateQueryBasicPK(DBO $dbo)
    {

        return sprintf(
            'UPDATE %s SET %s WHERE %s = ? %s',
            empty($this->table_override) ? static::$table : $this->table_override,
            implode(', ', $this->__updateGetFieldList($dbo)),
            SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine),
            $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
        );
    }

    /**
     * @param DBO $dbo
     * @return array
     * @throws UnexpectedSituation
     */
    private function __buildUpdateQueriesCompositePKExtending(DBO $dbo)
    {

        if (!static::$__foreign_extending) {
            throw new UnexpectedSituation(sprintf(
                'Unexpected call of %s in class %s',
                __METHOD__,
                $this->_top_class
            ));
        }

        $parent_fields = array();
        $fields_grouped_by_table = array();

        $stmt = array(); // generated queries

        foreach (self::getClassFamily(true) as $i_class) {
            if ($i_class::$__forcing_extending) {
                continue;
            }

            $i_class = self::$__reflections[$i_class];

            $table = $i_class['table'];
            $fields_grouped_by_table[$table] = $i_class['fields'];
            $parent_fields = array_merge($parent_fields, $i_class['fields']);

            if (count($i_class['fields']) == 0) {
                continue;
            }

            if (in_array($table, static::$__ignore_tables)) {
                continue;
            }

            $stmt[] = sprintf(
                'UPDATE %s SET %s WHERE (%s) = (%s) %s',
                $table,
                implode(', ', $this->__updateGetFieldList($dbo, $i_class['class'])),
                implode(
                    ', ',
                    array_map(
                        function ($x) use ($dbo) {
                            return SQLBuilder::escapeColumnName($x, $dbo->engine);
                        },
                        static::$pk_field
                    )
                ),
                implode(', ', array_fill(0, count(static::$pk_field), '?')),
                $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
            );

        }

        return $stmt;
    }

    /**
     * @param DBO $dbo
     * @return array
     * @throws UnexpectedSituation
     */
    private function __buildUpdateQueriesBasicPKExtending(DBO $dbo)
    {

        if (!static::$__foreign_extending) {
            throw new UnexpectedSituation(sprintf(
                'Unexpected call of %s in class %s',
                __METHOD__,
                $this->_top_class
            ));
        }

        $parent_fields = array();
        $fields_grouped_by_table = array();

        $stmt = array(); // generated queries

        foreach (self::getClassFamily(true) as $i_class) {
            if ($i_class::$__forcing_extending) {
                continue;
            }

            $i_class = self::$__reflections[$i_class];

            $table = $i_class['table'];
            $fields_grouped_by_table[$table] = $i_class['fields'];
            $parent_fields = array_merge($parent_fields, $i_class['fields']);

            if (count($i_class['fields']) == 0) {
                continue;
            }

            if (in_array($table, static::$__ignore_tables)) {
                continue;
            }

            $stmt[] = sprintf(
                'UPDATE %s SET %s WHERE %s = ? %s',
                $table,
                implode(', ', $this->__updateGetFieldList($dbo, $i_class['class'])),
                SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine),
                $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
            );

        }

        return $stmt;
    }

    /**
     * @param DBO $dbo
     * @return string
     */
    private function __buildInsertQuery(DBO $dbo)
    {

        return sprintf(
            'INSERT INTO %s (%s) VALUES (%s) %s',
            empty($this->table_override) ? static::$table : $this->table_override,
            implode(', ', $this->__insertGetFieldList($dbo)),
            implode(', ', $this->__insertGetPlaceholders($dbo)),
            $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
        );
    }

    /**
     * @param DBO $dbo
     * @param null $only_class
     * @return array|string
     * @throws \Halk\Core\Exception\UnexpectedSituation
     */
    private function __buildInsertQueryExtending(DBO $dbo, $only_class = null)
    {

        if (!static::$__foreign_extending) {
            throw new UnexpectedSituation(sprintf(
                'Unexpected call of %s in class %s',
                __METHOD__,
                $this->_top_class
            ));
        }

        if (!is_null($only_class)) {

            $table = self::$__reflections[$only_class]['table'];

            $stmt = sprintf(
                'INSERT INTO %s (%s) VALUES (%s) %s',
                self::$__reflections[$only_class]['table'],
                implode(
                    ', ',
                    $this->__insertGetFieldList(
                        $dbo,
                        $only_class,
                        !is_null($this->_pk_value) || (self::getBottomTable() !== $table)
                    )
                ),
                implode(
                    ', ',
                    $this->__insertGetPlaceholders(
                        $dbo,
                        $only_class,
                        !is_null($this->_pk_value) || (self::getBottomTable() !== $table)
                    )
                ),
                $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
            );
        } else {

            $stmt = array(); // generated queries

            foreach (self::getClassFamily(true) as $i_class) {
                if ($i_class::$__forcing_extending) {
                    continue;
                }

                $i_class = self::$__reflections[$i_class];
                $table = $i_class['table'];

                if (in_array($table, static::$__ignore_tables)) {
                    continue;
                }

                $stmt[] = sprintf(
                    'INSERT INTO %s (%s) VALUES (%s) %s',
                    $table,
                    implode(
                        ', ',
                        $this->__insertGetFieldList(
                            $dbo,
                            $i_class['class'],
                            !is_null($this->_pk_value) || (self::getBottomTable() !== $table)
                        )
                    ),
                    implode(
                        ', ',
                        $this->__insertGetPlaceholders(
                            $dbo,
                            $i_class['class'],
                            !is_null($this->_pk_value) || (self::getBottomTable() !== $table)
                        )
                    ),
                    $dbo->engine == 'pgsql' ? 'RETURNING *' : ''
                );
            }
        }

        return $stmt;
    }

    /**
     * @return array
     */
    private function __getPKValue()
    {

        $values = array();

        if (is_array(static::$pk_field)) {
            /**
             * Composite PK
             */
            foreach (static::$pk_field as $i => $key) {
                $i = array_search($key, static::$pk_field);
                $values[] = SQLBuilder::escapeColumnValue(static::$pk_type[$i], $this->_pk_value[$key]);
            }
        } else {

            $values[] = SQLBuilder::escapeColumnValue(static::$pk_type, $this->_pk_value);
        }

        return $values;
    }

    /**
     * @param null $only_class
     * @param bool $add_pk_value
     * @param DBO $dbo
     * @return array
     */
    private function __getValues($only_class = null, $add_pk_value = false, DBO $dbo)
    {

        $values = $this->__map;

        if (!is_null($only_class)) {
            $values = array_intersect_key($values, ActiveRecord::$__reflections[$only_class]['fields']);
        }

        // Skip readonly fields
        $values = array_filter(
            $values,
            function ($x) {
                return !isset($x['readonly']) || !$x['readonly'];
            }
        );

        // Skip default fields which are not set
        $values = array_filter(
            $values,
            function ($x) {
                return !((isset($x['default']) || $x['type'] == 'serial') && is_null($x['value']));
            }
        );

        $values = array_map(
            function ($x) use ($dbo) {
                return SQLBuilder::escapeColumnValue($x['type'], $x['value'], $dbo->engine);
            },
            $values
        );

        if ($add_pk_value && !is_null($this->_pk_value)) {

            /**
             * Add old PK value
             */
            if (is_array(static::$pk_field)) {
                /** @todo На скорую руку поправил работу с составным первичным ключом надо сделать элегантнее */
                $null_finded = false;

                foreach ($this->_pk_value_old as $old_value) {
                    if (is_null($old_value)) {
                        $null_finded = true;
                        break;
                    }
                }

                if (!$null_finded) {
                    $old_pk_values = array_values($this->_pk_value_old);
                    $pk_values = array_values($this->_pk_value);
                    $diff = array_diff($old_pk_values, $pk_values);
                    if (!empty($diff)) {
                        $values = array_merge($values, $this->__getPKValue());
                    }
                }

                /**
                 * Composite PK
                 */
                $values = array_merge(
                    $values,
                    $null_finded ? $this->_pk_value : $this->_pk_value_old
                );
            } else {
                /**
                 * Add PK value if it is changed
                 */
                if (!is_null($this->_pk_value_old) && $this->_pk_value_old != $this->_pk_value) {
                    $values = array_merge($values, $this->__getPKValue());
                }

                $values[] = SQLBuilder::escapeColumnValue(
                    static::$pk_type,
                    is_null($this->_pk_value_old) ? $this->_pk_value : $this->_pk_value_old,
                    $dbo->engine
                );
            }
        }

        return $values;
    }

    /**
     * @param DBO $dbo
     * @param null $only_class
     * @return array
     * @throws UnexpectedSituation
     */
    private function __getValuesExtending(DBO $dbo, $only_class = null)
    {

        if (!static::$__foreign_extending) {
            throw new UnexpectedSituation(sprintf(
                'Unexpected call of %s in class %s',
                __METHOD__,
                $this->_top_class
            ));
        }

        if (!is_null($only_class)) {

            $values = $this->__getValues($only_class, true, $dbo);

        } else {

            $values = array();

            foreach (self::getClassFamily(true) as $i_class) {
                if ($i_class::$__forcing_extending) {
                    continue;
                }

                if (count(self::$__reflections[$i_class]['fields']) == 0) {
                    continue;
                }

                if (in_array(self::$__reflections[$i_class]['table'], static::$__ignore_tables)) {
                    continue;
                }

                $values[] = $this->__getValues($i_class, true, $dbo);
            }
        }

        return $values;
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    private function __getUpdateValues(DBO $dbo)
    {

        $values = $this->__getValues(null, true, $dbo);

        return $values;
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    private function __getUpdateValuesExtending(DBO $dbo)
    {

        return $this->__getValuesExtending($dbo);
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    private function __getInsertValues(DBO $dbo)
    {

        $values = $this->__getValues(null, false, $dbo);

        /**
         * Add PK value if it is not empty
         */
        if (!empty($this->_pk_value)) {
            $values = array_merge($values, $this->__getPKValue());
        }

        return $values;
    }

    /**
     * @param DBO $dbo
     * @param null $only_class
     * @return array
     * @throws UnexpectedSituation
     */
    private function __getInsertValuesExtending(DBO $dbo, $only_class = null)
    {

        if (!static::$__foreign_extending) {
            throw new UnexpectedSituation(sprintf(
                'Unexpected call of %s in class %s',
                __METHOD__,
                $this->_top_class
            ));
        }

        $values = $this->__getValuesExtending($dbo, $only_class);

        return $values;
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    protected function _update(DBO $dbo)
    {

        $stmt = is_array(static::$pk_field) ?
            $stmt = $this->__buildUpdateQueryCompositePK($dbo) :
            $stmt = $this->__buildUpdateQueryBasicPK($dbo);

        $values = $this->__getUpdateValues($dbo);

        return array($stmt, $values);
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    protected function _updateExtending(DBO $dbo)
    {

        $stmt = is_array(static::$pk_field) ?
            $this->__buildUpdateQueriesCompositePKExtending($dbo) :
            $this->__buildUpdateQueriesBasicPKExtending($dbo);

        $values = $this->__getUpdateValuesExtending($dbo);

        return array($stmt, $values);
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    protected function _insert(DBO $dbo)
    {

        $stmt = $this->__buildInsertQuery($dbo);

        $values = $this->__getInsertValues($dbo);

        return array($stmt, $values);
    }

    /**
     * @param DBO $dbo
     * @return array
     */
    protected function _insertExtending(DBO $dbo)
    {

        $stmt = $this->__buildInsertQueryExtending($dbo);

        $values = $this->__getInsertValuesExtending($dbo);

        return array($stmt, $values);
    }

    /**
     * @param DBO $dbo
     * @param $stmt
     * @param array $values
     * @throws \Halk\Core\Exception\SaveException
     * @throws \Halk\Core\Exception\DatabaseUniqueException
     * @return mixed
     */
    protected function _save(DBO $dbo, $stmt, array $values)
    {
        try {
            $res = $dbo->query($stmt, array_values($values));
        } catch (\Halk\Core\Exception\DatabaseUniqueException $ue) {
            throw $ue;
        } catch (DatabaseException $pdo_e) {
            throw new \Halk\Core\Exception\SaveException(sprintf(
                'Failed to save class %s. Error was: %s. SQL was: %s (%s)',
                $this->_top_class,
                $pdo_e->getMessage(),
                $stmt,
                Utils::dump_sql_params($values)
            ));
        }

        return $res;
    }

    /**
     * @param DBO $dbo
     * @param array $stmt
     * @param array $values
     * @return bool
     * @throws \Halk\Core\Exception\SaveException
     */
    protected function _saveExtending(DBO $dbo, array $stmt, array $values)
    {

        // Открыта ли транзакция снаружи
        $OUTER_TRANSACTION = false;

        if (!$dbo->inTransaction()) {
            $dbo->begin();
        } else {
            $OUTER_TRANSACTION = true;
        }

        if ($this->getLastOperation() == 'insert') {
            $stmt = array_reverse($stmt);
            $values = array_reverse($values);
        }

        /* @todo php-5.4 */
        $obj = $this;
        $top_class = $this->_top_class;
        array_map(
            function ($x, $y) use ($dbo, &$obj, $top_class) {

                try {

                    is_null($y) && $y = array();

                    if ($obj->getLastOperation() == 'insert') {
                        if (!is_null($obj->getPKValue())) {
                            if (empty($y) || end($y) !== $obj->getPKValue()) {
                                /*
                                 * @todo
                                 * WARNING!!!! GOVNOKOD
                                 * плохая проверка, может быть совпадение когда какое то поле = pk и тогда все плохо.
                                 */
                                $y[] = $obj->getPKValue();
                            }
                        }
                    }

                    $res = $dbo->query($x, array_values($y));

                    $return_row = $obj->_returnRow($dbo, $res);

                    if (!is_null($return_row)) {
                        $obj->loadByRow($return_row);
                    }

                    if (static::$cache === true) {
                        try {
                            $cd = self::cache();
                            $cd->set(
                                'class:' . get_called_class() . ':' . $obj->getPKValue(),
                                serialize($obj->asArray())
                            );
                        } catch (\Exception $e) {

                        }
                    }

                } catch (\Halk\Core\Exception\DatabaseUniqueException $ue) {

                    if ($dbo->inTransaction()) {
                        $dbo->rollback();
                    }

                    throw new \Halk\Core\Exception\SaveException(sprintf('%s уже существует', (string)$obj));

                } catch (DatabaseException $pdo_e) {

                    if ($dbo->inTransaction()) {
                        $dbo->rollback();
                    }

                    throw new \Halk\Core\Exception\SaveException(sprintf(
                        'Failed to save class %s. Error was: %s. SQL was: %s (%s)',
                        $top_class,
                        $pdo_e->getMessage(),
                        $x,
                        Utils::dump_sql_params($y)
                    ));

                }

            },
            $stmt,
            $values
        );

        if (!$OUTER_TRANSACTION) {
            $dbo->commit();
        }

        return true;
    }

    /**
     * @param DBO $dbo
     * @param string $table
     * @return mixed
     */
    protected function _selectRow(DBO $dbo, $table = null)
    {
        /**
         *  If insert - get pk value with LastInsertId
         */

        if ($this->getLastOperation() == 'insert') {
            if (static::$pk_type == 'integer' && is_null($this->_pk_value)) {
                $this->{self::getPKField()} = (int)$dbo->getLastInsertId();
            }
        }

        if (is_array(static::$pk_field)) {

            /**
             * Composite PK
             */

            $tmp_fields = array_flip(static::$pk_field);
            $tmp_types = static::$pk_type;
            $column_list = array_map(
                function ($x) use ($dbo) {
                    return SQLBuilder::escapeColumnName($x, $dbo->engine);
                },
                is_null($table) ? static::getColumnFamily($this->_top_class) : static::getTableColumns($table)
            );

            $column_list = implode(', ', $column_list);
            $q = sprintf(
                'SELECT %s FROM %s WHERE (%s) = (%s)',
                $column_list,
                is_null($table) ? (empty($this->table_override) ? static::$table : $this->table_override) : $table,
                implode(
                    ', ',
                    array_map(
                        function ($x) use ($dbo) {
                            return SQLBuilder::escapeColumnName($x, $dbo->engine);
                        },
                        array_keys($this->_pk_value)
                    )
                ),
                implode(', ', array_fill(0, count(static::$pk_field), '?'))
            );
            $v = array_map(
                function ($key, $value) use ($tmp_fields, $tmp_types, $dbo) {
                    return SQLBuilder::escapeColumnValue($tmp_types[$tmp_fields[$key]], $value, $dbo->engine);
                },
                array_keys($this->_pk_value),
                array_values($this->_pk_value)
            );
            $return_row = $dbo->getRow($q,$v);

        } else {

            $column_list = array_map(
                function ($x) use ($dbo) {
                    return SQLBuilder::escapeColumnName($x, $dbo->engine);
                },
                is_null($table) ? static::getColumnFamily($this->_top_class) : static::getTableColumns($table)
            );

            $column_list = implode(', ', $column_list);

            $return_row = $dbo->getRow(
                sprintf(
                    'SELECT %s FROM %s WHERE %s = ?',
                    $column_list,
                    is_null($table) ? (empty($this->table_override) ? static::$table : $this->table_override) : $table,
                    SQLBuilder::escapeColumnName(static::$pk_field, $dbo->engine)
                ),
                array(SQLBuilder::escapeColumnValue(static::$pk_type, $this->_pk_value, $dbo->engine))
            );
        }

        return $return_row;
    }

    /**
     * @todo make protected when in php-5.4
     *
     * @param DBO $dbo
     * @param null|\PDOStatement $res
     *
     * @return mixed
     */
    public function _returnRow(DBO $dbo, \PDOStatement $res = null)
    {

        /**
        Get affecting row with PK and other...
         */

        if ($dbo->engine == 'pgsql') {

            $return_row = $res->fetch(PDO::FETCH_ASSOC);

        } elseif ($dbo->engine == 'mysql') {

            $return_row = $this->_selectRow($dbo);
        }

        return $return_row;
    }

    /**
     * Save object to database.
     * Builds query string to save.
     * Will generate new primary key if not set.
     * If object already exists it will be updated.
     *
     * @return boolean true if success
     */
    public function save()
    {

        if ($this->isLoaded() && !$this->isChanged()) {
            \Halk\Core\Logger::log(
                'Skipping save, model unchanged. Model=' . get_called_class() . ' pk=' . is_array($this->getPKValue()) ? json_encode($this->getPKValue()) : $this->getPKValue()
            );
            return;
        }

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        // Run all validators
        $this->__validate();

        // Check state
        $this->_checkState();

        // If class is loaded from database or its PK is set
        if (!is_null($this->_pk_value) && $this->isLoaded()) {

            $this->setLastOperation('update');
            $this->_postSignal($this, self::BEFORE_UPDATE);

            list($stmt, $values) = static::$__foreign_extending ? $this->_updateExtending($dbo) : $this->_update($dbo);

        } else {

            $this->setLastOperation('insert');
            $this->_postSignal($this, self::BEFORE_INSERT);

            list($stmt, $values) = static::$__foreign_extending ? $this->_insertExtending($dbo) : $this->_insert($dbo);
        }

        if (!static::$__foreign_extending) {
            $res = $this->_save($dbo, $stmt, $values);
            $return_row = $this->_returnRow($dbo, $res);
        } else {
            $this->_saveExtending($dbo, $stmt, $values);
            $return_row = $this->_selectRow($dbo, self::getBottomTable());
        }

        /**
         *Save return row to object
         */

        $this->loadByRow($return_row, true);
        if ($this->_changed) {
            $this->_was_changed = true;
            $this->_changed = false;
        }

        $this->_postSignal($this, $this->getLastOperation() == 'insert' ? self::AFTER_INSERT : self::AFTER_UPDATE);

        if (static::$cache === true) {
            try {
                if (!is_array(static::$pk_field) && $this->getPKValue()) {
                    $cd = self::cache();
                    $cd->set(
                        'class:' . get_called_class() . ':' . $this->getPKValue(),
                        serialize($this->asArray())
                    );
                }
            } catch (\Exception $e) {

            }
        }

        return true;
    }

    //end save()

    public function wasChanged()
    {
        return $this->_was_changed;
    }

    /**
     * Convert object to array.
     *
     * @return array result
     */
    public function asArray()
    {

        $array = array();

        foreach (static::$fields as $key => $params) {

            $array[$key] = $this->$key;
        }

        if (is_array(static::$pk_field)) {

            /**
             * Composite PK
             */
            $array = array_merge($array, $this->getPKValue());

        } else {

            $array[static::$pk_field] = $this->getPKValue();
        }

        $array = array_merge($array, Utils::halk_object_to_array($this));

        return $array;
    }

    //end asArray()
    /**
     * Get object by its primary key.
     *
     * @param mixed $pk_value primary key
     * @param bool $as_array [optional] whether to return result as array instead of object
     * @param boolean $throw_not_found_exception
     *
     * @throws
     *
     * @return static
     */
    public static function getByPK($pk_value, $as_array = false, $throw_not_found_exception = false)
    {
        if (empty($pk_value) && $pk_value !== 0) {
            return null;
        }

        if (is_array($pk_value)) {
            $object = static::getOne($pk_value, 0, '', null, $as_array);
        } else {
            if (static::$cache === true) {
                try {
                    $cd = self::cache();
                    if ($cd->exists('class:' . get_called_class() . ':' . $pk_value)) {
                        $data = unserialize($cd->get('class:' . get_called_class() . ':' . $pk_value));
                        if ($as_array) {
                            return $data;
                        } else {
                            $class_name = get_called_class();
                            $object = new $class_name();
                            $object->loadByRow($data, true);
                        }
                    } else {
                        $object = static::getByField($pk_value, static::$pk_field, $as_array);
                        if (!empty($object)) {
                            $cd->set('class:' . get_called_class() . ':' . $pk_value,
                                serialize($as_array ? $object : $object->asArray()));
                        }
                    }
                } catch (\Exception $e) {
                    $object = static::getByField($pk_value, static::$pk_field, $as_array);
                }
            } else {
                $object = static::getByField($pk_value, static::$pk_field, $as_array);
            }
        }

        if ($throw_not_found_exception && is_null($object)) {
            throw new NotFoundException('Object not found');
        }

        return $object;
    }

    //end getByPK()

    /**
     * @param $pk_value
     * @return array
     */
    public static function getByPK_array($pk_value)
    {

        return self::getByPK($pk_value, true);
    }

    /**
     * Get first object with specified field value.
     *
     * @param mixed $value      value
     * @param string $field_name [optional] name of the field (default = 'name')
     * @param bool $as_array [optional] whether to return result as array instead of object
     *
     * @return static
     */
    public static function getByField($value, $field_name = 'name', $as_array = false)
    {

        $params = array(
            $field_name => $value,
        );

        return self::getOne($params, 0, '', null, $as_array);
    }

    //end getByField()

    public static function getByField_array($value, $field_name = 'name')
    {

        return self::getByField($value, $field_name, true);
    }

    /**
     * Get first object with specified field value.
     *
     * @param string $field_name name of the field
     * @param mixed $value      value
     *
     * @return static
     */
    public static function getBy($field_name, $value)
    {

        return self::getByField($value, $field_name);
    }

    //end getBy()

    /**
     * Get first object with specified params.
     *
     * @param array $params sql params
     * @param integer $offset [optional] sql offset (default = 0)
     * @param string $sort   [optional] name of the field to sort by (default = PK)
     * @param string $table  [optional] table-name where data is stored (defaul = null)
     * @param bool $as_array [optional] whether to return result as array instead of object
     *
     * @return static
     */
    public static function getOne(array $params = array(), $offset = 0, $sort = '', $table = null, $as_array = false)
    {

        $result = static::getByParams($params, $offset, 1, $sort, $table, $as_array);

        return count($result) > 0 ? $result[0] : null;
    }

    //end getOne()

    public static function getOne_array(array $params = array(), $offset = 0, $sort = '', $table = null)
    {

        return self::getOne($params, $offset, $sort, $table, true);
    }

    /**
     * Get whole object collection.
     * Warning: if objects are too many this can lead to memory overrun.
     * Use carefully.
     *
     * @param string $sort  [optional] name of the field to sort by (default = PK)
     * @param string $table [optional] table where data is stored (default = null)
     * @param bool $as_array [optional] whether to return result as array of arrays instead of array of objects
     * @return static[]
     */
    public static function getAll($sort = '', $table = null, $as_array = false)
    {

        return static::getByParams(array(), 0, -1, $sort, $table, $as_array);
    }

    //end getAll()

    public static function getAll_array($sort = '', $table = null)
    {

        return self::getAll($sort, $table, true);
    }

    /**
     * Check if model contains certain field.
     *
     * @param string $field field to check
     *
     * @return boolean true if contains
     */
    static function containsField($field)
    {

        // Если поле отсутствует в fields
        if (!array_key_exists($field, static::$fields)) {
            // И поле не входит в PK
            if ((is_array(static::$pk_field) && !in_array($field, static::$pk_field)) && $field !== static::$pk_field) {
                return false;
            }
        }

        return true;
    }

    /**
     * Возвращает число строк, с использованием \Halk\Core\Db\Sql\Expression,
     * генерирующем набор условий WHERE.
     *
     * @author Mikhail Levykin <zend53@yandex.ru>
     * @param \Halk\Core\Db\Sql\Expression $expression
     * @return Int
     */
    public static function getCountByExpression(
        \Halk\Core\Db\Sql\Expression $expression,
        \Halk\Core\Db\Sql\Expression $expression_props = null,
        $join_left = false
    ) {
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $expr = $expression->getExpressionString(false);
        $expr = !empty($expr) ? ' WHERE ' . $expr : '';

        $table = self ::__getTableForSelect($dbo, true);
        $sql = 'SELECT COUNT(*) AS total FROM ' . $table . $expr;
        $values = $expression->getValues();

        try {
            $result = (array)$dbo->fetchAll($sql, $values);

        } catch (DatabaseException $pdo_e) {

            throw new LoadException(
                sprintf(
                    "Failed to load class %s by expression.\n\nError was: %s. \nSQL was: %s",
                    get_called_class(),
                    $pdo_e->getMessage(),
                    $sql
                ));
        }

        if (isset($result[0]['total'])) {
            return $result[0]['total'];
        }

        return false;
    } //end getCountByExpression()

    /**
     * Определяет включение в запрос SELECT дополнительных условий WHERE
     * <code>
     * $expr = new \Halk\Core\Db\Sql\Expression;
     * $expr->addExpressionLike('subject', 'new');
     * $expr->addExpression('subject', 'test',
     *      \Halk\Core\Db\Sql\Expression::OP_ILIKE_TWO,
     *      \Halk\Core\Db\Sql\Expression::PREDICATE_OR);
     *
     * Halk_Core_Model_TaskReadonly::injectWhereExpression($expr);
     * //Halk_Core_Model_TaskReadonly::showSQL();
     * $objects = Halk_Core_Model_TaskReadonly::getByParams_array();
     *
     * Halk_Core_Model_TaskReadonly::injectWhereExpression(null);
     * $objects = Halk_Core_Model_TasksReadonly::getByParams();
     * </code>
     *
     * @param \Halk\Core\Db\Sql\Expression $expr
     * @access public
     * @return void
     * @static
     * @author Mikhail Levykin
     */
    public static function injectWhereExpression(\Halk\Core\Db\Sql\Expression $expr = null)
    {
        if ($expr == null) {
            static::unsetServiceParam('expression_where');
        } else {
            static::setServiceParam('expression_where', $expr);
        }
    }

    /**
     * Возвращает массив объектов ActiveRecord.
     * Для выборки используется объект \Halk\Core\Db\Sql\Expression,
     * генерирующий набор условий WHERE.
     * @author Mikhail Levykin <zend53@yandex.ru>
     * @param \Halk\Core\Db\Sql\Expression $expression
     * @param Int $offset стартовая позиция выборки
     * @param Int $limit число строк
     * @param String $sort поле, по которому осуществляется сортировка. '<->fildname'
     * @return array
     */
    public static function getByExpression(
        \Halk\Core\Db\Sql\Expression $expression = null,
        $offset = 0,
        $limit = -1,
        $sort = '',
        $inject_props = false,
        \Halk\Core\Db\Sql\Expression $expression_props = null,
        $join_left = false
    ) {
        self::reflect();

        $values = array();
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        //$expr = $expression->getExpressionString(false);

        $table = self ::__getTableForSelect($dbo, true);

        //$sql  = 'SELECT * FROM ' . $table . $expr;

        if ($expression !== null) {
            $expr = $expression->getExpressionString(false);
            $values = $expression->getValues();
        }

        if (null === $expression_props && false === $inject_props) {
            $sql = 'SELECT ' . $table . '.* FROM ' . $table;
            if ($expression !== null) {
                $expr = !empty($expr) ? ' WHERE ' . $expr : '';
                $sql .= $expr;
            }
        }

        // ORDER BY
        if (!empty($sort)) {
            $order = 'ASC';

            if (strpos($sort, '-') === 0) {
                $order = 'DESC';
                $sort = substr($sort, 1);
            }

            if (!in_array($sort, array_keys(static::$fields))) {
                $sort = static::$pk_field;
            }

            $sql .= sprintf(" ORDER BY \"%s\" %s", $sort, $order);
        }

        $limit = intval($limit);
        $offset = intval($offset);
        $offset < 0 && $offset = 0;

        // OFFSET/LIMIT
        if ($limit > 0) {
            switch ($dbo->engine) {
                case 'pgsql':
                    $sql .= sprintf(" OFFSET %s LIMIT %s", $offset, $limit);
                    break;
                case 'mysql':
                    $sql .= sprintf(" LIMIT %s OFFSET %s", $limit, $offset);
                    break;
            }
        }

        //$values = $expression->getValues();

        /**
         * Run query
         */
        try {
            $result = (array)$dbo->fetchAll($sql, $values);

        } catch (DatabaseException $pdo_e) {
            throw new LoadException(sprintf(
                "Failed to load class %s by expression.\n\nError was: %s. \nSQL was: %s (%s)",
                get_called_class(),
                $pdo_e->getMessage(),
                $sql,
                implode(', ', $values)
            ));
        }

        return $result;
    }

    // end getByExpression

    /**
     * @static
     * @param array $params
     * @param int $offset
     * @param        $limit
     * @param string $sort
     * @param null $table
     * @return array
     * @throws LoadException
     */
    public static function getByParams_array(
        array $params = array(),
        $offset = 0,
        $limit = -1,
        $sort = '',
        $table = null
    ) {
        return self::getByParams($params, $offset, $limit, $sort, $table, true);
    }

    /**
     * @param array $params
     * @param int $offset
     * @param int $limit
     * @param string $sort
     * @param null $table
     * @return Generator
     * @throws LoadException
     */
    public static function getGeneratorByParams(
        array $params = array(),
        $offset = 0,
        $limit = -1,
        $sort = '',
        $table = null
    ) {
        self::reflect();

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $values = array();

        $statement = static::__buildQuery($dbo, $values, $params, $offset, $limit, $sort, $table);
        try {
            $query = $dbo->query($statement, array_values($values));

            // удаляем все "сервисные" параметры, чтобы не лезли в следующие запросы.
            if (static::getServiceParam('no_cache_injection') === true) {
                static::unsetAllServiceParams();
            }

            while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
                foreach (static::$fields as $field => $params) {
                    if ($params['type'] == 'hstore') {
                        if (isset($row[$field])) {
                            $row[$field] = Utils::hstoreToArray($row[$field]);
                        }
                    } elseif ($params['type'] == 'array'){
                        if (isset($row[$field])) {
                            $row[$field] = unserialize($row[$field]);
                        }
                    } elseif ($params['type'] == 'json'){
                        if (isset($row[$field])) {
                            $row[$field] = json_decode($row[$field], true);
                        }
                    }
                }

                $object = new static();
                !empty($table) && $object->table_override = $table;
                $object->loadByRow($row);
                yield $object;
            }
        } catch (DatabaseException $e) {
            throw new LoadException(sprintf(
                "Failed to load class %s by params.\n\nError was: %s. \nSQL was: %s (%s)",
                get_called_class(),
                $e->getMessage(),
                $statement,
                implode(', ', $values)
            ), 0, $e);
        }
    }

    /**
     * Get objects by specified params.
     *
     * @param array $params sql params
     * @param int $offset [optional] sql offset default (default = 0)
     * @param int $limit  [optional] sql limit (default = -1)
     * @param string $sort   [optional] name of the field to sort by (default = PK)
     * @param null|string $table  [optional] table where data is stored (default = null)
     * @param bool $as_array [optional] whether to return result as array of arrays rather than as array of objects
     *
     * @throws LoadException
     * @return array|static[]
     */
    public static function getByParams(
        array $params = array(),
        $offset = 0,
        $limit = -1,
        $sort = '',
        $table = null,
        $as_array = false
    ) {

        self::reflect();

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $values = array();

        $stmt = static::__buildQuery($dbo, $values, $params, $offset, $limit, $sort, $table);

        /**
         * Run query
         */
        try {

            $rows = (array)$dbo->fetchAll($stmt, array_values($values));
            //hstore conversion
            foreach (static::$fields as $field => $params) {
                if ($params['type'] == 'hstore') {
                    foreach ($rows as $i => $row) {
                        if (isset($row[$field])) {
                            $rows[$i][$field] = Utils::hstoreToArray($row[$field]);
                        }
                    }
                } elseif ($params['type'] == 'array'){
                    foreach ($rows as $i => $row) {
                        if (isset($row[$field])) {
                            $rows[$i][$field] = unserialize($row[$field]);
                        }
                    }
                } elseif ($params['type'] == 'json'){
                    foreach ($rows as $i => $row) {
                        if (isset($row[$field])) {
                            $rows[$i][$field] = json_decode($row[$field], true);
                        }
                    }
                }
            }

        } catch (DatabaseException $pdo_e) {

            throw new LoadException(sprintf(
                "Failed to load class %s by params.\n\nError was: %s. \nSQL was: %s (%s)",
                get_called_class(),
                $pdo_e->getMessage(),
                $stmt,
                implode(', ', $values)
            ));
        }

        if ($as_array) {

            if (static::hasServiceParam('props')) {
                $props_params = static::getServiceParam('props');
                if (isset($props_params['names'], $props_params['prefix'], $props_params['array'])) {
                    if ($props_params['array'] === true) {
                        $pref = & $props_params['prefix'];

                        foreach ($rows as &$row) {
                            $props_array = array();
                            foreach ($props_params['names'] as $name) {
                                if (false !== ($key = array_search($pref . $name['key'], $row))) {

                                    $props_array[$name['key']] = $row[$pref . $name['key']];
                                    unset($row[$pref . $name['key']]);
                                }
                            }
                            $row['properties'] = $props_array;
                        }
                    }
                }
            }

            $result = $rows;

        } else {
            /**
             * @todo PHP 5.4 (use $this)
             */
            $class_name = get_called_class();

            $result = array_map(
                function ($x) use ($table, $class_name) {

                    $self = new $class_name();
                    !empty($table) && $self->table_override = $table;
                    $self->loadByRow($x);
                    return $self;

                },
                $rows
            );
        }

        // удаляем все "сервисные" параметры, чтобы не лезли в следующие запросы.
        if (static::getServiceParam('no_cache_injection') === true) {
            static::unsetAllServiceParams();
        }

        return $result;

    }

    //end getByParams();

    /**
     * Get object collection by array of PK values.
     *
     * @param array $pk     primary key
     * @param integer $offset [optional] sql offset (default = 0)
     * @param integer $limit  [optional] sql limit (default = -1)
     * @param string $sort   [optional] sort key (default = PK)
     *
     * @param bool $as_array
     * @return array
     */
    public static function getByMultiplePK(array $pk, $offset = 0, $limit = -1, $sort = '', $as_array = false)
    {

        if (count($pk) > 0) {
            $placeholders = array_fill(0, count($pk), '?');

            $params[] = array(sprintf('%s IN (%s)', static::$pk_field, implode(', ', $placeholders)), $pk);

            return self::getByParams($params, $offset, $limit, $sort, null, $as_array);
        }

        return array();
    }

    //end getByPKArray()

    public static function getByMultiplePK_array(array $pk, $offset = 0, $limit = -1, $sort = '')
    {
        return self::getByMultiplePK($pk, $offset, $limit, $sort, true);
    }

    /**
     * Get count of objects by parameters.
     *
     * @param array $params sql params
     * @param integer $offset [optional] sql offset (default = 0)
     * @param integer $limit  [optional] sql limit (default = -1)
     * @param string $table  [optional] table where data is stored (default = null)
     *
     * @return integer
     */
    public static function getCount(array $params = array(), $offset = 0, $limit = -1, $table = null)
    {

        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $values = array();

        $stmt = static::__buildQuery($dbo, $values, $params, $offset, $limit, '', $table, true);

        $count = $dbo->getOne($stmt, $values);

        return $count;
    }

    //end getCount()

    /**
     * Check if object with certain primary key exists.
     *
     * @param mixed $pk primary key
     *
     * @return boolean true if exists
     */
    public static function exists($pk)
    {

        if (!is_array($pk)) {

            $params = array(
                static::$pk_field => $pk,
            );

        } else {

            $params = $pk;
        }

        return self::getCount($params, 0, 1) > 0;
    }

    //end exists()

    /**
     * Get new instance of class.
     * Should be used instead of "new Object" constructions for mocking purposes.
     *
     * @return static self
     */
    public static function newInstance()
    {

        return new static();
    }

    /**
     * Get random record.
     *
     * @static
     * @param int $count
     * @return array|ActiveRecord
     */
    public static function random($count = 1)
    {
        $count <= 0 && $count = 1;
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        $sql = 'SELECT ' . SQLBuilder::escapeColumnName(
                static::$pk_field,
                $dbo->engine
            ) . ' FROM ' . static::$table . ' ORDER BY RANDOM() LIMIT ' . $count;
        if ($count > 1) {
            $pk = $dbo->getAll($sql);
            return static::getByMultiplePK($pk);
        } else {
            $pk = $dbo->getOne($sql);
            return static::getByPK($pk);
        }
    }

    /**
     * @static
     * @return mixed
     */
    public static function nextId()
    {

        if (isset(static::$sequence)) {
            $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
            return $dbo->getOne('select nextval(\'' . static::$sequence . '\')');
        }
    }

    /**
     * @return mixed
     */
    public function begin()
    {
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        if (!$dbo->inTransaction()) {
            $dbo->begin();
        }
    }

    public function commit()
    {
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);
        if ($dbo->inTransaction()) {
            $dbo->commit();
        }
    }

    public function isChanged()
    {
        return $this->_changed;
    }

    public function isLoaded()
    {
        return $this->_loaded;
    }

    public function getChangeset()
    {

        if (is_null($this->_old_copy)) {
            if ($this->getLastOperation() == 'insert') {
                $changeset = $this->asArray();
            } elseif ($this->getLastOperation() == 'delete') {
                $changeset = $this->asArray();
            } else {
                $changeset = array();
            }
        } else {
            if ($this->getLastOperation() == 'delete') {
                $changeset = $this->_old_copy->asArray();
            } else {
                // TODO array values will not be diffed
                $left = array_filter(
                    $this->asArray(),
                    function ($x) {
                        return is_scalar($x);
                    }
                );
                $right = array_filter(
                    $this->_old_copy->asArray(),
                    function ($x) {
                        return is_scalar($x);
                    }
                );
                $tmp = array_diff($left, $right);
                $changeset = array();
                foreach ($tmp as $k => $v) {
                    $changeset[$k] = array($this->_old_copy->$k, $v);
                }
            }
        }

        foreach ($changeset as $k => $v) {
            if($k == 'updated') {
                unset($changeset[$k]);
            } else if (strpos($k, 'password') !== false) {
                $changeset[$k] = '********'; // не показываем пароли
            }
        }

        return $changeset;
    }

    public function getLastOperation()
    {
        return $this->_last_op;
    }

    public function setLastOperation($o)
    {
        $this->_last_op = $o;
    }

    public function getOldCopy()
    {
        return $this->_old_copy;
    }

    public function setOldCopy($old_copy)
    {
        $this->_old_copy = $old_copy;
    }

    /**
     * @return string
     */
    public function __toString()
    {

        if ($this->_old_copy) {
            return $this->_old_copy->__toString();
        }

        $result = sprintf('Object NEW');

        if ($this->getPKValue()) {
            $result = sprintf('Object #%s', $this->getPKValue());
        }

        return $result;
    }

    public static function getTitleField()
    {

        return static::$title_field;
    }

    /**
     * Контейнер, для сервисных параметров
     *
     * @var array
     * @access protected
     * @author Mikhail Levykin
     */
    protected $service_params = array( //'props' => null,
        //'expression_where' => null,
        //'show_sql' => false
    );

    /**
     * Удаляет все установленные "сервисные" параметры
     *
     * @access public
     * @return void
     * @author Mikhail Levykin
     */
    public static function unsetAllServiceParams()
    {
        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            self::reflect($class);
        }
        self::$__reflections[$class]['service_params'] = array();
    }

    /**
     * Включает/отключает печать строки SQL - запроса, вместо его выполнения.
     * Хорош, для отладки.
     *
     * @param boolean $flag
     * @access public
     * @return void
     * @static
     * @author Mikhail Levykin
     */
    public static function showSQL($flag = true, $wordwrap = 150, $exit = true)
    {
        if ($flag == true) {
            $params['wordwrap'] = $wordwrap;
            $params['exit'] = $exit;
            static::setServiceParam('show_sql', $params);
        } else {
            static::unsetServiceParam('show_sql');
        }
    }

    /**
     * Позволяет установить "сервисный" параметр
     *
     * @param String $param
     * @param mixed $value
     * @access public
     * @return void
     * @static
     * @author Mikhail Levykin
     */
    public static function setServiceParam($param, $value)
    {
        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            self::reflect($class);
        }
        self::$__reflections[$class]['service_params'][$param] = $value;
    }

    /**
     * Удаляет "сервисный" параметр
     *
     * @param string $param
     * @access public
     * @return void
     * @static
     * @author Mikhail Levykin
     */
    public static function unsetServiceParam($param)
    {
        if (static::hasServiceParam($param)) {
            unset(self::$__reflections[get_called_class()]['service_params'][$param]);
        }
    }

    /**
     * Удаляет все статические инъекции, после успешного запроса
     *
     * @static
     * @return void
     * @author Mikhail Levykin
     */
    public static function setNoCacheInjection()
    {
        static::setServiceParam('no_cache_injection', true);
    }

    /**
     * Возвращает значение сервисного параметра
     *
     * @param String $param Имя параметра
     * @param Boolean $flag_not_exists Возвращаемое значение,
     *     в случае отсутствия запрошенного параметра
     * @return mixed
     * @static
     * @author Mikhail Levykin
     */
    public static function getServiceParam($param, $flag_not_exists = false)
    {
        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            return $flag_not_exists;
        }

        if (isset(self::$__reflections[$class]['service_params'][$param])) {
            return self::$__reflections[$class]['service_params'][$param];
        }

        return $flag_not_exists;
    }

    /**
     * Проверяет наличие сервисного параметра
     *
     * @param String $param
     * @return boolean
     * @static
     * @author Mikhail Levykin
     */
    public static function hasServiceParam($param)
    {
        $class = get_called_class();
        if (!isset(self::$__reflections[$class])) {
            return false;
        }
        return isset(self::$__reflections[$class]['service_params'][$param]);
    }

    protected static function ___buildSort($sort, $engine = 'pgsql')
    {
        $temp = [];

        if (is_array($sort)) {

            foreach ($sort as $s) {
                $temp[] = self::___buildSort($s, $engine);
            }

        } else {
            $order = 'ASC';

            if (strpos($sort, '-') === 0) {
                $order = 'DESC';
                $sort = substr($sort, 1);
            }
            if (is_array(static::$pk_field) && in_array($sort, static::$pk_field)) {
                // allow sort order
            } elseif (!in_array($sort, array_keys(static::$fields))) {
                $sort = static::$pk_field;
            }

            if ($engine == 'pgsql') {
                $temp[] = "\"$sort\" $order ";
            } else {
                $temp[] = "$sort $order ";
            }
        }

        return implode(',', $temp);
    }

    /**
     * Раздербанил ванин билдквери на отдельные компоненты,
     * для удобства преобразования buildSort/добавления функционала
     *
     * @param mixed $sort
     * @param string $engine
     * @return string
     * @static
     * @author Mikhail Levykin
     */
    protected static function __buildSort($sort, $engine = 'pgsql')
    {
        if (empty($sort)) {
            return '';
        }

        $sort = self::___buildSort($sort, $engine);

        return sprintf(" ORDER BY %s", $sort);
    }

    /**
     * Собирает строку LIMIT
     *
     * @param int $offset
     * @param int $limit
     * @param int $engine
     * @access protected
     * @return string
     * @author Mikhail Levykin
     */
    protected static function __buildLimit($offset, $limit, $engine)
    {
        $limit = (int)$limit;
        $offset = (int)$offset;
        $offset < 0 && $offset = 0;

        if ($limit > 0) {
            switch ($engine) {
                case 'pgsql':
                    return sprintf(" OFFSET %s LIMIT %s", $offset, $limit);
                case 'mysql':
                    return sprintf(" LIMIT %s OFFSET %s", $limit, $offset);
            }
        }
    }

    /**
     * Собирает строку с перечнем колонок
     *
     * @param String $engine
     * @return String
     * @author Mikhail Levykin
     */
    protected static function __buildColumns($engine)
    {
        $columns = array_map(
            function ($x) use ($engine) {
                return SQLBuilder::escapeColumnName($x, $engine);
            },
            static::getColumnFamily()
        );

        return implode(', ', $columns);
    }

    /**
     * Готовит строку с плейсхолдерами, для плдстановки значений
     * в SQL - выражение.
     *
     * @param array $values
     * @access protected
     * @author Mikhail Levykin
     * @return string
     */
    protected static function __buildPlaceholders(array $values)
    {
        $placeholders = array_fill(0, count($values), '?');

        return implode(', ', $placeholders);
    }

    /**
     * Из многомерного массива значений делает одномерный
     *
     * @param array $values
     * @return array
     * @access protected
     * @author Mikhail Levykin
     */
    protected static function __buildStripValues(array $values)
    {
        $strip_values = array();
        foreach ($values as $value) {
            if (is_array($value)) {
                foreach ($value as $tmp_value) {
                    $strip_values[] = $tmp_value;
                }
            } else {
                $strip_values[] = $value;
            }
        }

        return $strip_values;
    }

    /**
     * Выполняет работу по подготовке SQL - запроса
     *
     * @param DBO $dbo
     * @param array $values
     * @param array $params
     * @param int $offset
     * @param int $limit
     * @param string $sort
     * @param string $table
     * @param boolean $count
     * @throws CoreException
     * @return string
     * @author Mikhail Levykin
     */
    public static function __buildQuery(
        DBO $dbo,
        array &$values,
        array $params = array(),
        $offset = 0,
        $limit = -1,
        $sort = '',
        $table = null,
        $count = false
    ) {

        $values = array();
        $used_where = false;

        $is_search = true;
        if (!empty($params)) {
            $keys = array_keys($params);
            $is_search = !is_string($keys[0]);
        }


        $from = empty($table) ? static::__getTableForSelect($dbo, true) : $table;
        $columns = !$count ? static::__buildColumns($dbo->engine) : 'COUNT(*)';
        $stmt = sprintf("SELECT %s FROM %s", $columns, $from);

        /*
         * WHERE clause
        */
        $iter = 0;
        foreach ($params as $key => $param) {

            if ($iter == 0 && !$used_where) {
                $used_where = true;
                $stmt .= ' WHERE';
            }

            if ($is_search) {

                $clause = ($iter > 0) ? (isset($param[2]) ? $param[2] : ' AND') : '';
                $stmt .= sprintf(" %s %s", $clause, $param[0]);

                if (!self::containsField($param[0])) {
                    throw new CoreException(sprintf(
                        '[%s:getByParams] Unexpected field: %s',
                        get_called_class(),
                        $param[0]
                    ));
                }

                if (isset($param[1])) {
                    if (is_array($param[1])) {
                        $values = array_merge($values, $param[1]);
                    } else {
                        $values[] = $param[1];
                    }
                }

            } else {

                if (!self::containsField($key)) {
                    throw new CoreException(sprintf(
                        '[%s:getByParams] Unexpected field: %s',
                        get_called_class(),
                        $key
                    ));
                }

                if (is_array($params[$key])) {
                    if (count($params[$key])) {
                        $placeholders = static::__buildPlaceholders($params[$key]);
                        $stmt .= sprintf(
                            '%s %s IN (%s)',
                            $iter != 0 ? ' AND' : '',
                            SQLBuilder::escapeColumnName($key, $dbo->engine),
                            $placeholders
                        );
                    }
                } else {
                    if (!is_null($params[$key])) {
                        $stmt .= sprintf(
                            '%s %s = ?',
                            $iter != 0 ? ' AND' : '',
                            SQLBuilder::escapeColumnName($key, $dbo->engine)
                        );
                    } else {
                        $stmt .= sprintf(
                            '%s %s IS NULL',
                            $iter != 0 ? ' AND' : '',
                            SQLBuilder::escapeColumnName($key, $dbo->engine)
                        );
                    }
                }
            }

            $iter++;
        }

        if (!$is_search) {
            $vs = array_filter(
                array_values($params),
                function ($x) {
                    return !is_null($x);
                }
            );
            $vs = static::__buildStripValues($vs);
            $values = array_merge($values, $vs);
        }

        // inject Expression
        if (static::hasServiceParam('expression_where')) {
            $expr = static::getServiceParam('expression_where');
            if ($expr instanceof \Halk\Core\Db\Sql\Expression) {
                $expression_string = $expr->getExpressionString();
                $expression_values = $expr->getValues();

                $stmt .= !$used_where ? ' WHERE ' : ' AND ';
                $stmt .= $expression_string;
                $values = array_merge($values, $expression_values);
            }
        }

        $stmt .= static::__buildSort($sort, $dbo->engine);
        $stmt .= static::__buildLimit($offset, $limit, $dbo->engine);

        if (static::hasServiceParam('show_sql')) {
            $params = static::getServiceParam('show_sql');
            if (is_array($params) && isset($params['exit'], $params['wordwrap'])) {
                $str = wordwrap($stmt, $params['wordwrap'], '<br />', 1);
                printf("<pre>%s\nvalues:%s</pre>", $str, print_r($values, true));

                $params['exit'] === true && exit;
            }
        }
        //static::unsetAllServiceParams();

        return $stmt;
    }

    //end buildQuery()

    /**
     * @return string
     */
    public static function getClassName()
    {
        return get_called_class();
    }

    /**
     * Возвращает перечисленные в аргументах свойства объекта в виде массива
     * @return array
     */
    public function extract()
    {
        $array = [];
        foreach (func_get_args() as $p) {
            if (isset($this->$p)) {
                $array[$p] = $this->$p;
            }
        }

        return $array;

    }

    protected static function cache()
    {
        if (!self::$cache_driver) {
            $config = Config::getInstance();
            if($config->containsKey('cache_driver')) {
                $driver = strtolower(trim($config->getValue('cache_driver')));
                if ($driver) {
                    self::$cache_driver = new CacheAdapter(CacheFactory::get($driver));
                }
            }
        }

        if (!self::$cache_driver) {
            throw new CoreException('Cache driver is not set');
        }

        return self::$cache_driver;
    }

    public static function isModified($saved_modify_hash)
    {
        $current_modify_hash = static::getModifyState();

        return ($current_modify_hash != $saved_modify_hash);
    }

    public static function getModifyState()
    {
        $dbo = ConnectionPool::getInstance()->get(static::$connection_id);

        $full_table_info = explode('.', static::$table);

        $schema = $full_table_info[0];
        $table_name = $full_table_info[1];

        if ($dbo->engine == 'pgsql') {

            return md5($dbo->getOne('
                SELECT
                  n_tup_ins+n_tup_upd+n_tup_del 
                FROM 
                  pg_stat_user_tables 
                WHERE schemaname=? AND relname=?
            ', [$schema, $table_name]));

        } elseif ($dbo->engine == 'mysql') {

            return md5($dbo->getOne('
                SELECT UNIX_TIMESTAMP(update_time) FROM information_schema.tables
                WHERE TABLE_SCHEMA=? AND table_name=?
            ', [$schema, $table_name]));

        }
    }

    public function loadFromData($data)
    {
        $this->setPKValue($data[static::getPKField()]);
        return $this->loadByRow($data);
    }
}
